<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\API\Traits\ValidationTrait;
use App\Http\Controllers\API\Traits\EmailTrait;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\Controller;
use App\Models\Contactu;
use App\Models\NewsLetterSubscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class NewLetterSubscriptionController extends Controller
{

    use ValidationTrait, EmailTrait, JsonResponseTrait;
    public function store(NewsLetterSubscription $newsLetterSubscription, Request $request)
    {
        return $this->newsLetter($newsLetterSubscription, $request);
    }

    public function contactUs(Request $request)
    {
        // Verify reCAPTCHA token
        $captchaToken = $request->input('captcha_token');
        if (!$captchaToken) {
            return $this->error('reCAPTCHA token is required', 422);
        }

        $secretKey = config('services.recaptcha.secret');
        $response = Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
            'secret' => $secretKey,
            'response' => $captchaToken,
        ]);

        $result = $response->json();
        // Log the response for debugging (remove in production if needed)
        \Illuminate\Support\Facades\Log::info('reCAPTCHA verification response', [
            'status' => $response->status(),
            'result' => $result
        ]);

        // Check if HTTP request was successful
        if (!$response->successful()) {
            \Illuminate\Support\Facades\Log::error('reCAPTCHA API request failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            return $this->error('reCAPTCHA verification service unavailable. Please try again later.', 503);
        }

        // Check if reCAPTCHA verification was successful
        if (!isset($result['success']) || !$result['success']) {
            $errorCodes = $result['error-codes'] ?? [];
            \Illuminate\Support\Facades\Log::warning('reCAPTCHA verification failed', [
                'error_codes' => $errorCodes,
                'result' => $result
            ]);
            return $this->error('reCAPTCHA verification failed. Please try again.', 422);
        }

        $validator = $this->contactUsValidationRequest($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $data = $request->all();
        $contact = Contactu::create($data);
        if (!$contact) {
            return $this->message('something went wrong');
        }

        // Send contact us email
        try {
            $emailSent = $this->contactUsEmail($contact);
            if ($emailSent) {
                return $this->message('Message sent successfully. We will get back to you soon.');
            } else {
                // Email failed but contact was saved
                \Illuminate\Support\Facades\Log::warning('Contact us email failed to send', [
                    'contact_id' => $contact->id,
                    'email' => $contact->email ?? 'unknown'
                ]);
                return $this->message('Your message has been received. We will get back to you soon.');
            }
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Contact us email exception', [
                'contact_id' => $contact->id,
                'error' => $e->getMessage()
            ]);
            return $this->message('Your message has been received. We will get back to you soon.');
        }
    }
}
