<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\Quote;
use Illuminate\Http\Request;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\QuoteCalculationTrait as TraitsQuoteCalculationTrait;
use App\Http\Controllers\API\Traits\StripeTrait;
use App\Http\Controllers\API\Traits\EmailTrait;
use App\Models\Property;
use App\Models\PropertyDetail;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class QuoteController extends Controller
{
    use JsonResponseTrait, TraitsQuoteCalculationTrait, StripeTrait, EmailTrait;

   

    /**
     * Get quote by UUID
     */
    public function getByUuid(string $uuid)
    {
        $quote = Quote::where('uuid', $uuid)
            ->with(['user', 'truck'])
            ->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        $pickUpProperty = Property::where('uuid', $uuid)->where('type', 'pick_up')->first();
        $dropOffProperty = Property::where('uuid', $uuid)->where('type', 'drop_off')->first();
        $quote['pickup_property'] = $pickUpProperty;
        $quote['dropoff_property'] = $dropOffProperty;
        // Get property details with only id and property_category
        $propertyDetailsPickUp = null;
        $propertyDetailsDropOff = null;
        if ($pickUpProperty) {
            $propertyDetailsPickUp = PropertyDetail::where('property_id', $pickUpProperty->id)
                ->where('property_type', 'pick_up')
                ->select('id', 'property_category')
                ->first();
        }

        if ($dropOffProperty) {
            $propertyDetailsDropOff = PropertyDetail::where('property_id', $dropOffProperty->id)
                ->where('property_type', 'drop_off')
                ->select('id', 'property_category')
                ->first();
        }
        $quote['property_details_pickup'] = $propertyDetailsPickUp;
        $quote['property_details_dropoff'] = $propertyDetailsDropOff;
        return $this->success($quote, 'Quote fetched successfully');
    }

    /**
     * Update quote and process Stripe payment (30% deposit or full amount)
     * Simplified one-step flow: If payment_method_id is provided, payment is charged immediately
     */
    public function update(Request $request)
    {
        $quote = Quote::where('uuid', $request->uuid)->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        $updateData = $request->only(['status']);
        if ($request->has('payment_type')) {
            $paymentMethod = $request->input('payment_method', 'stripe'); // 'stripe' or 'bank_transfer'
            if (!in_array($paymentMethod, ['stripe', 'bank_transfer'])) {
                return $this->error('Invalid payment method. Must be "stripe" or "bank_transfer"', 422);
            }
            $paymentType = $request->payment_type; // 'deposit' or 'full'
            if (!in_array($paymentType, ['deposit', 'full'])) {
                return $this->error('Invalid payment type. Must be "deposit" or "full"', 422);
            }
            if ($paymentMethod == 'bank_transfer') {

                if (!$request->hasFile('quote_payment_proof')) {
                    return $this->error('Payment proof image is required for bank transfer', 422);
                }
                $file = $request->file('quote_payment_proof');
                if (!$file->isValid() || !in_array(strtolower($file->getClientOriginalExtension()), ['jpg', 'jpeg', 'webp', 'png', 'pdf'])) {
                    return $this->error('Invalid file type. Please upload an image (jpg, jpeg, webp, png) or PDF', 422);
                }
                $uploadPath = public_path('quotes/payment_proofs');
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }
                $filename = 'payment_proof_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $filePath = $uploadPath . '/' . $filename;
                $file->move($uploadPath, $filename);
                $relativePath = 'quotes/payment_proofs/' . $filename;
                $totalAmount = (float) $quote->total_cost;
                $jobType = null;
                if ($quote->raw_data && isset($quote->raw_data['job_type'])) {
                    $jobType = $quote->raw_data['job_type'];
                } elseif ($quote->job_type) {
                    $jobType = $quote->job_type;
                }
                $paymentAmount = 0;
                $remainingAmount = 0;
                if ($paymentType === 'deposit') {
                    $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
                    if ($jobType === 'local_job') {
                        $depositAmount = (float) ($generalSettings['_local_job_deposit_amount'] ?? 100);
                        $paymentAmount = round($depositAmount, 2);
                        $remainingAmount = round($totalAmount - $paymentAmount, 2);
                    } elseif ($jobType === 'interstate_job') {
                        // Interstate job: percentage deposit
                        $depositPercentage = (float) ($generalSettings['_interstate_job_deposit_percentage'] ?? 20);
                        $paymentAmount = round($totalAmount * ($depositPercentage / 100), 2);
                        $remainingAmount = round($totalAmount - $paymentAmount, 2);
                    } else {
                        // Fallback: 30% deposit if job type is not set
                        $paymentAmount = round($totalAmount * 0.30, 2);
                        $remainingAmount = round($totalAmount - $paymentAmount, 2);
                    }
                } else {
                    // Full payment
                    $paymentAmount = $totalAmount;
                    $remainingAmount = 0;
                }

                // Get existing raw_data and update it
                $rawData = $quote->raw_data ?? [];
                $rawData['payment_method'] = 'bank_transfer';

                // Update quote with bank transfer payment information
                $updateData['payment_method'] = 'bank_transfer'; // Save directly to payment_method column
                $updateData['payment_type'] = $paymentType;
                $updateData['quote_payment_proof'] = $relativePath;
                $updateData['deposit_amount'] = $paymentAmount;
                $updateData['remaining_amount'] = $remainingAmount;
                $updateData['deposit_payment_status'] = 'pending'; // Pending until admin approves
                $updateData['raw_data'] = $rawData; // Also store in raw_data for consistency

                // For full payment, also set remaining payment status
                if ($paymentType === 'full') {
                    $updateData['remaining_payment_status'] = 'pending';
                }

                // Convert to order if not already
                if ($quote->quote_type !== 'order') {
                    $updateData['quote_type'] = 'order';
                    if (!$quote->order_number) {
                        $updateData['order_number'] = Quote::generateOrderNumber();
                    }
                }

                // Update quote
                $quote->update($updateData);
                $quote->refresh();
                $quote->load(['user']);


                if ($paymentType === 'deposit') {
                    // Send deposit order confirmation email (payment is pending but order is placed)
                    $emailResults = $this->sendOrderConfirmationEmail($quote, 'deposit', 'pending', false);
                    if (!$emailResults['customer']) {
                        Log::warning('Failed to send order confirmation email to customer for bank transfer deposit - quote ID: ' . $quote->id);
                    }
                    if (!$emailResults['admin']) {
                        Log::warning('Failed to send order confirmation email to admin for bank transfer deposit - quote ID: ' . $quote->id);
                    }
                } else {
                    // Full payment - send full order confirmation email
                    $emailResults = $this->sendOrderConfirmationEmail($quote, 'full', 'pending', false);
                    if (!$emailResults['customer']) {
                        Log::warning('Failed to send order confirmation email to customer for bank transfer full payment - quote ID: ' . $quote->id);
                    }
                    if (!$emailResults['admin']) {
                        Log::warning('Failed to send order confirmation email to admin for bank transfer full payment - quote ID: ' . $quote->id);
                    }
                }
                // Prepare response data
                $responseData = [
                    'quote' => $quote,
                    'payment_method' => 'bank_transfer',
                    'payment_status' => 'pending',
                    'payment_amount' => $paymentAmount,
                    'remaining_amount' => $remainingAmount,
                    'total_amount' => $totalAmount,
                    'payment_proof_path' => $relativePath,
                ];

                $responseData['pickup_property'] = Property::where('uuid', $quote->uuid)->where('type', 'pick_up')->first();
                $responseData['dropoff_property'] = Property::where('uuid', $quote->uuid)->where('type', 'drop_off')->first();
                return $this->success($responseData, 'Bank transfer payment proof uploaded successfully. Payment is pending admin approval.');
            }
            
            // Handle Stripe payment (existing code)
            try {
                $paymentMethodId = $request->input('payment_method_id'); // From frontend Stripe Card Element
                // Check if payment already exists
                if ($paymentType === 'deposit' && $quote->stripe_payment_intent_deposit_id) {
                    return $this->error('Deposit payment intent already exists', 422);
                }
                // Create and optionally confirm payment intent (one-step if payment_method_id provided)
                $paymentResult = $this->createPaymentIntentForQuote($quote, $quote->total_cost, $paymentType, $paymentMethodId);
                $paymentIntent = $paymentResult['payment_intent'];
                $customer = $paymentResult['customer'];

                // Update quote with Stripe information
                $updateData['stripe_customer_id'] = $customer->id;

                // Determine payment status based on payment intent status
                $paymentStatus = 'pending';
                if ($paymentIntent->status === 'succeeded') {
                    $paymentStatus = 'succeeded';
                } elseif ($paymentIntent->status === 'requires_action') {
                    $paymentStatus = 'processing'; // 3D Secure required
                } elseif ($paymentIntent->status === 'canceled' || $paymentIntent->status === 'payment_failed') {
                    $paymentStatus = 'failed';
                }

                if ($paymentType === 'deposit') {
                    $updateData['stripe_payment_intent_deposit_id'] = $paymentIntent->id;
                    $updateData['deposit_amount'] = $paymentResult['payment_amount'];
                    $updateData['remaining_amount'] = $paymentResult['remaining_amount'];
                    $updateData['deposit_payment_status'] = $paymentStatus;
                } else {
                    // Full payment - store in deposit fields for consistency
                    $updateData['stripe_payment_intent_deposit_id'] = $paymentIntent->id;
                    $updateData['deposit_amount'] = $paymentResult['payment_amount'];
                    $updateData['remaining_amount'] = 0;
                    $updateData['deposit_payment_status'] = $paymentStatus;
                    // When payment is full, set remaining_payment_status to succeeded
                    if ($paymentStatus === 'succeeded') {
                        $updateData['remaining_payment_status'] = 'succeeded';
                    }
                }

                // Store payment_method directly to column and in raw_data for consistency
                $updateData['payment_method'] = 'stripe';
                $rawData = $quote->raw_data ?? [];
                $rawData['payment_method'] = 'stripe';
                $updateData['raw_data'] = $rawData;

                // Convert to order if not already
                if ($quote->quote_type !== 'order') {
                    $updateData['quote_type'] = 'order';
                    if (!$quote->order_number) {
                        $updateData['order_number'] = Quote::generateOrderNumber();
                    }
                }

                // Update quote
                $quote->update($updateData);
                $quote->refresh();
                $quote->load(['user']);

                // Prepare response data for frontend
                $responseData = [
                    'quote' => $quote,
                    'payment_status' => $paymentStatus,
                    'amount_charged' => $paymentResult['payment_amount'],
                    'remaining_amount' => $paymentResult['remaining_amount'],
                    'total_amount' => $paymentResult['total_amount'],
                ];

                $responseData['pickup_property'] = Property::where('uuid', $quote->uuid)->where('type', 'pick_up')->first();
                $responseData['dropoff_property'] = Property::where('uuid', $quote->uuid)->where('type', 'drop_off')->first();

                // If deposit succeeded, also create and return remaining payment link
                if ($paymentType === 'deposit' && $paymentStatus === 'succeeded') {
                    $paymentLinkResult = $this->createRemainingPaymentLink($quote);
                    $responseData['remaining_payment_link'] = $paymentLinkResult['payment_link_url'];
                    // Update quote with payment link
                    $quote->update(['remaining_payment_url' => $paymentLinkResult['payment_link_url']]);
                }

                $message = $paymentStatus === 'succeeded'
                    ? 'Payment charged successfully!'
                    : ($paymentStatus === 'requires_action'
                        ? 'Payment requires authentication. Use client_secret to complete 3D Secure.'
                        : 'Payment intent created. Use client_secret to confirm payment on frontend.');
                return $this->success($responseData, $message);
            } catch (\Stripe\Exception\ApiErrorException $e) {
                Log::error('Stripe API Error in quote update: ' . $e->getMessage());
                $errorMessage = $this->getStripeErrorMessage($e);
                return $this->error($errorMessage, 422);
            } catch (\Exception $e) {
                Log::error('Error processing payment: ' . $e->getMessage());
                return $this->error('Error processing payment: ' . $e->getMessage(), 500);
            }
        }
        // Regular update without payment
        $quote->update($updateData);
        $quote->refresh();
        $quote->load(['user']);

        return $this->success($quote, 'Quote updated successfully');
    }

    /**
     * Create remaining payment link (70%)
     * This is called after work completion to send payment link to customer via email
     */
    public function createRemainingPaymentLinkEndpoint(Request $request)
    {
        $quote = Quote::where('uuid', $request->uuid)->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        // Check if deposit was paid
        if ($quote->deposit_payment_status !== 'succeeded') {
            return $this->error('Deposit payment must be completed first', 422);
        }
        // Check if remaining payment already exists and is completed
        if ($quote->remaining_payment_status === 'succeeded') {
            return $this->error('Remaining payment already completed', 422);
        }
        // Check if payment link already exists
        if ($quote->remaining_payment_url) {
            return $this->success([
                'quote' => $quote->fresh(),
                'payment_link' => $quote->remaining_payment_url,
                'amount' => $quote->remaining_amount,
            ], 'Payment link already exists');
        }
        try {
            // Create remaining payment link using StripeTrait
            $paymentLinkResult = $this->createRemainingPaymentLink($quote);
            $paymentLink = $paymentLinkResult['payment_link'];
            $paymentLinkUrl = $paymentLinkResult['payment_link_url'];
            $customer = $paymentLinkResult['customer'];
            // Ensure customer ID is saved
            if (!$quote->stripe_customer_id) {
                $quote->update(['stripe_customer_id' => $customer->id]);
            }
            // Update quote with remaining payment link
            $quote->update([
                'remaining_payment_url' => $paymentLinkUrl,
                'remaining_payment_status' => 'pending',
            ]);
            return $this->success([
                'quote' => $quote->fresh(),
                'payment_link' => $paymentLinkUrl,
                'amount' => $paymentLinkResult['remaining_amount'],
            ], 'Payment link created successfully. Send this link to customer via email.');
        } catch (\Stripe\Exception\ApiErrorException $e) {
            Log::error('Stripe API Error creating payment link: ' . $e->getMessage());
            $errorMessage = $this->getStripeErrorMessage($e);
            return $this->error($errorMessage, 422);
        } catch (\Exception $e) {
            Log::error('Error creating payment link: ' . $e->getMessage());
            return $this->error('Error creating payment link: ' . $e->getMessage(), 500);
        }
    }


    /**
     * Get payment intent status (for checking payment status)
     */
    public function getPaymentIntentStatus(Request $request)
    {
        $request->validate([
            'payment_intent_id' => 'required|string',
        ]);

        try {
            $this->initializeStripe();
            $paymentIntent = \Stripe\PaymentIntent::retrieve($request->payment_intent_id);

            return $this->success([
                'payment_intent_id' => $paymentIntent->id,
                'status' => $paymentIntent->status,
                'amount' => $paymentIntent->amount / 100,
                'currency' => $paymentIntent->currency,
            ], 'Payment intent status retrieved');
        } catch (\Exception $e) {
            Log::error('Error retrieving payment intent status: ' . $e->getMessage());
            return $this->error('Error retrieving payment status: ' . $e->getMessage(), 500);
        }
    }

    public function getUserQuotes(Request $request)
    {
        $user = $request->user();
        $booking = Booking::where('user_id', $user->id)->get();
        $quotes = Quote::whereIn('booking_id', $booking->pluck('id'))
            ->orderByDesc('created_at')
            ->get();

        // Get all UUIDs from quotes
        $uuids = $quotes->pluck('uuid')->filter()->unique();

        // Fetch all properties for these quotes (both pick_up and drop_off)
        $properties = Property::whereIn('uuid', $uuids)
            ->get()
            ->groupBy('uuid');

        // Format quotes with their related properties
        $formatted = $quotes->map(function (Quote $quote) use ($properties) {
            $data = $quote->toArray();

            // Get properties for this quote's UUID
            $quoteProperties = $properties->get($quote->uuid, collect());

            // Separate pick_up and drop_off properties
            $pickUpProperty = $quoteProperties->where('type', 'pick_up')->first();
            $dropOffProperty = $quoteProperties->where('type', 'drop_off')->first();

            // Add properties to the quote data
            $data['pickup_property'] = $pickUpProperty ? $pickUpProperty->toArray() : null;
            $data['dropoff_property'] = $dropOffProperty ? $dropOffProperty->toArray() : null;

            return $data;
        });

        return $this->success($formatted, 'User quotes fetched successfully');
    }

    public function getUserQuotesUuid(string $uuid)
    {
        $quotes = Quote::with([
            'user',
            'truck',
            'booking.bookingItems.entity',
            'booking.pickUpProperty.propertyDetails',
            'booking.dropOffProperty.propertyDetails',
            'booking.truck',
        ])
            ->where('uuid', $uuid)
            ->orderByDesc('created_at')
            ->get();

        // Get all UUIDs from quotes
        $uuids = $quotes->pluck('uuid')->filter()->unique();

        // Fetch all properties for these quotes (both pick_up and drop_off) matched by UUID
        $properties = Property::whereIn('uuid', $uuids)
            ->with('propertyDetails')
            ->get()
            ->groupBy('uuid');

        // Format quotes with all related data
        $formatted = $quotes->map(function (Quote $quote) use ($properties) {
            $data = $quote->toArray();

            // Get properties for this quote's UUID (matched by UUID, not through booking)
            $quoteProperties = $properties->get($quote->uuid, collect());

            // Separate pick_up and drop_off properties matched by UUID
            $pickUpProperty = $quoteProperties->where('type', 'pick_up')->first();
            $dropOffProperty = $quoteProperties->where('type', 'drop_off')->first();

            // Add properties matched by UUID to the quote data
            $data['pickup_property'] = $pickUpProperty ? $pickUpProperty->toArray() : null;
            $data['dropoff_property'] = $dropOffProperty ? $dropOffProperty->toArray() : null;

            // Get property details for UUID-matched properties
            $propertyDetailsPickUp = null;
            $propertyDetailsDropOff = null;
            if ($pickUpProperty && $pickUpProperty->propertyDetails) {
                $propertyDetailsPickUp = $pickUpProperty->propertyDetails->toArray();
            }
            if ($dropOffProperty && $dropOffProperty->propertyDetails) {
                $propertyDetailsDropOff = $dropOffProperty->propertyDetails->toArray();
            }
            $data['property_details_pickup'] = $propertyDetailsPickUp;
            $data['property_details_dropoff'] = $propertyDetailsDropOff;

            return $data;
        });
        return $this->success($formatted, 'User quotes fetched successfully');
    }
}
