<?php

namespace App\Http\Controllers\API\Traits;
use stdClass;
use App\Models\GeneralSettings;

trait JsonResponseTrait
{

    public function success($data, $message = '', $statusCode = 200)
    {
        $preparedData = $this->prepareData($data, $message);
        if (empty($preparedData)) {
            return $this->notFound("Record Not Found", 404);
        }
        $response = [
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode
        ];
        if (is_array($preparedData) && isset($preparedData['items']) && isset($preparedData['pagination'])) {
            $response['data'] = $preparedData['items'];
            $response['pagination'] = $preparedData['pagination'];
        } else {
            $response['data'] = $preparedData;
        }
        return response()->json($response, $statusCode);
    }

    public function successWithNestedPagination($data, $message = '', $statusCode = 200)
    {
        $response = [
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode
        ];
        if (is_array($data) && count($data) > 0) {
            $firstKey = array_key_first($data);
            $firstValue = $data[$firstKey];            
            $preparedData = $this->prepareData($firstValue, $message);
            if (empty($preparedData)) {
                return $this->notFound("Record Not Found", 404);
            }
            $response['data'] = [];
            if (is_array($preparedData) && isset($preparedData['items']) && isset($preparedData['pagination'])) {
                $response['data'][$firstKey] = $preparedData['items'];
                $response['pagination'] = $preparedData['pagination'];
            } else {
                $response['data'][$firstKey] = $preparedData;
            }
            foreach ($data as $key => $value) {
                if ($key !== $firstKey) {
                    $response['data'][$key] = $value;
                }
            }
        } else {
            $response['data'] = $data;
        }
        return response()->json($response, $statusCode);
    }


    public function errorWithData($data, $message = '', $statusCode = 500)
    {
        $data = $this->prepareData($data, $message);
        if (empty($data)) {
            return $this->notFound("Record Not Found", 404);
        }
        return response()->json([
            'data' => $data,
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode,
        ], $statusCode);
    }

    public function withToken($data, $token = '', $message = '', $statusCode = 200, $permissions = [], $settings = [])
    {
        $data = $this->prepareData($data, $message);
        if (empty($data)) {
            return $this->notFound("Record Not Found", 404);
        }
        return response()->json([
            'data' => $data,
            'token' => $token,
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode,
            'settings' => $settings
        ], $statusCode);
    }

    public function withCompanyToken($data, $token = '', $admin = '', $message = '', $statusCode = 200)
    {
        $data = $this->prepareData($data, $message);
        if (empty($data)) {
            return $this->notFound("Record Not Found", 404);
        }
        return response()->json([
            'data' => $data,
            'token' => $token,
            'admin' => $admin,
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode,
        ], $statusCode);
    }

    public function withCompanyTokenWithSubscription($data, $token = '', $admin = '',  $subscription = '', $permissions = '', $settings = '', $message = '', $statusCode = 200)
    {
        $data = $this->prepareData($data, $message);
        if (empty($data)) {
            return $this->notFound("Record Not Found", 404);
        }
        return response()->json([
            'data' => $data,
            'token' => $token,
            'admin' => $admin,
            'subscription' => $subscription ?? null,
            'permissions' => $permissions ?? null,
            'settings' => $settings ?? null,
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode,
        ], $statusCode);
    }

    public function WithFilter($data, $filter = null, $message = '', $statusCode = 200)
    {
        $preparedData = $this->prepareData($data, $message);
        if (empty($preparedData)) {
            return $this->notFound("Record Not Found", 404);
        }
        $response = [
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode,
            'filter' => $filter
        ];
        if (is_array($preparedData) && isset($preparedData['items']) && isset($preparedData['pagination'])) {
            $response['data'] = $preparedData['items'];
            $response['totalCount'] = $preparedData['pagination']['total'];
            $response['pagination'] = $preparedData['pagination'];
        } else {
            $response['data'] = $preparedData;
            $response['totalCount'] = count($preparedData);
        }
        return response()->json($response, $statusCode);
    }

    public function withCount($data, $message = '', $statusCode = 200)
    {
        $preparedData = $this->prepareData($data, $message);
        if (empty($preparedData)) {
            return $this->notFound("Record Not Found", 404);
        }
        $response = [
            'message' => $message ? $message : "success",
            'statusCode' => $statusCode
        ];
        if (is_array($preparedData) && isset($preparedData['items']) && isset($preparedData['pagination'])) {
            $response['data'] = $preparedData['items'];
            $response['totalCount'] = $preparedData['pagination']['total'];
            $response['pagination'] = $preparedData['pagination'];
        } else {
            $response['data'] = $preparedData;
            $response['totalCount'] = count($preparedData);
        }
        return response()->json($response, $statusCode);
    }


    public function error($message = '', $statusCode = 500)
    {
        return response()->json([
            'message' => $message ? $message : "something went wrong",
            'statusCode' => $statusCode,
        ], $statusCode);
    }

    public function notFound($message = '', $statusCode = 404)
    {
        return response()->json([
            'message' => $message ? $message : 'Record Not Found',
            'statusCode' => $statusCode,
        ], $statusCode);
    }

    public function message($message = '', $statusCode = 200)
    {
        return response()->json([
            'message' => $message ? $message : 'Record Not Found',
            'statusCode' => $statusCode,
        ], $statusCode);
    }

    private function prepareData($data, $message)
    {
        if ($data instanceof \Illuminate\Database\Eloquent\Model) {
            $data = $data->toArray();
        } elseif ($data instanceof \Illuminate\Database\Eloquent\Builder) {
            // Add default ordering by created_at desc to get latest data first
            $data = $data->orderBy('created_at', 'desc');
            if (request()->has('per_page') || request()->has('page')) {
                $perPage = request()->get('per_page', 10);
                $data = $data->paginate($perPage);
                if ($data->isEmpty()) {
                    return null;
                }
                $currentParams = request()->query();
                $nextPageUrl = $data->hasMorePages() ? 
                    request()->fullUrlWithQuery(array_merge($currentParams, ['page' => $data->currentPage() + 1])) : null;
                $prevPageUrl = $data->currentPage() > 1 ? 
                    request()->fullUrlWithQuery(array_merge($currentParams, ['page' => $data->currentPage() - 1])) : null;
                return [
                    'items' => $data->items(),
                    'pagination' => [
                        'total' => $data->total(),
                        'per_page' => $data->perPage(),
                        'current_page' => $data->currentPage(),
                        'last_page' => $data->lastPage(),
                        'from' => $data->firstItem(),
                        'to' => $data->lastItem(),
                        'next_page_url' => $nextPageUrl,
                        'prev_page_url' => $prevPageUrl
                    ]
                ];
            }
            $data = $data->get();
            if ($data->isEmpty()) {
                return null;
            }
            return $data->toArray();
        } elseif ($data instanceof \Illuminate\Support\Collection) {
            if ($data->isEmpty()) {
                return null;
            }
            // Sort collection by created_at desc to get latest data first
            $data = $data->sortByDesc(function ($item) {
                if (is_array($item) && isset($item['created_at'])) {
                    return $item['created_at'];
                } elseif (is_object($item) && isset($item->created_at)) {
                    return $item->created_at;
                }
                return null;
            });
            if (request()->has('per_page') || request()->has('page')) {
                $page = (int)request()->get('page', 1);
                $perPage = (int)request()->get('per_page', 10);
                $total = $data->count();
                $items = $data->forPage($page, $perPage);
                $lastPage = ceil($total / $perPage);
                $currentParams = request()->query();
                $nextPageUrl = $page < $lastPage ? 
                    request()->fullUrlWithQuery(array_merge($currentParams, ['page' => $page + 1])) : null; 
                $prevPageUrl = $page > 1 ? 
                    request()->fullUrlWithQuery(array_merge($currentParams, ['page' => $page - 1])) : null;
                return [
                    'items' => array_values($items->toArray()),
                    'pagination' => [
                        'total' => $total,
                        'per_page' => $perPage,
                        'current_page' => $page,
                        'last_page' => $lastPage,
                        'from' => ($page - 1) * $perPage + 1,
                        'to' => min($page * $perPage, $total),
                        'next_page_url' => $nextPageUrl,
                        'prev_page_url' => $prevPageUrl
                    ]
                ];
            }
            return $data->values()->toArray();
        } elseif ($data instanceof \Illuminate\Database\Eloquent\Relations\HasMany) {
            // Add default ordering by created_at desc to get latest data first
            $data = $data->orderBy('created_at', 'desc');
            if (request()->has('per_page') || request()->has('page')) {
                $perPage = request()->get('per_page', 10);
                $data = $data->paginate($perPage);
                if ($data->isEmpty()) {
                    return null;
                }
                $currentParams = request()->query();
                $nextPageUrl = $data->hasMorePages() ? 
                    request()->fullUrlWithQuery(array_merge($currentParams, ['page' => $data->currentPage() + 1])) : null;
                $prevPageUrl = $data->currentPage() > 1 ? 
                    request()->fullUrlWithQuery(array_merge($currentParams, ['page' => $data->currentPage() - 1])) : null;
                return [
                    'items' => $data->items(),
                    'pagination' => [
                        'total' => $data->total(),
                        'per_page' => $data->perPage(),
                        'current_page' => $data->currentPage(),
                        'last_page' => $data->lastPage(),
                        'from' => $data->firstItem(),
                        'to' => $data->lastItem(),
                        'next_page_url' => $nextPageUrl,
                        'prev_page_url' => $prevPageUrl
                    ]
                ];
            }
            $data = $data->get();
            if ($data->isEmpty()) {
                return null;
            }
            return $data->toArray();
        }
        return empty($data) ? null : $data;
    }
}
