<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Booking;
use App\Models\Quote;
use App\Models\Job;
use App\Models\JobApplication;
use App\Models\Blog;
use App\Models\Truck;
use App\Models\Testimonial;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $filter = $request->get('filter', 'this_year');
        $dateRange = $this->getDateRange($filter);
        
        // Get basic metrics
        $metrics = $this->getDashboardMetrics($dateRange);
        
        // Get revenue data for chart
        $revenueData = $this->getRevenueData($dateRange);
        
        // Get top locations (based on bookings/quotes)
        $topLocations = $this->getTopLocations($dateRange);
        
        // Get recent quotes/orders
        $recentQuotes = $this->getRecentQuotes($dateRange, 10);
        
        // Get recent customers
        $recentCustomers = $this->getRecentCustomers($dateRange, 10);
        
        // Get statistics
        $statistics = $this->getStatistics($dateRange);
        
        return view('Admin.dashboard', compact(
            'metrics',
            'revenueData',
            'topLocations',
            'recentQuotes',
            'recentCustomers',
            'statistics',
            'filter'
        ));
    }
    
    /**
     * Get filtered metrics data for dashboard chart
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFilteredDashboardMetrics(Request $request)
    {
        $filter = $request->get('filter', 'all');
        $dateRange = $this->getDateRange($filter);
        
        $metrics = $this->getDashboardMetrics($dateRange);
        $revenueData = $this->getRevenueData($dateRange);
        $topLocations = $this->getTopLocations($dateRange);
        $statistics = $this->getStatistics($dateRange);
        
        return response()->json([
            'success' => true,
            'metrics' => $metrics,
            'revenueData' => $revenueData,
            'topLocations' => $topLocations,
            'statistics' => $statistics
        ]);
    }
    
    /**
     * Export dashboard report to CSV
     */
    public function exportReport(Request $request)
    {
        $filter = $request->get('filter', 'all');
        $dateRange = $this->getDateRange($filter);
        
        $metrics = $this->getDashboardMetrics($dateRange);
        $quotes = Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->with(['user', 'booking'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        $filename = 'dashboard_report_' . date('Y-m-d_His') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];
        
        $callback = function() use ($metrics, $quotes, $filter) {
            $file = fopen('php://output', 'w');
            
            // Write filter info
            fputcsv($file, ['Dashboard Report']);
            fputcsv($file, ['Filter Period:', ucfirst($filter)]);
            fputcsv($file, ['Generated:', date('Y-m-d H:i:s')]);
            fputcsv($file, []);
            
            // Write metrics
            fputcsv($file, ['Metrics']);
            fputcsv($file, ['Total Bookings', $metrics['total_bookings']]);
            fputcsv($file, ['Completed Bookings', $metrics['completed_bookings']]);
            fputcsv($file, ['Pending Bookings', $metrics['pending_bookings']]);
            fputcsv($file, ['Total Quotes', $metrics['total_quotes']]);
            fputcsv($file, ['Approved Quotes', $metrics['approved_quotes']]);
            fputcsv($file, ['Total Orders', $metrics['total_orders']]);
            fputcsv($file, ['Total Revenue', '$' . number_format($metrics['total_revenue'], 2)]);
            fputcsv($file, ['Total Customers', $metrics['total_customers']]);
            fputcsv($file, ['Job Applications', $metrics['job_applications']]);
            fputcsv($file, ['Active Blogs', $metrics['active_blogs']]);
            fputcsv($file, ['Active Trucks', $metrics['active_trucks']]);
            fputcsv($file, ['Active Testimonials', $metrics['active_testimonials']]);
            fputcsv($file, []);
            
            // Write quotes/orders
            fputcsv($file, ['Quotes/Orders Details']);
            fputcsv($file, ['Quote Number', 'Order Number', 'Customer Name', 'Customer Email', 'Total Cost', 'Status', 'Quote Type', 'Date']);
            
            foreach ($quotes as $quote) {
                fputcsv($file, [
                    $quote->quote_number ?? 'N/A',
                    $quote->order_number ?? 'N/A',
                    $quote->user->name ?? 'N/A',
                    $quote->user->email ?? 'N/A',
                    '$' . number_format($quote->total_cost ?? 0, 2),
                    ucfirst($quote->status ?? 'N/A'),
                    ucfirst($quote->quote_type ?? 'N/A'),
                    $quote->created_at->format('Y-m-d H:i:s')
                ]);
            }
            
            fclose($file);
        };
        
        return Response::stream($callback, 200, $headers);
    }
    
    /**
     * Get date range based on filter
     */
    private function getDateRange($filter)
    {
        $start = Carbon::now()->startOfDay();
        $end = Carbon::now()->endOfDay();
        
        switch ($filter) {
            case 'today':
                $start = Carbon::today()->startOfDay();
                $end = Carbon::today()->endOfDay();
                break;
            case 'yesterday':
                $start = Carbon::yesterday()->startOfDay();
                $end = Carbon::yesterday()->endOfDay();
                break;
            case 'this_week':
                $start = Carbon::now()->startOfWeek();
                $end = Carbon::now()->endOfWeek();
                break;
            case 'last_week':
                $start = Carbon::now()->subWeek()->startOfWeek();
                $end = Carbon::now()->subWeek()->endOfWeek();
                break;
            case 'this_month':
                $start = Carbon::now()->startOfMonth();
                $end = Carbon::now()->endOfMonth();
                break;
            case 'last_month':
                $start = Carbon::now()->subMonth()->startOfMonth();
                $end = Carbon::now()->subMonth()->endOfMonth();
                break;
            case 'last_6_months':
                $start = Carbon::now()->subMonths(6)->startOfMonth();
                $end = Carbon::now()->endOfMonth();
                break;
            case 'this_year':
                $start = Carbon::now()->startOfYear();
                $end = Carbon::now()->endOfYear();
                break;
            case 'last_year':
                $start = Carbon::now()->subYear()->startOfYear();
                $end = Carbon::now()->subYear()->endOfYear();
                break;
            case 'all':
            default:
                $start = Carbon::parse('2020-01-01');
                $end = Carbon::now()->endOfDay();
                break;
        }
        
        return [
            'start' => $start,
            'end' => $end,
            'filter' => $filter
        ];
    }
    
    /**
     * Get dashboard metrics
     */
    private function getDashboardMetrics($dateRange)
    {
        return [
            'total_bookings' => Booking::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])->count(),
            'completed_bookings' => Booking::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('is_completed', true)->count(),
            'pending_bookings' => Booking::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('is_completed', false)->count(),
            'inprogress_bookings' => Booking::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('status', 'in_progress')->count(),
            'cancelled_bookings' => Booking::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('status', 'cancelled')->count(),
            'total_quotes' => Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])->count(),
            'approved_quotes' => Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('status', 'approved')->count(),
            'pending_quotes' => Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('status', 'pending')->count(),
            'total_orders' => Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('quote_type', 'order')->count(),
            'total_revenue' => Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->where('quote_type', 'order')
                ->sum('total_cost') ?? 0,
            'total_customers' => User::where('role_id', 2)
                ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
                ->count(),
            'job_applications' => JobApplication::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])->count(),
            'active_jobs' => Job::where('is_active', true)->count(),
            'active_blogs' => Blog::where('active', 1)->count(),
            'total_blogs' => Blog::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])->count(),
            'active_trucks' => Truck::where('is_active', true)->count(),
            'total_trucks' => Truck::count(),
            'active_testimonials' => Testimonial::where('is_active', true)->count(),
            'total_testimonials' => Testimonial::count(),
        ];
    }
    
    /**
     * Get revenue data for chart (monthly)
     */
    private function getRevenueData($dateRange)
    {
        $revenueData = [];
        $start = $dateRange['start']->copy()->startOfMonth();
        $end = $dateRange['end']->copy()->endOfMonth();
        
        $current = $start->copy();
        while ($current->lte($end)) {
            $monthStart = $current->copy()->startOfMonth();
            $monthEnd = $current->copy()->endOfMonth();
            
            $revenue = Quote::where('quote_type', 'order')
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->sum('total_cost') ?? 0;
            
            $revenueData[] = [
                'month' => $current->format('M'),
                'month_full' => $current->format('F Y'),
                'revenue' => round($revenue, 2),
                'year' => $current->year
            ];
            
            $current->addMonth();
        }
        
        return $revenueData;
    }
    
    /**
     * Get top locations based on bookings
     */
    private function getTopLocations($dateRange)
    {
        $bookings = Booking::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->with(['pickUpProperty', 'quote'])
            ->get();
        
        $locations = [];
        foreach ($bookings as $booking) {
            $location = 'Unknown Location';
            
            // Get location from Property table's location field
            if ($booking->pickUpProperty && !empty($booking->pickUpProperty->location)) {
                $location = trim($booking->pickUpProperty->location);
            } elseif ($booking->pickUpProperty) {
                // Fallback if location is empty but property exists
                $location = 'Location #' . $booking->pickUpProperty->id;
            }
            
            if (!isset($locations[$location])) {
                $locations[$location] = [
                    'location' => $location,
                    'count' => 0,
                    'revenue' => 0
                ];
            }
            
            $locations[$location]['count']++;
            if ($booking->quote && $booking->quote->total_cost) {
                $locations[$location]['revenue'] += $booking->quote->total_cost;
            }
        }
        
        // Sort by revenue descending
        usort($locations, function($a, $b) {
            return $b['revenue'] <=> $a['revenue'];
        });
        
        // Get top 10
        return array_slice($locations, 0, 10);
    }
    
    /**
     * Get recent quotes/orders
     */
    private function getRecentQuotes($dateRange, $limit = 10)
    {
        return Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->with(['user', 'booking'])
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }
    
    /**
     * Get recent customers
     */
    private function getRecentCustomers($dateRange, $limit = 10)
    {
        return User::where('role_id', 2)
            ->whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }
    
    /**
     * Get statistics for charts
     */
    private function getStatistics($dateRange)
    {
        // Quote type breakdown
        $quoteTypes = Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->select('quote_type', DB::raw('count(*) as count'))
            ->groupBy('quote_type')
            ->get()
            ->pluck('count', 'quote_type');
        
        // Status breakdown
        $quoteStatuses = Quote::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');
        
        // Booking status breakdown
        $bookingStatuses = Booking::whereBetween('created_at', [$dateRange['start'], $dateRange['end']])
            ->select('status', DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');
        
        return [
            'quote_types' => $quoteTypes,
            'quote_statuses' => $quoteStatuses,
            'booking_statuses' => $bookingStatuses,
        ];
    }
}
