<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\API\Traits\EmailTrait;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\StripeTrait;
use App\Http\Controllers\API\Traits\QuoteCalculationTrait;
use App\Models\AssembleDisassembleCategory;
use App\Models\ChildCategory;
use App\Models\Quote;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\Property;
use App\Models\StorageItem;
use App\Models\GeneralSettings;
use App\General\GeneralSettingsClass;
use Carbon\Carbon;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\Log;


class QuoteController extends Controller
{
    use EmailTrait, StripeTrait, QuoteCalculationTrait, JsonResponseTrait;
    /**
     * Display a listing of quotes/orders
     */
    public function index(Request $request)
    {
        $query = Quote::with(['booking.pickUpProperty.propertyDetails', 'booking.dropOffProperty.propertyDetails', 'truck', 'user']);

        // Keyword search
        if ($request->filled('keyword')) {
            $keyword = $request->keyword;
            $query->where(function ($q) use ($keyword) {
                $q->where('quote_number', 'like', "%{$keyword}%")
                    ->orWhere('order_number', 'like', "%{$keyword}%")
                    ->orWhere('uuid', 'like', "%{$keyword}%")
                    ->orWhereHas('user', function ($userQuery) use ($keyword) {
                        $userQuery->where('name', 'like', "%{$keyword}%")
                            ->orWhere('email', 'like', "%{$keyword}%")
                            ->orWhere('phone', 'like', "%{$keyword}%");
                    })
                    ->orWhereHas('booking', function ($bookingQuery) use ($keyword) {
                        $bookingQuery->where('name', 'like', "%{$keyword}%")
                            ->orWhere('email', 'like', "%{$keyword}%");
                    });
            });
        }

        // Quote type filter
        if ($request->filled('quote_type')) {
            $query->where('quote_type', $request->quote_type);
        }

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Calculation type filter
        if ($request->filled('calculation_type')) {
            $query->where('calculation_type', $request->calculation_type);
        }

        // Job type filter (stored in raw_data JSON column)
        if ($request->filled('job_type')) {
            $query->whereJsonContains('raw_data->job_type', $request->job_type);
        }

        // Payment method filter
        if ($request->filled('payment_method')) {
            $paymentMethod = $request->payment_method;
            $query->where(function ($q) use ($paymentMethod) {
                $q->where('payment_method', $paymentMethod)
                    ->orWhereJsonContains('raw_data->payment_method', $paymentMethod);
            });
        }

        // Date range filter
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Order by created_at descending (newest first), then by id descending as secondary sort
        // Use pagination with 10 items per page
        $perPage = $request->get('per_page', 10);
        $quotes = $query->orderBy('created_at', 'desc')
            ->orderBy('id', 'desc')
            ->paginate($perPage)
            ->withQueryString(); // Preserve query parameters in pagination links

        return view('Admin.Quote.index', compact('quotes'));
    }

    /**
     * Display the specified quote details
     */
    public function show($id)
    {
        $quote = Quote::with([
            'booking.bookingItems.entity',
            'booking.pickUpProperty.propertyDetails',
            'booking.dropOffProperty.propertyDetails',
            'truck',
            'user'
        ])->findOrFail($id);

        $childCategories = ChildCategory::orderBy('title')->get(['id', 'title']);

        return view('Admin.Quote.show', compact('quote', 'childCategories'));
    }

    /**
     * Show the form for editing the specified quote
     */

    public function edit($id)
    {

        return view('Admin.Quote.edit');
    }


    public function editData($id)
    {
        $quote = Quote::with([
            'booking.bookingItems.entity',
            'booking.pickUpProperty.propertyDetails',
            'booking.dropOffProperty.propertyDetails',
            'truck',
            'user'
        ])->findOrFail($id);

        // Check if quote is editable
        if ($quote->deposit_payment_status === 'succeeded' || $quote->quote_type === 'order') {
            return response()->json([
                'statusCode' => 403,
                'message' => 'This quote cannot be edited. Deposit has been paid or quote type is order.',
                'error' => 'Quote is not editable'
            ], 403);
        }

        $childCategories = ChildCategory::orderBy('title')->get(['id', 'title', 'volumn']);
        $assembleCategories = AssembleDisassembleCategory::where('is_active', true)->orderBy('name')->get(['id', 'name', 'assemble_price', 'disassemble_price', 'both_price']);
        $storageItems = StorageItem::where('is_active', true)->orderBy('title')->get(['id', 'title', 'price']);
        $pickUpProperty = Property::where('uuid', $quote->uuid)->where('type', 'pick_up')->first();
        $dropOffProperty = Property::where('uuid', $quote->uuid)->where('type', 'drop_off')->first();
        return $this->success([
            'quote' => $quote,
            'childCategories' => $childCategories,
            'assembleCategories' => $assembleCategories,
            'storageItems' => $storageItems,
            'pickUpProperty' => $pickUpProperty,
            'dropOffProperty' => $dropOffProperty
        ], 'Quote fetched successfully');
    }

    /**
     * Update quote status
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:pending,approved,rejected'
        ]);
        $quote = Quote::findOrFail($id);
        $quote->status = $request->status;
        $quote->save();
        return response()->json([
            'status' => 'success',
            'message' => 'Quote status updated successfully.',
            'status_value' => $quote->status
        ]);
    }

    /**
     * Update quote
     */
    public function update(Request $request)
    {
        $uuid = $request->uuid;
        if (!$uuid) {
            return $this->error('UUID is required', 422);
        }

        // Check if quote is editable
        $quote = Quote::where('uuid', $uuid)->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        
        if ($quote->deposit_payment_status === 'succeeded' || $quote->quote_type === 'order') {
            return $this->error('This quote cannot be edited. Deposit has been paid or quote type is order.', 403);
        }
        $createdRegularItems = [];
        $createdAssembleItems = [];
        $createdStorageItems = [];
        $result = [];
        $payloadRegularItemIds = [];
        $payloadAssembleItemIds = [];
        $payloadStorageItemIds = [];

        if ($request->has('items') && is_array($request->items)) {
            foreach ($request->items as $item) {
                if (!isset($item['category_id'])) {
                    continue;
                }
                $categoryId = $item['category_id'];
                $payloadRegularItemIds[] = $categoryId;

                $bookingItem = BookingItem::where('uuid', $uuid)
                    ->where('entity_type', ChildCategory::class)
                    ->where('entity_id', $categoryId)
                    ->first();
                // Ensure quantity is at least 1
                $quantity = max(1, (int)($item['quantity'] ?? 1));
                
                $itemData = [
                    'uuid' => $uuid,
                    'entity_type' => ChildCategory::class,
                    'entity_id' => $categoryId,
                    'is_assemble_disassemble' => false,
                    'quantity' => $quantity,
                ];
                if ($bookingItem) {
                    $fillableFields = [
                        'entity_type',
                        'entity_id',
                        'is_assemble_disassemble',
                        'quantity'
                    ];
                    foreach ($fillableFields as $field) {
                        if (!array_key_exists($field, $itemData)) {
                            $itemData[$field] = null;
                        }
                    }
                    $bookingItem->update($itemData);
                    $bookingItem->refresh();
                } else {
                    $bookingItem = BookingItem::create($itemData);
                }
                $createdRegularItems[] = $bookingItem->load(['entity']);
            }
        }

        // Delete regular items that are not in the payload (only if 'items' key exists in request)
        if ($request->has('items')) {
            $query = BookingItem::where('uuid', $uuid)
                ->where('entity_type', ChildCategory::class)
                ->where('is_assemble_disassemble', false);

            // If payload has items, delete those not in the list; if empty array, delete all
            if (!empty($payloadRegularItemIds)) {
                $query->whereNotIn('entity_id', $payloadRegularItemIds);
            }
            // If empty array, the query will delete all regular items (no whereNotIn condition)
            $query->delete();
        }

        // Handle assemble/disassemble items
        if ($request->has('assemble_items') && is_array($request->assemble_items)) {
            foreach ($request->assemble_items as $item) {
                if (!isset($item['assemble_disassemble_category_id'])) {
                    continue;
                }
                $assembleCategoryId = $item['assemble_disassemble_category_id'];
                $assembleType = $item['assemble_disassemble_type'] ?? 'null';
                $payloadAssembleItemIds[] = $assembleCategoryId;

                // Find booking item by entity_id AND assemble_disassemble_type
                // This ensures we update the correct item if same entity has different types
                $bookingItem = BookingItem::where('uuid', $uuid)
                    ->where('entity_type', AssembleDisassembleCategory::class)
                    ->where('entity_id', $assembleCategoryId)
                    ->where('assemble_disassemble_type', $assembleType)
                    ->first();
                // Ensure quantity is at least 1
                $quantity = max(1, (int)($item['quantity'] ?? 1));
                
                $itemData = [
                    'uuid' => $uuid,
                    'entity_type' => AssembleDisassembleCategory::class,
                    'entity_id' => $assembleCategoryId,
                    'is_assemble_disassemble' => true,
                    'assemble_disassemble_type' => $assembleType,
                    'quantity' => $quantity,
                ];
                if ($bookingItem) {
                    // If updating, set missing fields to null
                    $fillableFields = [
                        'entity_type',
                        'entity_id',
                        'is_assemble_disassemble',
                        'assemble_disassemble_type',
                        'quantity'
                    ];
                    foreach ($fillableFields as $field) {
                        if (!array_key_exists($field, $itemData)) {
                            $itemData[$field] = null;
                        }
                    }
                    $bookingItem->update($itemData);
                    $bookingItem->refresh();
                } else {
                    $bookingItem = BookingItem::create($itemData);
                }

                // Format assemble items - only include selected pricing, not all entity fields
                $bookingItem->load(['entity']);
                $assembleItem = [
                    'id' => $bookingItem->id,
                    'uuid' => $bookingItem->uuid,
                    'entity_type' => $bookingItem->entity_type,
                    'entity_id' => $bookingItem->entity_id,
                    'quantity' => $bookingItem->quantity,
                    'is_assemble_disassemble' => $bookingItem->is_assemble_disassemble,
                    'assemble_disassemble_type' => $bookingItem->assemble_disassemble_type,
                    'created_at' => $bookingItem->created_at,
                    'updated_at' => $bookingItem->updated_at,
                    'entity' => [
                        'id' => $bookingItem->entity->id ?? null,
                        'name' => $bookingItem->entity->name ?? null,
                        'description' => $bookingItem->entity->description ?? null,
                        'image' => $bookingItem->entity->image ?? null,
                    ],
                    'selected_pricing' => $bookingItem->selected_pricing,
                ];
                $createdAssembleItems[] = $assembleItem;
            }
        }

        if ($request->has('storage_items') && is_array($request->storage_items)) {
            foreach ($request->storage_items as $item) {
                if (!isset($item['storage_item_id'])) {
                    continue;
                }

                $storageItemId = $item['storage_item_id'];
                $payloadStorageItemIds[] = $storageItemId;

                $bookingItem = BookingItem::where('uuid', $uuid)
                    ->where('entity_type', StorageItem::class)
                    ->where('entity_id', $storageItemId)
                    ->first();

                // Ensure quantity is at least 1
                $quantity = max(1, (int)($item['quantity'] ?? 1));
                
                $itemData = [
                    'uuid' => $uuid,
                    'entity_type' => StorageItem::class,
                    'entity_id' => $storageItemId,
                    'is_assemble_disassemble' => false,
                    'assemble_disassemble_type' => null,
                    'quantity' => $quantity,
                ];

                if ($bookingItem) {
                    $fillableFields = [
                        'entity_type',
                        'entity_id',
                        'is_assemble_disassemble',
                        'assemble_disassemble_type',
                        'quantity',
                    ];

                    foreach ($fillableFields as $field) {
                        if (!array_key_exists($field, $itemData)) {
                            $itemData[$field] = null;
                        }
                    }

                    $bookingItem->update($itemData);
                    $bookingItem->refresh();
                } else {
                    $bookingItem = BookingItem::create($itemData);
                }

                $createdStorageItems[] = $bookingItem->load(['entity']);
            }
        }
        // Delete assemble/disassemble items that are not in the payload
        // If 'assemble_items' key exists in request, sync based on payload
        // If 'assemble_items' key doesn't exist but 'items' exists, delete all assemble items
        if ($request->has('assemble_items')) {
            $queryAssemble = BookingItem::where('uuid', $uuid)
                ->where('entity_type', AssembleDisassembleCategory::class)
                ->where('is_assemble_disassemble', true);

            // If payload has items, delete those not in the list; if empty array, delete all
            if (!empty($payloadAssembleItemIds)) {
                $queryAssemble->whereNotIn('entity_id', $payloadAssembleItemIds);
            }
            // If empty array, the query will delete all assemble items (no whereNotIn condition)
            $queryAssemble->delete();
        } elseif ($request->has('items')) {
            // If only 'items' is provided (no 'assemble_items'), delete all assemble items
            // This ensures the database matches exactly what's in the payload
            BookingItem::where('uuid', $uuid)
                ->where('entity_type', AssembleDisassembleCategory::class)
                ->where('is_assemble_disassemble', true)
                ->delete();
        }
        if ($request->has('storage_items')) {
            $storageDeleteQuery = BookingItem::where('uuid', $uuid)
                ->where('entity_type', StorageItem::class)
                ->where('is_assemble_disassemble', false);

            if (!empty($payloadStorageItemIds)) {
                $storageDeleteQuery->whereNotIn('entity_id', $payloadStorageItemIds);
            }

            $storageDeleteQuery->delete();
        } elseif ($request->has('items')) {
            BookingItem::where('uuid', $uuid)
                ->where('entity_type', StorageItem::class)
                ->where('is_assemble_disassemble', false)
                ->delete();
        }

        $result['booking_items'] = array_merge($createdRegularItems, $createdStorageItems);
        $result['items'] = $createdRegularItems;
        $result['assemble_items'] = $createdAssembleItems;
        $result['storage_items'] = $createdStorageItems;
        $result['uuid'] = $uuid;

        $quote = Quote::where('uuid', $uuid)->first();
        if ($quote) {

            if ($request->has('status')) {
                $quote->status = $request->status;
                $quote->save();
            }

            // Get raw_data as array to modify it properly
            $rawData = $quote->raw_data ?? [];
            $needsSave = false;

            if ($request->has('job_type')) {
                $rawData['job_type'] = $request->job_type;
                $needsSave = true;
            }
            if ($request->has('extra_movers')) {
                $rawData['extra_movers'] = $request->extra_movers;
                $needsSave = true;
            }
            if ($request->has('base_to_pickup_time')) {
                $rawData['base_to_pickup_time'] = $request->base_to_pickup_time;
                $needsSave = true;
            }
            if ($request->has('dropof_to_base')) {
                $rawData['dropof_to_base'] = $request->dropof_to_base;
                $needsSave = true;
            }
            if ($request->has('pickup_to_dropof')) {
                $rawData['pickup_to_dropof'] = $request->pickup_to_dropof;
                $needsSave = true;
            }

            // Handle discount (percentage or fixed)
            if ($request->has('discount')) {
                $discountValue = (float) $request->discount;
                $discountType = $request->has('discount_type') ? $request->discount_type : 'percentage';
                $rawData['discount'] = $discountValue;
                $rawData['discount_type'] = $discountType;
                $needsSave = true;
            }

            // Update raw_data if any changes were made
            if ($needsSave) {
                $quote->raw_data = $rawData;
                $quote->save();
            }
        }

        // Update move_date in pick_up property and booking
        $pickupProperty = Property::where('uuid', $uuid)->where('type', 'pick_up')->first();
        if ($pickupProperty) {
            if ($request->has('move_date')) {
                $pickupProperty->move_date = Carbon::parse($request->move_date)->format('Y-m-d');
                $pickupProperty->save();
            }
            if ($request->has('pickup_location')) {
                $pickupProperty->location = $request->pickup_location;
                $pickupProperty->save();
            }
        }
        
        // Update booking move_date
        $booking = Booking::where('uuid', $uuid)->first();
        if ($booking && $request->has('move_date')) {
            $booking->move_date = Carbon::parse($request->move_date)->format('Y-m-d');
            $booking->save();
        }
        
        $dropoffProperty = Property::where('uuid', $uuid)->where('type', 'drop_off')->first();
        if ($dropoffProperty && $request->has('dropoff_location')) {
            $dropoffProperty->location = $request->dropoff_location;
            $dropoffProperty->save();
        }

        // Recalculate quote after items are added/removed
        // $booking is already defined above, refresh it to get latest data
        if (!$booking) {
            $booking = Booking::where('uuid', $uuid)->first();
        }
        if ($booking && $booking->truck_id) {
            // Refresh booking to get latest items
            $booking->refresh();
            // Clear relationship cache to ensure fresh data
            $booking->unsetRelation('bookingItems');
            $booking->load(['bookingItems.entity', 'truck']);

            // Get existing quote to preserve distance and other fields
            $quote = Quote::where('uuid', $uuid)->first();
            $distanceKm = $quote ? $quote->distance_km : null;

            // Create a request object with job_type and other calculation parameters
            // Priority: Use request values if available, otherwise use values from raw_data
            $requestForCalculation = new Request();
            
            // Job type: use from request if available, otherwise from raw_data
            if ($request->has('job_type')) {
                $requestForCalculation->merge(['job_type' => $request->job_type]);
            } elseif ($quote && isset($quote->raw_data['job_type'])) {
                $requestForCalculation->merge(['job_type' => $quote->raw_data['job_type']]);
            }
            
            // Extra movers: use from request if available, otherwise from raw_data
            if ($request->has('extra_movers')) {
                $requestForCalculation->merge(['extra_movers' => $request->extra_movers]);
            } elseif ($quote && isset($quote->raw_data['extra_movers'])) {
                $requestForCalculation->merge(['extra_movers' => $quote->raw_data['extra_movers']]);
            }
            
            // Callout fee times: use from request if available, otherwise from raw_data
            if ($request->has('base_to_pickup_time')) {
                $requestForCalculation->merge([
                    'base_to_pickup_time' => $request->base_to_pickup_time,
                ]);
            } elseif ($quote && isset($quote->raw_data['base_to_pickup_time'])) {
                $requestForCalculation->merge([
                    'base_to_pickup_time' => $quote->raw_data['base_to_pickup_time'],
                ]);
            }
            
            if ($request->has('dropof_to_base')) {
                $requestForCalculation->merge([
                    'dropof_to_base' => $request->dropof_to_base,
                ]);
            } elseif ($quote && isset($quote->raw_data['dropof_to_base'])) {
                $requestForCalculation->merge([
                    'dropof_to_base' => $quote->raw_data['dropof_to_base'],
                ]);
            }
            
            if ($request->has('pickup_to_dropof')) {
                $requestForCalculation->merge([
                    'pickup_to_dropof' => $request->pickup_to_dropof,
                ]);
            } elseif ($quote && isset($quote->raw_data['pickup_to_dropof'])) {
                $requestForCalculation->merge([
                    'pickup_to_dropof' => $quote->raw_data['pickup_to_dropof'],
                ]);
            }
            
            // Check if items are being managed in the request
            // If 'items', 'assemble_items', or 'storage_items' keys exist (even if empty arrays),
            // it means we're explicitly managing items, so don't preserve truck volume
            $isManagingItems = $request->has('items') || 
                                $request->has('assemble_items') || 
                                $request->has('storage_items');
            
            // Only preserve volume for truck_only if we're NOT managing items at all
            // If items are being managed (even if empty arrays), calculate from items (which will be 0 if removed)
            // This ensures:
            // 1. When items are added to truck_only -> calculation changes to items-based
            // 2. When items are removed from truck_only -> volume is zeroed (no truck volume preserved)
            if (!$isManagingItems && $quote && $quote->calculation_type === 'truck_only' && $quote->total_cubic_meters) {
                // Not managing items at all - preserve truck volume for truck_only calculations
                $requestForCalculation->merge([
                    'required_truck_cubic_meter' => $quote->total_cubic_meters,
                ]);
            }
            // If $isManagingItems is true, we don't preserve truck volume
            // The calculation will use items from the database (which will be 0 if all items were removed)

            // Calculate quote using trait - this will recalculate all costs including storage and assemble/disassemble
            $quoteData = $this->calculateQuote($booking, $distanceKm, 0, $requestForCalculation);

            if ($quoteData) {
                $quoteData['uuid'] = $uuid;
                $quoteData['user_id'] = $booking->user_id;
                $quoteData['distance_km'] = $distanceKm;

                // Ensure total_cost is always included and properly calculated
                // Always recalculate to ensure storage and assemble costs are included
                $totalCost = 0;
                if (isset($quoteData['movers_cost'])) {
                    $totalCost += (float) $quoteData['movers_cost'];
                }
                // Check for callout_fee in response or raw_data
                $calloutFee = 0;
                if (isset($quoteData['callout_fee'])) {
                    $calloutFee = (float) $quoteData['callout_fee'];
                } elseif (isset($quoteData['raw_data']['breakdown']['callout_fee'])) {
                    $calloutFee = (float) $quoteData['raw_data']['breakdown']['callout_fee'];
                }
                $totalCost += $calloutFee;
                if (isset($quoteData['assemble_disassemble_cost'])) {
                    $totalCost += (float) $quoteData['assemble_disassemble_cost'];
                }
                if (isset($quoteData['storage_items_cost'])) {
                    $totalCost += (float) $quoteData['storage_items_cost'];
                }
                // Always set total_cost to ensure it includes all costs
                $quoteData['total_cost'] = round($totalCost, 2);

                // Get existing discount information from quote if it exists
                $existingDiscount = null;
                $existingDiscountType = null;
                $applyOnRemaining = false;
                if ($quote) {
                    $existingRawData = $quote->raw_data ?? [];
                    if (isset($existingRawData['discount']) && $existingRawData['discount'] > 0) {
                        $existingDiscount = (float) $existingRawData['discount'];
                        $existingDiscountType = $existingRawData['discount_type'] ?? 'percentage';
                        $applyOnRemaining = isset($existingRawData['breakdown']['discount_on_remaining']) 
                            && $existingRawData['breakdown']['discount_on_remaining'];
                    }
                }

                // Apply discount if provided in request OR if existing discount exists
                if ($quoteData['total_cost'] > 0) {
                    $discountValue = null;
                    $discountType = 'percentage';
                    $shouldApplyDiscount = false;
                    
                    // Priority 1: Use discount from request if provided
                    if ($request->has('discount')) {
                        $discountValue = (float) $request->discount;
                        $discountType = $request->has('discount_type') ? $request->discount_type : 'percentage';
                        $shouldApplyDiscount = true;
                    }
                    // Priority 2: Use existing discount if no discount in request
                    elseif ($existingDiscount !== null && $existingDiscount > 0) {
                        $discountValue = $existingDiscount;
                        $discountType = $existingDiscountType;
                        $shouldApplyDiscount = true;
                    }
                    
                    if ($shouldApplyDiscount && $discountValue > 0) {
                        // Check if discount should be applied on remaining amount
                        $depositAmount = (float) ($quote->deposit_amount ?? 0);
                        
                        // Determine if discount should be on remaining amount
                        if ($request->has('discount')) {
                            // New discount from request - check if should apply on remaining based on current state
                            $willApplyOnRemaining = ($quote && $quote->deposit_payment_status === 'succeeded' 
                                && $quote->remaining_amount > 0);
                        } else {
                            // Existing discount - use the existing flag from raw_data
                            $willApplyOnRemaining = ($quote && $quote->deposit_payment_status === 'succeeded' 
                                && $quote->remaining_amount > 0 && $applyOnRemaining);
                        }
                        
                        if ($willApplyOnRemaining) {
                            // Apply discount on remaining amount (after deposit payment)
                            // Calculate original remaining: new total cost - deposit
                            $newOriginalTotalCost = $quoteData['total_cost'];
                            $newOriginalRemaining = $newOriginalTotalCost - $depositAmount;
                            
                            // Calculate discount amount on remaining
                            $discountAmount = 0;
                            if ($discountType === 'percentage') {
                                $discountAmount = ($newOriginalRemaining * $discountValue) / 100;
                            } else {
                                // Fixed amount - cannot exceed remaining amount
                                $discountAmount = min($discountValue, $newOriginalRemaining);
                            }
                            
                            // Calculate new remaining amount
                            $newRemainingAmount = round($newOriginalRemaining - $discountAmount, 2);
                            
                            // Calculate new total cost: deposit + discounted remaining
                            $quoteData['total_cost'] = round($depositAmount + $newRemainingAmount, 2);
                            
                            // Update raw_data
                            if (!isset($quoteData['raw_data']['breakdown'])) {
                                $quoteData['raw_data']['breakdown'] = [];
                            }
                            $quoteData['raw_data']['original_total_cost'] = $newOriginalTotalCost;
                            $quoteData['raw_data']['original_remaining_amount'] = $newOriginalRemaining;
                            $quoteData['raw_data']['discount'] = $discountValue;
                            $quoteData['raw_data']['discount_type'] = $discountType;
                            $quoteData['raw_data']['discount_on_remaining'] = $discountValue;
                            $quoteData['raw_data']['breakdown']['discount_amount'] = round($discountAmount, 2);
                            $quoteData['raw_data']['breakdown']['discount_on_remaining'] = true;
                            if ($discountType === 'percentage') {
                                $quoteData['raw_data']['breakdown']['discount_percent'] = $discountValue;
                            } else {
                                unset($quoteData['raw_data']['breakdown']['discount_percent']);
                            }
                        } else {
                            // Apply discount on total cost (original behavior)
                            $originalTotalCost = $quoteData['total_cost'];
                            
                            // Calculate discount amount
                            $discountAmount = 0;
                            if ($discountType === 'percentage') {
                                if ($discountValue <= 100) {
                                    $discountAmount = ($originalTotalCost * $discountValue) / 100;
                                }
                            } else {
                                // Fixed amount - cannot exceed total cost
                                $discountAmount = min($discountValue, $originalTotalCost);
                            }
                            
                            if ($discountAmount > 0) {
                                $quoteData['total_cost'] = round($originalTotalCost - $discountAmount, 2);
                                
                                // Store discount information in breakdown
                                if (!isset($quoteData['raw_data']['breakdown'])) {
                                    $quoteData['raw_data']['breakdown'] = [];
                                }
                                $quoteData['raw_data']['original_total_cost'] = $originalTotalCost;
                                $quoteData['raw_data']['discount'] = $discountValue;
                                $quoteData['raw_data']['discount_type'] = $discountType;
                                unset($quoteData['raw_data']['discount_on_remaining']);
                                $quoteData['raw_data']['breakdown']['discount_amount'] = round($discountAmount, 2);
                                unset($quoteData['raw_data']['breakdown']['discount_on_remaining']);
                                if ($discountType === 'percentage') {
                                    $quoteData['raw_data']['breakdown']['discount_percent'] = $discountValue;
                                } else {
                                    unset($quoteData['raw_data']['breakdown']['discount_percent']);
                                }
                            }
                        }
                    }
                }

                // Ensure user_id is always set
                if (!isset($quoteData['user_id']) || !$quoteData['user_id']) {
                    // Try to get user_id from booking
                    if ($booking && $booking->user_id) {
                        $quoteData['user_id'] = $booking->user_id;
                    } elseif ($quote && $quote->user_id) {
                        // Preserve existing user_id if quote exists
                        $quoteData['user_id'] = $quote->user_id;
                    } elseif ($quote && $quote->booking_id) {
                        // Try to get from booking relationship
                        $bookingForUser = Booking::find($quote->booking_id);
                        if ($bookingForUser && $bookingForUser->user_id) {
                            $quoteData['user_id'] = $bookingForUser->user_id;
                        }
                    }
                }

                if ($quote) {
                    // Update existing quote, but preserve quote_number, order_number, and status
                    if (!$quote->quote_number) {
                        $quoteData['quote_number'] = Quote::generateQuoteNumber();
                    }
                    // Preserve existing status
                    unset($quoteData['status']);
                    // Preserve distance_km
                    $quoteData['distance_km'] = $distanceKm;
                    
                    // Get the newly calculated calculation_type from quoteData
                    // This is determined by determineCalculationType() based on current booking items
                    $newCalculationType = $quoteData['calculation_type'] ?? $quote->calculation_type;
                    
                    // Check if items are being managed in the request
                    $isManagingItems = $request->has('items') || 
                                        $request->has('assemble_items') || 
                                        $request->has('storage_items');
                    
                    // Check if there are actually items in the request (non-empty arrays)
                    $hasItemsInRequest = ($request->has('items') && is_array($request->items) && count($request->items) > 0) ||
                                         ($request->has('assemble_items') && is_array($request->assemble_items) && count($request->assemble_items) > 0) ||
                                         ($request->has('storage_items') && is_array($request->storage_items) && count($request->storage_items) > 0);
                    
                    // Handle total_cubic_meters and calculation_type based on new calculation type
                    if ($newCalculationType === 'truck_only') {
                        // Still truck_only: preserve existing volume only if we're NOT managing items
                        if ($quote->calculation_type === 'truck_only' && $quote->total_cubic_meters && !$isManagingItems) {
                            // Was truck_only and not managing items, preserve the volume
                            $quoteData['total_cubic_meters'] = $quote->total_cubic_meters;
                        } else {
                            // Transitioned from items to truck_only or items were removed
                            // Use the calculated value from quoteData (which will be 0 if items were removed)
                            if (isset($quoteData['total_cubic_meters'])) {
                                $quoteData['total_cubic_meters'] = round((float) $quoteData['total_cubic_meters'], 2);
                            }
                        }
                        // Preserve volume_cost if it exists
                        if ($quote->volume_cost) {
                            $quoteData['volume_cost'] = $quote->volume_cost;
                        }
                    } else {
                        // Items-based calculation (items_only or items_and_packages)
                        // Calculate total_cubic_meters from items: volume * quantity for each regular item
                        // The calculateQuote method calculates total_cubic_meters by summing (quantity * volume) for all regular items
                        if (isset($quoteData['total_cubic_meters'])) {
                            $quoteData['total_cubic_meters'] = round((float) $quoteData['total_cubic_meters'], 2);
                        }
                        // Clear volume_cost for items-based calculations
                        $quoteData['volume_cost'] = null;
                    }
                    
                    // Always update calculation_type to match the current booking items
                    // This handles transitions: truck_only -> items_only/items_and_packages when items are added
                    $quoteData['calculation_type'] = $newCalculationType;
                    
                    // Ensure user_id is preserved if not in quoteData
                    if (!isset($quoteData['user_id']) && $quote->user_id) {
                        $quoteData['user_id'] = $quote->user_id;
                    }

                    // Ensure raw_data is properly set from quoteData (includes all breakdown information)
                    // This ensures prices are properly updated when items are removed
                    if (isset($quoteData['raw_data']) && is_array($quoteData['raw_data'])) {
                        // Get existing raw_data to preserve discount and other non-calculation fields
                        $existingRawData = $quote->raw_data ?? [];
                        
                        // Only preserve discount information if it wasn't just recalculated above
                        // Check if discount was already set in quoteData (meaning it was just recalculated)
                        $discountWasRecalculated = isset($quoteData['raw_data']['discount']) && 
                                                   isset($quoteData['raw_data']['discount_type']);
                        
                        if (!$discountWasRecalculated) {
                            // Preserve discount information if it exists (these are not recalculated)
                            if (isset($existingRawData['discount'])) {
                                $quoteData['raw_data']['discount'] = $existingRawData['discount'];
                            }
                            if (isset($existingRawData['discount_type'])) {
                                $quoteData['raw_data']['discount_type'] = $existingRawData['discount_type'];
                            }
                            if (isset($existingRawData['original_total_cost'])) {
                                $quoteData['raw_data']['original_total_cost'] = $existingRawData['original_total_cost'];
                            }
                            if (isset($existingRawData['discount_on_remaining'])) {
                                $quoteData['raw_data']['discount_on_remaining'] = $existingRawData['discount_on_remaining'];
                            }
                            if (isset($existingRawData['original_remaining_amount'])) {
                                $quoteData['raw_data']['original_remaining_amount'] = $existingRawData['original_remaining_amount'];
                            }
                            
                            // Preserve discount information in breakdown if it exists
                            if (isset($existingRawData['breakdown']['discount_amount'])) {
                                $quoteData['raw_data']['breakdown']['discount_amount'] = $existingRawData['breakdown']['discount_amount'];
                            }
                            if (isset($existingRawData['breakdown']['discount_percent'])) {
                                $quoteData['raw_data']['breakdown']['discount_percent'] = $existingRawData['breakdown']['discount_percent'];
                            }
                            if (isset($existingRawData['breakdown']['discount_on_remaining'])) {
                                $quoteData['raw_data']['breakdown']['discount_on_remaining'] = $existingRawData['breakdown']['discount_on_remaining'];
                            }
                        }
                        // If discount was recalculated, the values are already set in quoteData['raw_data'] above
                        
                        // Ensure storage_items_cost and assemble_disassemble_cost are always in breakdown
                        // Always set them from quoteData to ensure they match the calculated values
                        // This ensures they're present even if calculateQuote didn't include them (shouldn't happen, but safety check)
                        $quoteData['raw_data']['breakdown']['storage_items_cost'] = round($quoteData['storage_items_cost'] ?? 0, 2);
                        $quoteData['raw_data']['breakdown']['assemble_disassemble_cost'] = round($quoteData['assemble_disassemble_cost'] ?? 0, 2);
                        
                        // Preserve other non-calculation fields (like name, email, phone) that might exist
                        $fieldsToPreserve = ['name', 'email', 'phone'];
                        foreach ($fieldsToPreserve as $field) {
                            if (isset($existingRawData[$field])) {
                                $quoteData['raw_data'][$field] = $existingRawData[$field];
                            }
                        }
                        
                        // Note: breakdown, times, and costs are completely replaced by calculateQuote
                        // This ensures accurate pricing when items are added/removed
                    }

                    // Update quote with recalculated costs
                    $quote->update($quoteData);
                    $quote->refresh();
                    
                    // Final check: if user_id is still null, try to set it from booking
                    if (!$quote->user_id && $booking && $booking->user_id) {
                        $quote->user_id = $booking->user_id;
                    }

                    // Update remaining_amount - use provided value or calculate from discounted total_cost
                    $depositAmount = (float) ($quote->deposit_amount ?? 0);
                    if ($request->has('remaining_amount') && $request->remaining_amount !== null && $request->remaining_amount !== '') {
                        // Use provided remaining amount
                        $quote->remaining_amount = (float) $request->remaining_amount;
                    } else {
                        // Calculate remaining amount: discounted total_cost - deposit_amount
                        // If discount was applied on remaining, total_cost already includes deposit + discounted remaining
                        // So remaining_amount = total_cost - deposit_amount will give the correct discounted remaining
                        $discountedTotalCost = (float) $quoteData['total_cost'];
                        $quote->remaining_amount = max(0, round($discountedTotalCost - $depositAmount, 2));
                    }
                    $quote->save();
                } else {
                    // Create new quote with unique quote_number
                    $quoteData['quote_number'] = Quote::generateQuoteNumber();
                    // Ensure user_id is set before creating
                    if (!isset($quoteData['user_id']) || !$quoteData['user_id']) {
                        if ($booking && $booking->user_id) {
                            $quoteData['user_id'] = $booking->user_id;
                        }
                    }
                    $quote = Quote::create($quoteData);
                }

                // Add quote to result
                $result['quote'] = $quote->load(['user']);
            }
        }
        return $this->success($result, 'Quote updated successfully');
    }


    /**
     * Remove the specified quote
     */
    public function destroy($id)
    {
        $quote = Quote::findOrFail($id);

        // Delete payment proof if exists
        if ($quote->quote_payment_proof && file_exists(public_path($quote->quote_payment_proof))) {
            unlink(public_path($quote->quote_payment_proof));
        }

        $quote->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Quote deleted successfully.'
        ]);
    }

    /**
     * Send remaining payment link email to customer
     */
    public function sendRemainingPaymentEmail($id)
    {
        try {
            $quote = Quote::with('user')->findOrFail($id);

            // Check if quote has remaining amount
            if (!$quote->remaining_amount || $quote->remaining_amount <= 0) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No remaining amount to pay for this order.'
                ], 422);
            }

            // Check if remaining payment is already completed
            if ($quote->remaining_payment_status === 'succeeded') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Remaining payment has already been completed.'
                ], 422);
            }

            // Create remaining payment link if it doesn't exist
            if (!$quote->remaining_payment_url) {
                try {
                    $paymentLinkResult = $this->createRemainingPaymentLink($quote);
                    $quote->update(['remaining_payment_url' => $paymentLinkResult['payment_link_url']]);
                } catch (\Exception $e) {
                    Log::error('Error creating remaining payment link: ' . $e->getMessage());
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Failed to create payment link: ' . $e->getMessage()
                    ], 500);
                }
            }

            // Send email to customer
            $settings = $this->getGeneralSettingsData();
            $user = $quote->user;

            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Customer not found for this order.'
                ], 422);
            }

            $emailParams = [
                'subject' => 'Complete Your Payment | ' . $settings['business_name'],
                'to' => $user->email,
                'msg' => view('Admin.Emails.remaining-payment-reminder', [
                    'quote' => $quote,
                    'user' => $user,
                    'settings' => $settings,
                ])->render(),
            ];

            $emailSent = $this->SendInstantEmail($emailParams);

            if ($emailSent) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'Remaining payment link email sent successfully to customer.'
                ]);
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to send email. Please try again.'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error sending remaining payment email: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Release/waive remaining payment
     */
    public function releaseRemainingPayment($id)
    {
        try {
            $quote = Quote::findOrFail($id);

            // Check if quote has remaining amount
            if (!$quote->remaining_amount || $quote->remaining_amount <= 0) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No remaining amount to release for this order.'
                ], 422);
            }

            // Check if remaining payment is already completed
            if ($quote->remaining_payment_status === 'succeeded') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Remaining payment has already been completed.'
                ], 422);
            }

            // Update quote to mark remaining payment as released/waived
            $quote->update([
                'remaining_payment_status' => 'succeeded',
                'remaining_amount' => 0,
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Remaining payment has been released successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error releasing remaining payment: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get or create remaining payment link
     */
    public function getRemainingPaymentLink($id)
    {
        try {
            $quote = Quote::findOrFail($id);

            // Check if quote has remaining amount
            if (!$quote->remaining_amount || $quote->remaining_amount <= 0) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No remaining amount to pay for this order.'
                ], 422);
            }

            // Check if remaining payment is already completed
            if ($quote->remaining_payment_status === 'succeeded') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Remaining payment has already been completed.'
                ], 422);
            }

            // Create remaining payment link if it doesn't exist
            if (!$quote->remaining_payment_url) {
                try {
                    $paymentLinkResult = $this->createRemainingPaymentLink($quote);
                    $quote->update(['remaining_payment_url' => $paymentLinkResult['payment_link_url']]);
                    $quote->refresh();
                } catch (\Exception $e) {
                    Log::error('Error creating remaining payment link: ' . $e->getMessage());
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Failed to create payment link: ' . $e->getMessage()
                    ], 500);
                }
            }

            return response()->json([
                'status' => 'success',
                'payment_link' => $quote->remaining_payment_url,
                'message' => 'Payment link retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting remaining payment link: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Apply discount to quote
     */
    public function applyDiscount(Request $request, $id)
    {
        try {
            $quote = Quote::findOrFail($id);
            // Ensure user_id is set if missing
            if (!$quote->user_id) {
                if ($quote->booking_id) {
                    $booking = Booking::find($quote->booking_id);
                    if ($booking && $booking->user_id) {
                        $quote->user_id = $booking->user_id;
                        $quote->save();
                    }
                } elseif ($quote->raw_data && isset($quote->raw_data['email'])) {
                    $user = \App\Models\User::where('email', $quote->raw_data['email'])->first();
                    if ($user) {
                        $quote->user_id = $user->id;
                        $quote->save();
                    }
                }
            }
            $request->validate([
                'discount_type' => 'required|in:percentage,fixed',
                'discount' => 'required|numeric|min:0'
            ]);
            $discountType = $request->discount_type;
            $discountValue = (float) $request->discount;

            // Check payment method
            $paymentMethod = $quote->payment_method ?? ($quote->raw_data['payment_method'] ?? 'stripe');
            
            // For bank transfer: check if deposit is approved before allowing discount
            if ($paymentMethod === 'bank_transfer') {
                // If deposit payment status is pending, don't allow discount application
                if ($quote->deposit_payment_status === 'pending' || !$quote->deposit_payment_status) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Please first approve the deposit payment, then apply discount.'
                    ], 422);
                }
            }
            
            // Check if payment is fully paid (for both Stripe and Bank Transfer)
            $isFullyPaid = false;
            if ($paymentMethod === 'bank_transfer') {
                // For bank transfer: fully paid if remaining_payment_status is succeeded OR remaining_amount is 0
                $isFullyPaid = ($quote->remaining_payment_status === 'succeeded') || 
                               (!$quote->remaining_amount || $quote->remaining_amount <= 0);
            } else {
                // For Stripe: fully paid if remaining_payment_status is succeeded
                $isFullyPaid = ($quote->remaining_payment_status === 'succeeded');
            }
            
            // Prevent discount application after full payment
            if ($isFullyPaid) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Discount cannot be applied after full payment has been completed.'
                ], 422);
            }
            
            // Check if discount should be applied on remaining amount
            // Condition: deposit is paid AND there's a remaining amount
            // This ensures discount is applied on remaining amount after deposit payment
            $applyOnRemaining = ($quote->deposit_payment_status === 'succeeded' 
                && $quote->remaining_amount > 0);
            
            // Validation
            if ($discountType === 'percentage' && $discountValue > 100) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Discount percentage cannot exceed 100%.'
                ], 422);
            }
            // If applying on remaining amount, validate it doesn't exceed remaining
            if ($applyOnRemaining) {
                if ($discountType === 'fixed' && $discountValue > $quote->remaining_amount) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Discount amount cannot exceed the remaining amount of $' . number_format($quote->remaining_amount, 2) . '.'
                    ], 422);
                }
            }
            // Get raw_data
            $rawData = $quote->raw_data ?? [];
            if ($applyOnRemaining) {
                // Apply discount on remaining amount (not total cost)
                // First, get or restore original total cost
                if (!isset($rawData['original_total_cost'])) {
                    // If there was a previous discount on total, reverse calculate
                    if (isset($rawData['discount']) && isset($rawData['discount_type']) && $rawData['discount'] > 0 
                        && (!isset($rawData['discount_on_remaining']) || !$rawData['discount_on_remaining'])) {
                        $existingDiscount = (float) $rawData['discount'];
                        if ($rawData['discount_type'] === 'percentage') {
                            $originalTotalCost = $quote->total_cost / (1 - ($existingDiscount / 100));
                        } else {
                            $originalTotalCost = $quote->total_cost + $existingDiscount;
                        }
                    } else {
                        // No previous discount on total, current total_cost is original
                        $originalTotalCost = $quote->total_cost;
                    }
                    $rawData['original_total_cost'] = $originalTotalCost;
                } else {
                    $originalTotalCost = $rawData['original_total_cost'];
                }
                // Restore total_cost to original (in case it was previously discounted)
                $quote->total_cost = $originalTotalCost;
                // Calculate original remaining amount: original_total_cost - deposit_amount
                $depositAmount = (float) ($quote->deposit_amount ?? 0);
                $originalRemaining = $originalTotalCost - $depositAmount;
                // If there was a previous discount on remaining, we need to get the original remaining before that discount
                if (isset($rawData['discount_on_remaining']) && isset($rawData['discount_type']) && $rawData['discount_on_remaining'] > 0) {
                    $existingDiscount = (float) $rawData['discount_on_remaining'];
                    if ($rawData['discount_type'] === 'percentage') {
                        // Reverse: current = original * (1 - discount/100)
                        $originalRemaining = $quote->remaining_amount / (1 - ($existingDiscount / 100));
                    } else {
                        // Reverse: current = original - discount
                        $originalRemaining = $quote->remaining_amount + $existingDiscount;
                    }
                }
                // Store original remaining amount
                $rawData['original_remaining_amount'] = $originalRemaining;
                // Calculate discount amount on remaining
                $discountAmount = 0;
                if ($discountType === 'percentage') {
                    $discountAmount = ($originalRemaining * $discountValue) / 100;
                } else {
                    // Fixed amount - cannot exceed original remaining amount
                    $discountAmount = min($discountValue, $originalRemaining);
                }
                // Store discount info
                $rawData['discount'] = $discountValue;
                $rawData['discount_type'] = $discountType;
                $rawData['discount_on_remaining'] = $discountValue; // Flag that discount is on remaining
                // Update breakdown
                if (!isset($rawData['breakdown'])) {
                    $rawData['breakdown'] = [];
                }
                $rawData['breakdown']['discount_amount'] = round($discountAmount, 2);
                $rawData['breakdown']['discount_on_remaining'] = true; // Flag for display
                if ($discountType === 'percentage') {
                    $rawData['breakdown']['discount_percent'] = $discountValue;
                } else {
                    unset($rawData['breakdown']['discount_percent']);
                }
                // Calculate new remaining amount
                $newRemainingAmount = round($originalRemaining - $discountAmount, 2);
                
                // Calculate new total_cost: deposit_amount + new remaining_amount
                // This ensures total_cost reflects the discounted remaining amount
                // Note: $depositAmount is already defined above (line 1089)
                $newTotalCost = round($depositAmount + $newRemainingAmount, 2);
                
                // Update quote - update both total_cost and remaining_amount
                $quote->raw_data = $rawData;
                $quote->total_cost = $newTotalCost; // Update total_cost to reflect discount on remaining
                $quote->remaining_amount = max(0, $newRemainingAmount);
            } else {
                // Apply discount on total cost (original behavior)
                // Get original total cost (before any previous discount)
                if (!isset($rawData['original_total_cost'])) {
                    // If there's an existing discount, try to reverse calculate
                    if (isset($rawData['discount']) && isset($rawData['discount_type']) && $rawData['discount'] > 0) {
                        $existingDiscount = (float) $rawData['discount'];
                        if ($rawData['discount_type'] === 'percentage') {
                            // Reverse: current = original * (1 - discount/100)
                            $originalTotalCost = $quote->total_cost / (1 - ($existingDiscount / 100));
                        } else {
                            // Reverse: current = original - discount
                            $originalTotalCost = $quote->total_cost + $existingDiscount;
                        }
                    } else {
                        // No existing discount, current total_cost is the original
                        $originalTotalCost = $quote->total_cost;
                    }
                    $rawData['original_total_cost'] = $originalTotalCost;
                } else {
                    $originalTotalCost = $rawData['original_total_cost'];
                }
                // Calculate discount amount
                $discountAmount = 0;
                if ($discountType === 'percentage') {
                    $discountAmount = ($originalTotalCost * $discountValue) / 100;
                } else {
                    // Fixed amount - cannot exceed original total cost
                    $discountAmount = min($discountValue, $originalTotalCost);
                }
                // Update quote
                $rawData['discount'] = $discountValue;
                $rawData['discount_type'] = $discountType;
                unset($rawData['discount_on_remaining']); // Not on remaining
                // Update breakdown
                if (!isset($rawData['breakdown'])) {
                    $rawData['breakdown'] = [];
                }
                $rawData['breakdown']['discount_amount'] = round($discountAmount, 2);
                unset($rawData['breakdown']['discount_on_remaining']); // Not on remaining
                if ($discountType === 'percentage') {
                    $rawData['breakdown']['discount_percent'] = $discountValue;
                } else {
                    unset($rawData['breakdown']['discount_percent']);
                }
                // Calculate new total cost
                $newTotalCost = round($originalTotalCost - $discountAmount, 2);
                // Update quote
                $quote->raw_data = $rawData;
                $quote->total_cost = $newTotalCost;
                // Recalculate remaining amount
                $depositAmount = (float) ($quote->deposit_amount ?? 0);
                $quote->remaining_amount = max(0, round($newTotalCost - $depositAmount, 2));
            }
            // Check minimum order amount validation
            $jobType = null;
            if ($quote->raw_data && isset($quote->raw_data['job_type'])) {
                $jobType = $quote->raw_data['job_type'];
            }
            // Get minimum order amount from general settings based on job type
            $minimumOrderAmount = 0;
            $minimumOrderSetting = null;
            if ($jobType === 'local_job') {
                $minimumOrderSetting = '_local_job_minimum_order';
            } elseif ($jobType === 'interstate_job') {
                $minimumOrderSetting = '_interstate_job_minimum_order';
            }
            if ($minimumOrderSetting) {
                $settings = GeneralSettings::get();
                $minimumOrderValue = GeneralSettingsClass::getOptionValue($minimumOrderSetting, $settings);
                if ($minimumOrderValue !== '' && $minimumOrderValue > 0) {
                    $minimumOrderAmount = (float) $minimumOrderValue;
                }
            }
            // Validate remaining amount or total cost against minimum order
            // If there's a remaining amount, check that. Otherwise, check total cost.
            $amountToCheck = $quote->remaining_amount > 0 ? $quote->remaining_amount : $quote->total_cost;
            $amountLabel = $quote->remaining_amount > 0 ? 'remaining amount' : 'total cost';
            if ($minimumOrderAmount > 0 && $amountToCheck > 0 && $amountToCheck < $minimumOrderAmount) {
                $jobTypeLabel = $jobType === 'local_job' ? 'local' : 'interstate';
                return response()->json([
                    'status' => 'error',
                    'message' => "The {$amountLabel} after discount ($" . number_format($amountToCheck, 2) . ") 
                    cannot be less than the minimum order amount ($" . number_format($minimumOrderAmount, 2) . ") for 
                    {$jobTypeLabel} jobs. Please adjust the discount amount or Set the minimum order amount in the  settings."
                ], 422);
            }
            // Store old values before save to check if they changed
            $oldRemainingAmount = $quote->getOriginal('remaining_amount') ?? $quote->remaining_amount;
            $hadPaymentLink = !empty($quote->remaining_payment_url);
            $quote->save();
            // If there's an existing payment link and remaining amount is still > 0,
            // create a new payment link with the updated remaining amount
            // This handles both cases: discount on remaining OR discount on total (which affects remaining)
            if ($hadPaymentLink && $quote->remaining_amount > 0) {
                try {
                    // Refresh quote to get latest data
                    $quote->refresh();
                    // Create new payment link with updated remaining amount
                    $paymentLinkResult = $this->createRemainingPaymentLink($quote);
                    // Update quote with new payment link URL
                    $quote->update([
                        'remaining_payment_url' => $paymentLinkResult['payment_link_url']
                    ]);
                    Log::info('Payment link updated after discount applied', [
                        'quote_id' => $quote->id,
                        'discount_type' => $applyOnRemaining ? 'on_remaining' : 'on_total',
                        'old_remaining_amount' => $oldRemainingAmount,
                        'new_remaining_amount' => $paymentLinkResult['remaining_amount'],
                        'new_payment_link' => $paymentLinkResult['payment_link_url']
                    ]);
                } catch (\Exception $e) {
                    // Log error but don't fail the discount application
                    Log::error('Error updating payment link after discount: ' . $e->getMessage(), [
                        'quote_id' => $quote->id,
                        'error' => $e->getMessage()
                    ]);
                    // Continue - discount was applied successfully, just payment link update failed
                }
            } elseif ($hadPaymentLink && $quote->remaining_amount <= 0) {
                // If remaining amount is now 0 or negative, clear the payment link
                $quote->update(['remaining_payment_url' => null]);
                Log::info('Payment link cleared - remaining amount is now 0', [
                    'quote_id' => $quote->id,
                    'old_remaining_amount' => $oldRemainingAmount
                ]);
            }
            // Check if payment link was updated
            $quote->refresh();
            $paymentLinkUpdated = $quote->remaining_payment_url && $quote->remaining_amount > 0;
            // Send discount notification email to customer
            try {
                $user = $quote->user;
                if (!$user && $quote->booking_id) {
                    $booking = Booking::with('user')->find($quote->booking_id);
                    if ($booking && $booking->user) {
                        $user = $booking->user;
                    }
                }
                if ($user) {
                    $settings = $this->getGeneralSettingsData();   
                    // Get original amounts for email
                    $originalTotalCost = $rawData['original_total_cost'] ?? $quote->total_cost;
                    $originalRemainingAmount = null;
                    if ($applyOnRemaining) {
                        $originalRemainingAmount = $rawData['original_remaining_amount'] ?? ($quote->total_cost - ($quote->deposit_amount ?? 0));
                    }
                    $emailParams = [
                        'subject' => 'Discount Applied to Your Order | ' . $settings['business_name'],
                        'to' => $user->email,
                        'msg' => view('Admin.Emails.discount-applied', [
                            'quote' => $quote,
                            'user' => $user,
                            'settings' => $settings,
                            'discountType' => $discountType,
                            'discountValue' => $discountValue,
                            'discountAmount' => round($discountAmount, 2),
                            'discountOnRemaining' => $applyOnRemaining,
                            'originalTotalCost' => $originalTotalCost,
                            'originalRemainingAmount' => $originalRemainingAmount,
                        ])->render(),
                    ];
                    $this->SendInstantEmail($emailParams);
                    Log::info('Discount notification email sent', [
                        'quote_id' => $quote->id,
                        'user_id' => $user->id,
                        'email' => $user->email
                    ]);
                }
            } catch (\Exception $e) {
                // Log error but don't fail the discount application
                Log::error('Error sending discount notification email: ' . $e->getMessage(), [
                    'quote_id' => $quote->id,
                    'error' => $e->getMessage()
                ]);
            }
            
            $message = 'Discount applied successfully.';
            if ($paymentLinkUpdated) {
                $message .= ' Payment link has been updated with the new remaining amount.';
            } elseif ($quote->remaining_amount <= 0 && $oldRemainingAmount > 0) {
                $message .= ' Payment link has been cleared as remaining amount is now zero.';
            }            
            return response()->json([
                'status' => 'success',
                'message' => $message,
                'data' => [
                    'total_cost' => $quote->total_cost,
                    'remaining_amount' => $quote->remaining_amount,
                    'discount_amount' => round($discountAmount, 2),
                    'payment_link_updated' => $paymentLinkUpdated
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error applying discount: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Approve or reject bank transfer payment (deposit or remaining)
     */
    public function approveBankTransferPayment(Request $request, $id)
    {
        try {
            $quote = Quote::findOrFail($id);
            $paymentType = $request->input('payment_type'); // 'deposit' or 'remaining'
            $action = $request->input('action'); // 'approve' or 'reject'
            $paymentMethod = $quote->payment_method ?? ($quote->raw_data['payment_method'] ?? null);
            if ($paymentMethod !== 'bank_transfer') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'This quote does not use bank transfer payment method.'
                ], 422);
            }
            if (!in_array($paymentType, ['deposit', 'remaining'])) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid payment type. Must be "deposit" or "remaining".'
                ], 422);
            }

            // Validate action
            if (!in_array($action, ['approve', 'reject'])) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid action. Must be "approve" or "reject".'
                ], 422);
            }

            $updateData = [];

            if ($paymentType === 'deposit') {
                if (!$quote->deposit_amount) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'No deposit payment found for this quote.'
                    ], 422);
                }

                if ($action === 'approve') {
                    $updateData['deposit_payment_status'] = 'succeeded';
                    // Do NOT change total_cost - it should remain the same (same as Stripe handling)
                    // total_cost is the total order amount, deposit_amount and remaining_amount are just payment breakdowns
                } else {
                    $updateData['deposit_payment_status'] = 'failed';
                    $updateData['status'] = 'rejected';
                }
            } else {
                // remaining
                if (!$quote->remaining_amount || $quote->remaining_amount <= 0) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'No remaining payment found for this quote.'
                    ], 422);
                }

                if ($action === 'approve') {
                    // Handle remaining payment the same way as Stripe webhook
                    // Add remaining amount to deposit amount and set remaining amount to zero
                    $remainingAmount = $quote->remaining_amount ?? 0;
                    $newDepositAmount = ($quote->deposit_amount ?? 0) + $remainingAmount;
                    
                    $updateData['remaining_payment_status'] = 'succeeded';
                    $updateData['deposit_amount'] = $newDepositAmount;
                    $updateData['remaining_amount'] = 0;
                    $updateData['remaining_payment_url'] = null; // Clear payment URL
                    
                    Log::info('Bank transfer remaining payment approved - Amount transferred to deposit', [
                        'quote_id' => $quote->id,
                        'remaining_amount_transferred' => $remainingAmount,
                        'new_deposit_amount' => $newDepositAmount
                    ]);
                } else {
                    $updateData['remaining_payment_status'] = 'failed';
                    $updateData['status'] = 'rejected';
                }
            }


            $quote->update($updateData);
            $quote->refresh();

            // Send confirmation emails if payment is approved (same as Stripe)
            if ($action === 'approve') {
                    if ($paymentType === 'deposit') {
                        // Send deposit confirmation email
                        $emailResults = $this->sendOrderConfirmationEmail($quote, 'deposit', 'succeeded', false);
                        
                        if (!$emailResults['customer']) {
                            Log::warning('Failed to send order confirmation email to customer for bank transfer deposit - quote ID: ' . $quote->id);
                        }
                        if (!$emailResults['admin']) {
                            Log::warning('Failed to send order confirmation email to admin for bank transfer deposit - quote ID: ' . $quote->id);
                        }
                    } else {
                        // Send remaining payment confirmation email
                        $emailResults = $this->sendOrderConfirmationEmail($quote, 'full', 'succeeded', true);
                        
                        if (!$emailResults['customer']) {
                            Log::warning('Failed to send order confirmation email to customer for bank transfer remaining payment - quote ID: ' . $quote->id);
                        }
                        if (!$emailResults['admin']) {
                            Log::warning('Failed to send order confirmation email to admin for bank transfer remaining payment - quote ID: ' . $quote->id);
                        }
                    }
               
            }

            $actionText = $action === 'approve' ? 'approved' : 'rejected';
            $paymentTypeText = $paymentType === 'deposit' ? 'deposit' : 'remaining';

            return response()->json([
                'status' => 'success',
                'message' => ucfirst($paymentTypeText) . ' payment has been ' . $actionText . ' successfully.',
                'data' => [
                    'deposit_payment_status' => $quote->deposit_payment_status,
                    'remaining_payment_status' => $quote->remaining_payment_status,
                    'remaining_amount' => $quote->remaining_amount,
                    'total_cost' => $quote->total_cost,
                    'deposit_amount' => $quote->deposit_amount,
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error approving/rejecting bank transfer payment: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Adjust remaining amount and upload proof for bank transfer (combined)
     */
    public function adjustRemainingAmount(Request $request, $id)
    {
        try {
            $quote = Quote::findOrFail($id);
            // Validate payment method
            $paymentMethod = $quote->payment_method ?? ($quote->raw_data['payment_method'] ?? null);
            if ($paymentMethod !== 'bank_transfer') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'This quote does not use bank transfer payment method.'
                ], 422);
            }
            $newRemainingAmount = $request->input('remaining_amount');
            $shouldMarkAsPaid = $request->input('mark_as_paid', false);
            if ($newRemainingAmount !== null && (!is_numeric($newRemainingAmount) || $newRemainingAmount < 0)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid remaining amount. Must be a positive number.'
                ], 422);
            }
            
            // Validate that new remaining amount cannot be greater than current remaining amount
            if ($newRemainingAmount !== null) {
                $currentRemainingAmount = $quote->remaining_amount ?? 0;
                if ($newRemainingAmount > $currentRemainingAmount) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'New remaining amount cannot be greater than the current remaining amount ($' . number_format($currentRemainingAmount, 2) . ').'
                    ], 422);
                }
            }

            $updateData = [];
            $rawData = $quote->raw_data ?? [];
            if ($newRemainingAmount !== null) {
                $newRemainingAmount = round((float) $newRemainingAmount, 2);
                
                // Ensure remaining amount is not negative (can be 0)
                if ($newRemainingAmount < 0) {
                    $newRemainingAmount = 0;
                }

                $depositAmount = $quote->deposit_amount ?? 0;
                
                // Update remaining amount
                $updateData['remaining_amount'] = $newRemainingAmount;
                
                // Total cost should remain unchanged (it's the original order amount)
                // Only adjust remaining_amount, deposit_amount will be updated when payment is completed
                $updateData['total_cost'] = $quote->total_cost;
                if (!isset($rawData['original_total_cost_before_adjustment'])) {
                    $rawData['original_total_cost_before_adjustment'] = $quote->total_cost;
                }
                if (!isset($rawData['original_remaining_amount_before_adjustment'])) {
                    $rawData['original_remaining_amount_before_adjustment'] = $quote->remaining_amount;
                }
                if (!isset($rawData['original_total_cost'])) {
                    // If original_total_cost doesn't exist, set it to current total_cost before adjustment
                    $rawData['original_total_cost'] = $quote->total_cost;
                }
                if (!isset($rawData['breakdown'])) {
                    $rawData['breakdown'] = [];
                }
                $rawData['remaining_amount_adjusted_at'] = now()->toDateTimeString();

                // Handle payment completion (same as Stripe webhook)
                if ($newRemainingAmount == 0 || $shouldMarkAsPaid) {
                    // Payment is complete - handle like Stripe: add remaining to deposit
                    $currentRemainingAmount = $quote->remaining_amount ?? 0;
                    if ($currentRemainingAmount > 0) {
                        // Add remaining amount to deposit amount (same as Stripe)
                        $newDepositAmount = $depositAmount + $currentRemainingAmount;
                        $updateData['deposit_amount'] = $newDepositAmount;
                        $updateData['remaining_amount'] = 0;
                        $updateData['remaining_payment_url'] = null; // Clear payment URL
                        
                        // Total cost remains the same (don't change it - it's the total order amount)
                        $updateData['total_cost'] = $quote->total_cost;
                        
                        Log::info('Bank transfer remaining payment completed via adjust - Amount transferred to deposit', [
                            'quote_id' => $quote->id,
                            'remaining_amount_transferred' => $currentRemainingAmount,
                            'new_deposit_amount' => $newDepositAmount,
                            'original_total_cost' => $quote->total_cost
                        ]);
                    } else {
                        // Already 0, just mark as succeeded
                        $updateData['remaining_amount'] = 0;
                        // Keep total_cost unchanged
                        $updateData['total_cost'] = $quote->total_cost;
                    }
                    $updateData['remaining_payment_status'] = 'succeeded';
                } else {
                    // Amount is adjusted but not completed
                    // Always reset to pending if there's a remaining amount to pay
                    // This ensures that adjusting the amount requires re-approval
                    $updateData['remaining_payment_status'] = 'pending';
                }
            }

            // Handle payment proof upload (required)
            if (!$request->hasFile('remaining_payment_proof')) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Payment proof file is required.'
                ], 422);
            }

            $file = $request->file('remaining_payment_proof');
            if (!$file->isValid() || !in_array(strtolower($file->getClientOriginalExtension()), ['jpg', 'jpeg', 'webp', 'png', 'pdf'])) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid file type. Please upload an image (jpg, jpeg, webp, png) or PDF.'
                ], 422);
            }

            // Create directory if it doesn't exist
            $uploadPath = public_path('quotes/payment_proofs');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }

            // Generate unique filename
            $filename = 'remaining_payment_proof_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();

            // Move uploaded file
            $file->move($uploadPath, $filename);

            // Relative path for database storage
            $relativePath = 'quotes/payment_proofs/' . $filename;
            $rawData['remaining_payment_proof'] = $relativePath;

            // Preserve original_remaining_amount if not set (for discount calculations)
            if (!isset($rawData['original_remaining_amount'])) {
                // If original_remaining_amount doesn't exist, calculate it from original_total_cost
                if (isset($rawData['original_total_cost'])) {
                    $depositAmountForCalc = $quote->deposit_amount ?? 0;
                    $rawData['original_remaining_amount'] = $rawData['original_total_cost'] - $depositAmountForCalc;
                } else {
                    // Fallback: use current remaining_amount before adjustment
                    $rawData['original_remaining_amount'] = $quote->remaining_amount ?? 0;
                }
            }
            
            // Update raw_data (preserve all breakdown components)
            $updateData['raw_data'] = $rawData;

            // Update quote
            $quote->update($updateData);
            $quote->refresh();

            $messages = [];
            if ($newRemainingAmount !== null) {
                $messages[] = 'Remaining amount adjusted successfully.';
            }
            if ($request->hasFile('remaining_payment_proof')) {
                $messages[] = 'Payment proof uploaded successfully.';
            }

            // Refresh quote to get latest data including raw_data
            $quote->refresh();
            
            // Get breakdown from raw_data for response
            $breakdown = $quote->raw_data['breakdown'] ?? [];
            
            return response()->json([
                'status' => 'success',
                'message' => implode(' ', $messages) ?: 'Updated successfully.',
                'data' => [
                    'remaining_amount' => $quote->remaining_amount,
                    'total_cost' => $quote->total_cost,
                    'deposit_amount' => $quote->deposit_amount,
                    'remaining_payment_status' => $quote->remaining_payment_status,
                    'breakdown' => $breakdown,
                    'movers_cost' => $quote->movers_cost ?? ($breakdown['movers_cost'] ?? 0),
                    'callout_fee' => $quote->callout_fee ?? ($breakdown['callout_fee'] ?? 0),
                    'storage_items_cost' => $quote->storage_items_cost ?? ($breakdown['storage_items_cost'] ?? 0),
                    'assemble_disassemble_cost' => $quote->assemble_disassemble_cost ?? ($breakdown['assemble_disassemble_cost'] ?? 0),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error adjusting remaining amount: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Upload remaining payment proof for bank transfer
     */
    public function uploadRemainingProof(Request $request, $id)
    {
        try {
            $quote = Quote::findOrFail($id);
            
            // Validate payment method
            $paymentMethod = $quote->payment_method ?? ($quote->raw_data['payment_method'] ?? null);
            if ($paymentMethod !== 'bank_transfer') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'This quote does not use bank transfer payment method.'
                ], 422);
            }

            if (!$quote->remaining_amount || $quote->remaining_amount <= 0) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No remaining amount found for this quote.'
                ], 422);
            }

            // Validate file upload
            if (!$request->hasFile('remaining_payment_proof')) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Payment proof file is required.'
                ], 422);
            }

            $file = $request->file('remaining_payment_proof');
            if (!$file->isValid() || !in_array(strtolower($file->getClientOriginalExtension()), ['jpg', 'jpeg', 'webp', 'png', 'pdf'])) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid file type. Please upload an image (jpg, jpeg, webp, png) or PDF.'
                ], 422);
            }

            // Create directory if it doesn't exist
            $uploadPath = public_path('quotes/payment_proofs');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }

            // Generate unique filename
            $filename = 'remaining_payment_proof_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $filePath = $uploadPath . '/' . $filename;

            // Move uploaded file
            $file->move($uploadPath, $filename);

            // Relative path for database storage
            $relativePath = 'quotes/payment_proofs/' . $filename;

            // Store in raw_data
            $rawData = $quote->raw_data ?? [];
            $rawData['remaining_payment_proof'] = $relativePath;
            $quote->raw_data = $rawData;
            $quote->save();

            return response()->json([
                'status' => 'success',
                'message' => 'Remaining payment proof uploaded successfully.',
                'data' => [
                    'proof_path' => $relativePath,
                    'proof_url' => asset($relativePath),
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Error uploading remaining payment proof: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
}
