<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Testimonial;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;

class TestimonialController extends Controller
{
    /**
     * Display a listing of testimonials.
     */
    public function index(Request $request)
    {
        $query = Testimonial::query();

        // Search filter
        if ($request->filled('keyword')) {
            $keyword = $request->keyword;
            $query->where(function($q) use ($keyword) {
                $q->where('customer_name', 'like', "%{$keyword}%")
                  ->orWhere('company_name', 'like', "%{$keyword}%")
                  ->orWhere('short_description', 'like', "%{$keyword}%")
                  ->orWhere('long_description', 'like', "%{$keyword}%");
            });
        }

        // Status filter
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        // Rating filter
        if ($request->filled('rating')) {
            $query->where('rating', '>=', $request->rating);
        }

        $testimonials = $query->orderBy('sort_order')->orderBy('created_at', 'desc')->get();

        return view('Admin.Testimonial.index', compact('testimonials'));
    }

    /**
     * Show the form for creating a new testimonial.
     */
    public function create()
    {
        return view('Admin.Testimonial.create');
    }

    /**
     * Store a newly created testimonial.
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_name' => 'required|string|max:255',
            'short_description' => 'nullable|string|max:500',
            'long_description' => 'nullable|string',
            'rating' => 'required|numeric|min:0|max:5|regex:/^\d+(\.\d{1,2})?$/',
            'customer_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'customer_position' => 'nullable|string|max:255',
            'company_name' => 'nullable|string|max:255',
            'is_active' => 'nullable|boolean',
        ]);

        $data = $request->except(['customer_image']);
        $data['created_by'] = Auth::id();
        $data['updated_by'] = Auth::id();
        $data['is_active'] = $request->has('is_active') ? true : false;

        // Handle customer image upload
        if ($request->hasFile('customer_image')) {
            $imageDir = public_path('testimonials');
            if (!File::exists($imageDir)) {
                File::makeDirectory($imageDir, 0755, true);
            }

            $image = $request->file('customer_image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $image->move($imageDir, $imageName);
            $data['customer_image'] = 'testimonials/' . $imageName;
        }

        Testimonial::create($data);

        if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Testimonial created successfully.'
            ]);
        }

        return redirect()->route('admin.testimonials.index')
            ->with('success', 'Testimonial created successfully.');
    }

    /**
     * Display the specified testimonial.
     */
    public function show($id)
    {
        $testimonial = Testimonial::findOrFail($id);
        
        // Return HTML content for modal
        return view('Admin.Testimonial.show', compact('testimonial'))->render();
    }

    /**
     * Show the form for editing the specified testimonial.
     */
    public function edit($id)
    {
        $testimonial = Testimonial::findOrFail($id);
        return view('Admin.Testimonial.edit', compact('testimonial'));
    }

    /**
     * Update the specified testimonial.
     */
    public function update(Request $request, $id)
    {
        $testimonial = Testimonial::findOrFail($id);

        $request->validate([
            'customer_name' => 'required|string|max:255',
            'short_description' => 'nullable|string|max:500',
            'long_description' => 'nullable|string',
            'rating' => 'required|numeric|min:0|max:5|regex:/^\d+(\.\d{1,2})?$/',
            'customer_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'customer_position' => 'nullable|string|max:255',
            'company_name' => 'nullable|string|max:255',
            'is_active' => 'nullable|boolean',
        ]);

        $data = $request->except(['customer_image']);
        $data['updated_by'] = Auth::id();
        $data['is_active'] = $request->has('is_active') ? true : false;

        // Handle customer image upload
        if ($request->hasFile('customer_image')) {
            // Delete old image if exists
            if ($testimonial->customer_image && File::exists(public_path($testimonial->customer_image))) {
                File::delete(public_path($testimonial->customer_image));
            }

            $imageDir = public_path('testimonials');
            if (!File::exists($imageDir)) {
                File::makeDirectory($imageDir, 0755, true);
            }

            $image = $request->file('customer_image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $image->move($imageDir, $imageName);
            $data['customer_image'] = 'testimonials/' . $imageName;
        }

        $testimonial->update($data);

        if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Testimonial updated successfully.'
            ]);
        }

        return redirect()->route('admin.testimonials.index')
            ->with('success', 'Testimonial updated successfully.');
    }

    /**
     * Remove the specified testimonial.
     */
    public function destroy($id)
    {
        try {
            $testimonial = Testimonial::findOrFail($id);

            // Delete customer image if exists
            if ($testimonial->customer_image && File::exists(public_path($testimonial->customer_image))) {
                File::delete(public_path($testimonial->customer_image));
            }

            $testimonial->delete();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'Testimonial deleted successfully.'
                ]);
            }

            return redirect()->route('admin.testimonials.index')
                ->with('success', 'Testimonial deleted successfully.');
        } catch (\Exception $e) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to delete testimonial: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->route('admin.testimonials.index')
                ->with('error', 'Failed to delete testimonial.');
        }
    }

    /**
     * Change testimonial status.
     */
    public function changeStatus($id)
    {
        $testimonial = Testimonial::findOrFail($id);
        $testimonial->is_active = !$testimonial->is_active;
        $testimonial->save();

        return response()->json([
            'status' => 'success',
            'message' => 'Testimonial status updated successfully.',
            'is_active' => $testimonial->is_active
        ]);
    }
}
