<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Truck;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\General\GeneralSettingsClass;

class TruckController extends Controller
{
    public function index(Request $request)
    {
        $query = Truck::query();

        // Keyword search (truck name, registration number)
        if ($request->filled('keyword')) {
            $keyword = $request->keyword;
            $query->where(function($q) use ($keyword) {
                $q->where('truck_name', 'like', "%{$keyword}%")
                  ->orWhere('registration_number', 'like', "%{$keyword}%")
                  ->orWhere('make', 'like', "%{$keyword}%")
                  ->orWhere('model', 'like', "%{$keyword}%")
                  ->orWhere('driver_name', 'like', "%{$keyword}%");
            });
        }

        // Truck type filter
        if ($request->filled('truck_type')) {
            $query->where('truck_type', $request->truck_type);
        }

        // Status filter (is_active)
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        // Availability status filter
        if ($request->filled('availability')) {
            $query->where('availability_status', $request->availability);
        }

        // Pricing type filter
        if ($request->filled('pricing_type')) {
            $query->where('pricing_type', $request->pricing_type);
        }

        // Fuel type filter
        if ($request->filled('fuel_type')) {
            $query->where('fuel_type', $request->fuel_type);
        }

        // Driving license filter
        if ($request->filled('driving_license')) {
            $query->where('driving_license_required', $request->driving_license);
        }

        // Base location filter
        if ($request->filled('base_location')) {
            $query->where('base_location', 'like', "%{$request->base_location}%");
        }

        $trucks = $query->ordered()->get();
        
        return view('Admin.Truck.index', compact('trucks'));
    }

    public function create()
    {
        return view('Admin.Truck.create');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'truck_name' => 'required|string|max:255',
            'truck_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'truck_type' => 'required|string|in:light_rigid,medium_rigid,heavy_rigid,multi_combination',
            'payload_capacity' => 'nullable|numeric|min:0',
            'cubic_capacity' => 'nullable|numeric|min:0',
            'pricing_type' => 'required|string|in:global,custom',
            'custom_per_cubic_meter_price' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->all();
        
        // Handle image upload
        if ($request->hasFile('truck_image')) {
            $image = $request->file('truck_image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('trucks'), $imageName);
            $data['truck_image'] = 'trucks/' . $imageName;
        }

        Truck::create($data);

        return redirect()->route('admin.trucks.index')
            ->with('success', 'Truck created successfully.');
    }

    public function show($id)
    {
        $truck = Truck::findOrFail($id);
        return view('Admin.Truck.show', compact('truck'));
    }

    public function edit($id)
    {
        $truck = Truck::findOrFail($id);
        return view('Admin.Truck.edit', compact('truck'));
    }

    public function update(Request $request, $id)
    {
        $truck = Truck::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'truck_name' => 'required|string|max:255',
            'truck_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'truck_type' => 'required|string|in:light_rigid,medium_rigid,heavy_rigid,multi_combination',
            'payload_capacity' => 'nullable|numeric|min:0',
            'cubic_capacity' => 'nullable|numeric|min:0',
            'pricing_type' => 'required|string|in:global,custom',
            'custom_fixed_price' => 'nullable|numeric|min:0',
            'custom_per_cubic_meter_price' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->all();
        
        // Handle image upload
        if ($request->hasFile('truck_image')) {
            // Delete old image if exists
            if ($truck->truck_image && file_exists(public_path($truck->truck_image))) {
                unlink(public_path($truck->truck_image));
            }
            
            $image = $request->file('truck_image');
            $imageName = time() . '_' . $image->getClientOriginalName();
            $image->move(public_path('trucks'), $imageName);
            $data['truck_image'] = 'trucks/' . $imageName;
        }

        $truck->update($data);

        return redirect()->route('admin.trucks.index')
            ->with('success', 'Truck updated successfully.');
    }

    public function destroy($id)
    {
        try {
            $truck = Truck::findOrFail($id);
            $truck->delete();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'Truck deleted successfully.'
                ]);
            }

            return redirect()->route('admin.trucks.index')
                ->with('success', 'Truck deleted successfully.');
        } catch (\Exception $e) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to delete truck: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->route('admin.trucks.index')
                ->with('error', 'Failed to delete truck.');
        }
    }

    public function changeStatus(Request $request)
    {
        $truck = Truck::find($request->id);
        if ($truck) {
            $truck->is_active = !$truck->is_active;
            $truck->save();
            
            return response()->json([
                'status' => 'success',
                'message' => 'Truck status updated successfully.',
                'is_active' => $truck->is_active
            ]);
        }
        
        return response()->json(['status' => 'error', 'message' => 'Truck not found.'], 404);
    }

    public function getNextSortOrder()
    {
        $maxOrder = Truck::max('sort_order');
        return response()->json(['next_order' => ($maxOrder ?? 0) + 1]);
    }
}
