<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Blog extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'title',
        'permalink',
        'description',
        'content',
        'cover_image',
        'featured_images',
        'fimage_alt_text',
        'image_tagline',
        'image_description',
        'video_url',
        'video_content',
        'blog_categories_id',
        'is_featured',
        '_seo_title',
        '_seo_tags',
        '_seo_description',
        '_canonical_url',
        '_schema_markup',
        'active',
        'created_by',
        'updated_by',
        '_page_index',
        'image',
        'image_alt_text',
        
    ];

    protected $casts = [
        'active' => 'integer',
        'created_by' => 'integer',
        'updated_by' => 'integer',
        'is_featured' => 'boolean',
        'blog_categories_id' => 'array',
    ];
    // Accessor for featured images array
    public function getFeaturedImagesArrayAttribute()
    {
        if (empty($this->featured_images)) {
            return [];
        }
        return array_filter(explode(',', $this->featured_images));
    }

    // Mutator for featured images
    public function setFeaturedImagesAttribute($value)
    {
        if (is_array($value)) {
            $this->attributes['featured_images'] = implode(',', array_filter($value));
        } else {
            $this->attributes['featured_images'] = $value;
        }
    }

    // Relationship with blog categories - using blog_categories_id array
    // Since we store IDs in blog_categories_id column, we access them via accessor
    public function getBlogCategoriesAttribute()
    {
        $categoryIds = $this->blog_category_ids;
        if (empty($categoryIds) || !is_array($categoryIds)) {
            return collect([]);
        }
        return BlogCategory::whereIn('id', $categoryIds)->get();
    }

    // Helper methods to get IDs
    public function getBlogCategoryIdsAttribute()
    {
        // Get directly from the attributes array to avoid circular dependency
        $blogCategoriesId = $this->attributes['blog_categories_id'] ?? null;
        
        if (!empty($blogCategoriesId)) {
            // If it's already an array (from cast), return it
            if (is_array($blogCategoriesId)) {
                return $blogCategoriesId;
            }
            // Otherwise, try to decode JSON
            $decoded = json_decode($blogCategoriesId, true);
            if (is_array($decoded)) {
                return $decoded;
            }
        }
        // Return empty array if no categories found
        return [];
    }
    
    // Mutator for blog_categories_id to ensure it's stored as JSON
    public function setBlogCategoriesIdAttribute($value)
    {
        if (is_array($value)) {
            $this->attributes['blog_categories_id'] = json_encode(array_values(array_filter($value)));
        } else {
            $this->attributes['blog_categories_id'] = $value;
        }
    }

    public function getBlogwithPermalink($permalink)
    {
        $blog = $this->where('permalink', $permalink)->first();
        if ($blog) {
            return $blog;
        }
        return null; // or handle inactive blog case
    }

    // Helper methods
    public function isGeneral()
    {
        $categoryIds = $this->blog_category_ids;
        return empty($categoryIds) || (is_array($categoryIds) && count($categoryIds) === 0);
    }

    public function getStatusText()
    {
        $statuses = [
            0 => 'Draft',
            1 => 'Public',
            2 => 'Private',
            3 => 'Trash'
        ];
        return $statuses[$this->active] ?? 'Unknown';
    }

    public function getStatusBadge()
    {
        $badges = [
            0 => '<span class="badge bg-secondary">Draft</span>',
            1 => '<span class="badge bg-success">Public</span>',
            2 => '<span class="badge bg-warning">Private</span>',
            3 => '<span class="badge bg-danger">Trash</span>'
        ];
        return $badges[$this->active] ?? '<span class="badge bg-dark">Unknown</span>';
    }

    // Creator relationship
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
}
