<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class FAQ extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'faqs';

    protected $fillable = [
        'page_url',
        'permalink',
        'title',
        'description',
        'sort_order',
        'status',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'sort_order' => 'integer',
        'status' => 'integer',
        'created_by' => 'integer',
        'updated_by' => 'integer',
    ];

    /**
     * Get FAQ with permalink including active questions and answers
     */
    public function getFAQWithPermalink($permalink)
    {
        return $this->where('permalink', $permalink)
            ->where('status', 1)
            ->with(['questions' => function($query) {
                $query->where('status', 1)
                      ->orderBy('sort_order', 'asc')
                      ->orderBy('created_at', 'asc');
            }])
            ->first();
    }

    /**
     * Get FAQ with permalink for admin (includes inactive questions)
     */
    public function getFAQWithPermalinkForAdmin($permalink)
    {
        return $this->where('permalink', $permalink)
            ->with(['questions' => function($query) {
                $query->orderBy('sort_order', 'asc')
                      ->orderBy('created_at', 'asc');
            }])
            ->first();
    }

    /**
     * Get status text
     */
    public function getStatusText()
    {
        return $this->status ? 'Active' : 'Inactive';
    }

    /**
     * Get status badge
     */
    public function getStatusBadge()
    {
        return $this->status 
            ? '<span class="badge bg-success-subtle text-success">Active</span>'
            : '<span class="badge bg-secondary-subtle text-secondary">Inactive</span>';
    }

    /**
     * Get formatted questions and answers
     */
    public function getFormattedQA()
    {
        return $this->questions()->ordered()->get()->map(function ($question) {
            return [
                'id' => $question->id,
                'question' => $question->question,
                'answer' => $question->answer,
                'sort_order' => $question->sort_order,
                'status' => $question->status,
            ];
        })->toArray();
    }

    /**
     * Get active questions and answers only
     */
    public function getActiveQA()
    {
        return $this->questions()
            ->where('status', 1)
            ->orderBy('sort_order', 'asc')
            ->orderBy('created_at', 'asc')
            ->get()
            ->map(function ($question) {
                return [
                    'id' => $question->id,
                    'question' => $question->question,
                    'answer' => $question->answer,
                    'sort_order' => $question->sort_order,
                    'status' => $question->status,
                ];
            })->toArray();
    }

    /**
     * Static method to get FAQ with permalink including active questions
     */
    public static function getFAQByPermalink($permalink)
    {
        return self::where('permalink', $permalink)
            ->where('status', 1)
            ->with(['questions' => function($query) {
                $query->where('status', 1)
                      ->orderBy('sort_order', 'asc')
                      ->orderBy('created_at', 'asc');
            }])
            ->first();
    }

    /**
     * Get FAQ with all questions (active and inactive) for admin
     */
    public function getFAQWithAllQuestions()
    {
        return $this->load(['questions' => function($query) {
            $query->orderBy('sort_order', 'asc')
                  ->orderBy('created_at', 'asc');
        }]);
    }

   
    /**
     * Questions relationship
     */
    public function questions()
    {
        return $this->hasMany(FAQQuestion::class, 'faq_id');
    }

    /**
     * Active questions relationship
     */
    public function activeQuestions()
    {
        return $this->hasMany(FAQQuestion::class, 'faq_id')->active()->ordered();
    }

    /**
     * Creator relationship
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Updater relationship
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Scope for active FAQs
     */
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    /**
     * Scope for ordered FAQs
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order', 'asc');
    }
}