<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Job extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'permalink',
        'description',
        'user_type',
        'employment_type',
        'location',
        'salary_min',
        'salary_max',
        'valid_from',
        'valid_until',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'valid_from' => 'date',
        'valid_until' => 'date',
        'is_active' => 'boolean',
        'sort_order' => 'integer',
        'salary_min' => 'decimal:2',
        'salary_max' => 'decimal:2',
    ];

    // User type enum values
    const USER_TYPES = [
        'driver' => 'Driver',
        'mover' => 'Mover',
        'supervisor' => 'Supervisor',
        'manager' => 'Manager',
        'admin' => 'Admin',
        'other' => 'Other'
    ];

    // Employment type enum values
    const EMPLOYMENT_TYPES = [
        'full_time' => 'Full Time',
        'part_time' => 'Part Time',
        'contract' => 'Contract',
        'temporary' => 'Temporary',
        'internship' => 'Internship'
    ];

    public function applications()
    {
        return $this->hasMany(JobApplication::class);
    }

    public function activeApplications()
    {
        return $this->hasMany(JobApplication::class)->where('status', '!=', 'rejected');
    }

    public function getApplicationCountAttribute()
    {
        return $this->applications()->count();
    }

    public function getActiveApplicationCountAttribute()
    {
        return $this->activeApplications()->count();
    }

    // Scope for active jobs
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where('valid_from', '<=', now())
                    ->where('valid_until', '>=', now());
    }

    // Scope for ordering
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order');
    }

    // Scope for valid jobs (within date range)
    public function scopeValid($query)
    {
        return $query->where('valid_from', '<=', now())
                    ->where('valid_until', '>=', now());
    }

    // Get formatted salary range
    public function getFormattedSalaryAttribute()
    {
        if ($this->salary_min && $this->salary_max) {
            return '$' . number_format($this->salary_min) . ' - $' . number_format($this->salary_max);
        } elseif ($this->salary_min) {
            return 'From $' . number_format($this->salary_min);
        } elseif ($this->salary_max) {
            return 'Up to $' . number_format($this->salary_max);
        }
        return 'Salary not specified';
    }

    // Get employment type label
    public function getEmploymentTypeLabelAttribute()
    {
        return self::EMPLOYMENT_TYPES[$this->employment_type] ?? $this->employment_type;
    }

    // Get user type label
    public function getUserTypeLabelAttribute()
    {
        return self::USER_TYPES[$this->user_type] ?? $this->user_type;
    }

    // Get next sort order for new jobs
    public static function getNextSortOrder()
    {
        $lastJob = self::orderBy('sort_order', 'desc')->first();
        return $lastJob ? $lastJob->sort_order + 1 : 1;
    }
}
