<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Property extends Model
{
    use HasFactory;

    protected $fillable = [
        'location',
        'type',
        'uuid',
        'move_date',
    ];

    protected $casts = [
        'type' => 'string',
        'property_type' => 'string',
        'move_date' => 'date',
    ];
    
    /**
     * Prepare a date for array / JSON serialization.
     * This ensures move_date is always serialized as Y-m-d format (date only, no time)
     */
    protected function serializeDate(\DateTimeInterface $date)
    {
        // Return date only in Y-m-d format (no time component)
        return $date->format('Y-m-d');
    }
    
    /**
     * Get the move_date attribute formatted as Y-m-d (date only, no time)
     * This ensures the date is returned as a string in Y-m-d format, not a datetime
     */
    public function getMoveDateAttribute($value)
    {
        if ($value instanceof \Carbon\Carbon) {
            // Return date only, no time component
            return $value->format('Y-m-d');
        }
        if ($value) {
            // Parse and return date only, no time component
            return \Carbon\Carbon::parse($value)->format('Y-m-d');
        }
        return $value;
    }
    
    /**
     * Set the move_date attribute (mutator)
     * Ensures date is stored as date only without time component or timezone conversion
     */
    public function setMoveDateAttribute($value)
    {
        if ($value) {
            // Parse and format as date only (no time, no timezone conversion)
            // Just extract the date part from the value
            if (is_string($value)) {
                // If it's already in Y-m-d format, use it directly
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
                    $this->attributes['move_date'] = $value;
                } else {
                    // Parse and extract date part only
                    $date = \Carbon\Carbon::parse($value);
                    $this->attributes['move_date'] = $date->format('Y-m-d');
                }
            } elseif ($value instanceof \Carbon\Carbon) {
                // If it's already a Carbon instance, just format as date
                $this->attributes['move_date'] = $value->format('Y-m-d');
            } else {
                $this->attributes['move_date'] = $value;
            }
        } else {
            $this->attributes['move_date'] = $value;
        }
    }
    
    /**
     * Convert the model instance to an array.
     * Override to ensure move_date is always a string (date only)
     */
    public function toArray()
    {
        $array = parent::toArray();
        
        // Ensure move_date is always a string in Y-m-d format (date only, no time)
        if (isset($array['move_date'])) {
            if ($array['move_date'] instanceof \Carbon\Carbon) {
                $array['move_date'] = $array['move_date']->format('Y-m-d');
            } elseif (is_string($array['move_date']) && strpos($array['move_date'], 'T') !== false) {
                // If it's a datetime string, extract just the date part
                $array['move_date'] = \Carbon\Carbon::parse($array['move_date'])->format('Y-m-d');
            }
        }
        
        return $array;
    }

    public function propertyDetails()
    {
        return $this->hasOne(PropertyDetail::class);
    }

    public function pickUpBookings()
    {
        return $this->hasMany(Booking::class, 'pick_up_property_id');
    }

    public function dropOffBookings()
    {
        return $this->hasMany(Booking::class, 'drop_off_property_id');
    }
    
    public function bookingItems()
    {
        return $this->hasMany(BookingItem::class, 'uuid', 'uuid');
    }

    // Scopes
    public function scopePickUp($query)
    {
        return $query->where('type', 'pick_up');
    }

    public function scopeDropOff($query)
    {
        return $query->where('type', 'drop_off');
    }

   

    // Accessors
    public function getTypeTextAttribute()
    {
        return ucfirst(str_replace('_', ' ', $this->type));
    }

    public function getPropertyTypeTextAttribute()
    {
        return $this->property_type;
    }
}