@include('Admin.Includes.header')
@php
    use Illuminate\Support\Facades\Auth;
    $logged_in_user = Auth::user();
@endphp

<div class="page-content">
    <div class="card shadow-sm mb-4">
        <div class="p-3 d-flex justify-content-between align-items-center flex-wrap">
            <div class="clearfix">
                <div class="pd-t-5 pd-b-5">
                    <h1 class="fs-4 pd-0 mg-0 tx-20 tx-dark">{{ __('Child Categories') }}</h1>
                </div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb bg-transparent p-0 m-0 align-items-center">
                        <li class="breadcrumb-item">
                            <a href="{{ route('Admin.Dashboard') }}" class="link-primary text-decoration-none">
                                <i class="fa-solid fa-house me-1"></i> Dashboard
                            </a>
                        </li>
                        <li class="breadcrumb-item active" aria-current="page">
                            Child Categories
                        </li>
                    </ol>
                </nav>
            </div>
        </div>
    </div>

    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    @if ($errors->any())
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    <div class="row">
        <div class="col-xxl-5">
            <!-- Basic Child Category Information Card -->
            <div class="card mb-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">Create Child Category</h6>
                </div>
                <div class="card-body">
                    <form autocomplete="off" method="POST" action="{{ route('child.category.store') }}"
                        enctype="multipart/form-data" id="childCategoryForm">
                        @csrf
                        <input type="hidden" name="child_category_id" id="child_category_id">
                        <div id="update_success_message"></div>
                        <div id="saveform_errList" role="alert"></div>

                        <div class="row">
                            <!-- Parent Category -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="parent_category" class="form-label">Parent Category<span
                                            class="text-danger">*</span></label>
                                    <select id="parent_category" name="main_parent_category" required
                                        class="form-select">
                                        <option value="">Select Parent Category</option>
                                        @foreach ($categories as $category)
                                            <option value="{{ $category->id }}">{{ $category->title }}</option>
                                        @endforeach
                                    </select>
                                    <div class="invalid-feedback">Please select a parent category.</div>
                                </div>
                            </div>

                            <!-- Title -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="child_category_title" class="form-label">Child Category Title<span
                                            class="text-danger">*</span></label>
                                    <input type="text" id="child_category_title" name="title" required
                                        class="form-control" placeholder="Enter Child Category">
                                    <div class="invalid-feedback">Please enter a child category title.</div>
                                </div>
                            </div>

                            <!-- Volumn -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="child_category_volumn" class="form-label">Volume (/m³)</label>
                                    <input type="number" id="child_category_volumn" name="volumn" class="form-control"
                                        placeholder="e.g., 15 /m³">
                                </div>
                            </div>

                            <!-- Description -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="child_category_description" class="form-label">Description</label>
                                    <textarea id="child_category_description" name="description" class="form-control" rows="3"
                                        placeholder="Enter child category description"></textarea>
                                </div>
                            </div>

                            <!-- Image -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="child_category_image" class="form-label">Image</label>
                                    <input type="file" id="child_category_image" name="image" class="form-control"
                                        accept="image/*" onchange="previewImage(this)">
                                    <div id="image_preview" class="mt-2" style="display: none;">
                                        <img id="preview_img" src="" alt="Preview"
                                            style="max-width: 200px; max-height: 150px; border-radius: 5px;">
                                    </div>
                                </div>
                            </div>

                            <!-- Status -->
                            <div class="col-6">
                                <div class="mb-3">
                                    <label for="child_category_status" class="form-label">Status</label>
                                    <select id="child_category_status" name="is_active" class="form-select">
                                        <option value="1">Active</option>
                                        <option value="0">Inactive</option>
                                    </select>
                                </div>
                            </div>

                            <!-- Sort Order -->
                            <div class="col-6">
                                <div class="mb-3">
                                    <label for="child_category_sort_order" class="form-label">Sort Order</label>
                                    <input type="number" id="child_category_sort_order" name="sort_order"
                                        class="form-control" value="0" min="0">
                                </div>
                            </div>
                        </div>

                        <div class="d-flex justify-content-end">
                            <button type="button" class="btn btn-light me-2" onclick="resetForm()">Reset</button>
                            <button type="submit" class="btn btn-success" id="submitBtn">Save Child
                                Category</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-xxl-7">
            <!-- Child Categories List Card -->
            <div class="card">
                <div class="card-header d-flex justify-content-between">
                    <h5 class="card-title mb-0">Child Category Table</h5>
                    <div class="d-flex gap-2">
                        <select id="parentFilter" class="form-select form-select-sm" style="width: 200px;">
                            <option value="">Filter by Parent Category</option>
                            @foreach ($categories as $category)
                                <option value="{{ $category->id }}">{{ $category->title }}</option>
                            @endforeach
                        </select>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="data_Table" class="table table-borderless">
                            <thead>
                                <tr>
                                    <th>Title</th>
                                    <th>Parent Category</th>
                                    <th>Volumn</th>
                                    <th>Status</th>
                                    <th>Sort Order</th>
                                    <th width="180px">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($childCategories as $childCategory)
                                    <tr class="align-middle">
                                        <td>
                                            <div class="d-flex align-items-center">
                                                @if ($childCategory->image)
                                                    <img src="{{ asset($childCategory->image) }}"
                                                        alt="{{ $childCategory->title }}" class="rounded me-2"
                                                        style="width: 30px; height: 30px; object-fit: cover;">
                                                @endif
                                                <div>
                                                    <div class="fw-medium">{{ $childCategory->title ?? '' }}</div>
                                                    @if ($childCategory->description)
                                                        <small
                                                            class="text-muted">{{ Str::limit($childCategory->description ?? '', 50) }}</small>
                                                    @endif
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <span
                                                class="badge bg-primary">{{ $childCategory->mainParentCategory->title ?? '' }}</span>
                                        </td>
                                        <td>
                                            @if ($childCategory->volumn)
                                                <span class="text-muted">{{ $childCategory->volumn ?? '' }} /m³</span>
                                            @else
                                                <span class="text-muted">-</span>
                                            @endif
                                        </td>
                                        <td class="status-cell" data-child-category-id="{{ $childCategory->id }}">
                                            @if ($childCategory->is_active)
                                                <span
                                                    class="badge bg-success-subtle text-success px-3 py-2">Active</span>
                                            @else
                                                <span
                                                    class="badge bg-danger-subtle text-danger px-3 py-2">Inactive</span>
                                            @endif
                                        </td>
                                        <td>{{ $childCategory->sort_order ?? '' }}</td>
                                        <td>
                                            <div class="dropdown">
                                                <button class="btn btn-sm text-dark rounded shadow-sm" type="button"
                                                    data-bs-toggle="dropdown" aria-expanded="false"
                                                    style="background-color: #c2d6e7;">
                                                    &#8942;
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li>
                                                        <button class="dropdown-item"
                                                            onclick="editChildCategory({{ $childCategory->id }})">
                                                            <i class="bi bi-pencil-square me-2"></i>Edit
                                                        </button>
                                                    </li>
                                                    <li>
                                                        <button class="dropdown-item"
                                                            onclick="updateStatus({{ $childCategory->id }}, '{{ $childCategory->is_active ? 0 : 1 }}', this)">
                                                            <i
                                                                class="fa fa-{{ $childCategory->is_active ? 'ban' : 'check' }} me-2"></i>{{ $childCategory->is_active ? 'Inactive' : 'Active' }}
                                                        </button>
                                                    </li>
                                                    <li>
                                                        <button class="dropdown-item text-danger"
                                                            onclick="confirmDeleteChildCategory({{ $childCategory->id }}, '{{ $childCategory->title }}')">
                                                            <i class="bi bi-trash me-2"></i>Delete
                                                        </button>
                                                    </li>
                                                </ul>
                                            </div>
                                        </td>
                                    </tr>
                                @empty
                                    <tr class="no-data-row">
                                        <td></td>
                                        <td></td>
                                        <td class="text-end text-muted py-4">No data</td>
                                        <td></td>
                                        <td></td>
                                        <td></td>
                                    </tr>
                                @endforelse
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th>Title</th>
                                    <th>Parent Category</th>
                                    <th>Volumn</th>
                                    <th>Status</th>
                                    <th>Sort Order</th>
                                    <th width="180px">Actions</th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Delete Confirmation Modal --}}
<div class="modal fade zoomIn" id="deleteConfirmModal" tabindex="-1" aria-labelledby="deleteConfirmModalLabel"
    aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>

            <div class="modal-body p-md-5">
                <div class="text-center">
                    <div class="text-danger">
                        <i class="bi bi-trash display-4"></i>
                    </div>
                    <div class="mt-4 fs-15">
                        <p class="text-muted mx-3 fs-16 mb-0">
                            Are you sure you want to delete
                            <strong id="deleteCategoryName"></strong>?<br>
                            This action cannot be undone.
                        </p>
                        <p class="text-muted mt-2 mb-0">
                            <small>Suggestion: mark it as <em>Inactive</em> instead.</small>
                        </p>
                    </div>
                </div>

                <div class="d-flex gap-2 justify-content-center mt-4 mb-2">
                    <button type="button" class="btn w-sm btn-light btn-hover"
                        data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn w-sm btn-danger btn-hover" id="confirmDeleteBtn">Yes, Delete
                        It!</button>
                </div>
            </div>
        </div>
    </div>
</div>

@include('Admin.Includes.footer')

<script>
    // Image preview
    function previewImage(input) {
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            reader.onload = function(e) {
                document.getElementById('preview_img').src = e.target.result;
                document.getElementById('image_preview').style.display = 'block';
            }
            reader.readAsDataURL(input.files[0]);
        }
    }

    // Reset form
    function resetForm() {
        document.getElementById('childCategoryForm').reset();
        document.getElementById('child_category_id').value = '';
        document.getElementById('image_preview').style.display = 'none';
        document.getElementById('submitBtn').textContent = 'Save Child Category';
        loadNextChildCategorySortOrder();
    }

    // Load next sort order for child categories
    function loadNextChildCategorySortOrder() {
        $.ajax({
            url: "{{ route('admin.child_categories.nextSortOrder') }}",
            type: "GET",
            success: function(response) {
                // Always set the next sort order as default value
                $('#child_category_sort_order').val(response.next_sort_order);
            },
            error: function() {
                $('#child_category_sort_order').val('1');
            }
        });
    }

    // Edit child category
    function editChildCategory(id) {
        fetch(`/edit_child_category?id=${id}`)
            .then(response => response.json())
            .then(data => {
                document.getElementById('child_category_id').value = data.id;
                document.getElementById('parent_category').value = data.main_parent_category;
                document.getElementById('child_category_title').value = data.title;
                document.getElementById('child_category_volumn').value = data.volumn || '';
                document.getElementById('child_category_description').value = data.description || '';
                document.getElementById('child_category_status').value = data.is_active ? '1' : '0';
                document.getElementById('child_category_sort_order').value = data.sort_order || 0;
                document.getElementById('submitBtn').textContent = 'Update Child Category';

                // Show current image if exists
                if (data.image) {
                    document.getElementById('preview_img').src = `{{ asset('') }}${data.image}`;
                    document.getElementById('image_preview').style.display = 'block';
                }

                // Scroll to form
                document.getElementById('childCategoryForm').scrollIntoView({
                    behavior: 'smooth'
                });
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error loading child category data');
            });
    }


    // Update status function
    function updateStatus(childCategoryId, status, element) {
        const statusText = status === '1' ? 'Active' : 'Inactive';

        $.ajaxSetup({
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        });

        $.ajax({
            url: `/update_child_category/${childCategoryId}`,
            type: 'POST',
            data: {
                is_active: status,
                _token: '{{ csrf_token() }}'
            },
            beforeSend: function() {
                $(element).prop('disabled', true);
            },
            success: function(response) {
                Swal.fire({
                    icon: 'success',
                    title: 'Status Updated',
                    text: 'Child category status updated successfully',
                    timer: 1500,
                    showConfirmButton: false
                });

                const row = $(element).closest('tr');
                const statusCell = row.find('.status-cell');

                if (status === '1') {
                    statusCell.html(
                        '<span class="badge bg-success-subtle text-success px-3 py-2">Active</span>');
                } else {
                    statusCell.html(
                        '<span class="badge bg-danger-subtle text-danger px-3 py-2">Inactive</span>');
                }
                setTimeout(function() {
                    window.location.reload();
                }, 1000); 
            },
            error: function(xhr) {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'Failed to update status'
                });
            },
            complete: function() {
                $(element).prop('disabled', false);
            }
        });
    }

    // Delete confirmation function
    function confirmDeleteChildCategory(id, name) {
        document.getElementById('deleteCategoryName').textContent = name;
        document.getElementById('confirmDeleteBtn').onclick = function() {
            deleteChildCategory(id);
        };
        new bootstrap.Modal(document.getElementById('deleteConfirmModal')).show();
    }

    // Delete child category function
    function deleteChildCategory(id) {
        // Create a form to submit DELETE request
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = `{{ route('child.category.delete') }}`;

        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'child_category';
        idInput.value = id;

        const tokenInput = document.createElement('input');
        tokenInput.type = 'hidden';
        tokenInput.name = '_token';
        tokenInput.value = '{{ csrf_token() }}';

        form.appendChild(idInput);
        form.appendChild(tokenInput);
        document.body.appendChild(form);
        form.submit();
    }

    // Parent category filter
    document.getElementById('parentFilter').addEventListener('change', function() {
        const filterValue = this.value;
        const table = $('#data_Table').DataTable();

        if (filterValue === '') {
            table.column(1).search('').draw(); // Clear filter on Parent Category column
        } else {
            // Get the selected option text
            const selectedText = this.options[this.selectedIndex].text;
            table.column(1).search(selectedText).draw(); // Filter by Parent Category column
        }
    });

    // Initialize DataTable properly
    $(document).ready(function() {
        // Load next sort order on page load
        loadNextChildCategorySortOrder();

        // Destroy existing DataTable if it exists
        if ($.fn.DataTable.isDataTable('#data_Table')) {
            $('#data_Table').DataTable().destroy();
        }

        // Initialize DataTable with proper configuration
        $('#data_Table').DataTable({
            responsive: true,
            autoWidth: false,
            columnDefs: [{
                    orderable: false,
                    targets: -1
                } // Disable sorting on "Actions" column
            ],
            language: {
                searchPlaceholder: 'Search...',
                sSearch: ''
            },
            "order": [
                [0, "desc"]
            ]
        });
    });
</script>
