<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Log;

class BlogImageHelper
{
    /**
     * Process summernote content and extract/save images
     */
    public static function processSummernoteContent($content, $blogId = null)
    {
        if (!$content) {
            return $content;
        }
        
        // Create directory if it doesn't exist
        $uploadPath = public_path('blogs/content-images');
        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }
        
        // First, process external HTTP/HTTPS image URLs
        $externalPattern = '/<img([^>]*?)src="(https?:\/\/[^"]+)"([^>]*?)>/i';
        $processedContent = preg_replace_callback($externalPattern, function($matches) use ($uploadPath, $blogId) {
            $beforeSrc = $matches[1];
            $imageUrl = $matches[2];
            $afterSrc = $matches[3];
            
            // Skip if already our local image
            if (strpos($imageUrl, '/blogs/content-images/') !== false || strpos($imageUrl, asset('blogs/content-images/')) !== false) {
                return $matches[0]; // Return original
            }
            
            try {
                // Download and save the external image
                $filename = 'blog_' . ($blogId ? $blogId . '_' : '') . time() . '_' . uniqid() . '.jpg';
                $filePath = $uploadPath . '/' . $filename;
                
                // Download the image
                $imageData = self::downloadImage($imageUrl);
                
                if ($imageData) {
                    // Save as JPEG
                    $image = imagecreatefromstring($imageData);
                    if ($image !== false) {
                        $width = imagesx($image);
                        $height = imagesy($image);
                        $jpegImage = imagecreatetruecolor($width, $height);
                        $white = imagecolorallocate($jpegImage, 255, 255, 255);
                        imagefill($jpegImage, 0, 0, $white);
                        imagecopy($jpegImage, $image, 0, 0, 0, 0, $width, $height);
                        imagejpeg($jpegImage, $filePath, 90);
                        imagedestroy($image);
                        imagedestroy($jpegImage);
                        
                        // Preserve existing attributes
                        $attributes = $beforeSrc . $afterSrc;
                        if (strpos($attributes, 'class=') === false) {
                            $attributes .= ' class="img-fluid"';
                        } else {
                            if (strpos($attributes, 'img-fluid') === false) {
                                $attributes = preg_replace('/class="([^"]*)"/', 'class="$1 img-fluid"', $attributes);
                            }
                        }
                        
                        return '<img' . $attributes . ' src="/blogs/content-images/' . $filename . '">';
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Failed to download external image', [
                    'url' => $imageUrl,
                    'error' => $e->getMessage()
                ]);
            }
            
            // Return original if download failed
            return $matches[0];
        }, $content);
        
        // Then, extract base64 images from summernote content while preserving all attributes
        $pattern = '/<img([^>]*?)src="data:image\/([^;]+);base64,([^"]+)"([^>]*?)>/i';
        $processedContent = preg_replace_callback($pattern, function($matches) use ($uploadPath, $blogId) {
            $beforeSrc = $matches[1];
            $imageType = $matches[2];
            $base64Data = $matches[3];
            $afterSrc = $matches[4];
            
            // Generate unique filename with JPEG extension
            $filename = 'blog_' . ($blogId ? $blogId . '_' : '') . time() . '_' . uniqid() . '.jpg';
            $filePath = $uploadPath . '/' . $filename;
            
            // Convert and save the image as JPEG
            self::convertToJpeg($base64Data, $imageType, $filePath);
            
            // Preserve existing attributes and add default ones if missing
            $attributes = $beforeSrc . $afterSrc;
            
            // Add alt attribute if not present
            if (strpos($attributes, 'alt=') === false) {
                $attributes .= ' alt="Blog Image"';
            }
            
            // Add class if not present
            if (strpos($attributes, 'class=') === false) {
                $attributes .= ' class="img-fluid"';
            } else {
                // Add img-fluid to existing class if not already present
                if (strpos($attributes, 'img-fluid') === false) {
                    $attributes = preg_replace('/class="([^"]*)"/', 'class="$1 img-fluid"', $attributes);
                }
            }
            
            // Return the new image with preserved and enhanced attributes
            return '<img' . $attributes . ' src="/blogs/content-images/' . $filename . '">';
        }, $processedContent);

        return $processedContent;
    }

    /**
     * Download image from external URL
     */
    private static function downloadImage($url)
    {
        try {
            // Initialize cURL
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            
            $imageData = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
            
            curl_close($ch);
            
            // Check if download was successful
            if ($httpCode === 200 && $imageData !== false) {
                // Verify it's an image
                if (strpos($contentType, 'image/') === 0) {
                    return $imageData;
                }
            }
            
            return false;
        } catch (\Exception $e) {
            Log::error('Failed to download external image', [
                'url' => $url,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Convert base64 image data to JPEG format and save
     */
    private static function convertToJpeg($base64Data, $originalType, $filePath)
    {
        try {
            // Decode the base64 data
            $imageData = base64_decode($base64Data);
            
            if ($imageData === false) {
                throw new \Exception('Failed to decode base64 data');
            }
            
            // Create image resource from the decoded data
            $image = imagecreatefromstring($imageData);
            
            if ($image === false) {
                throw new \Exception('Failed to create image from string');
            }
            
            // Create a white background for JPEG (since JPEG doesn't support transparency)
            $width = imagesx($image);
            $height = imagesy($image);
            $jpegImage = imagecreatetruecolor($width, $height);
            
            // Fill with white background
            $white = imagecolorallocate($jpegImage, 255, 255, 255);
            imagefill($jpegImage, 0, 0, $white);
            
            // Copy the original image onto the white background
            imagecopy($jpegImage, $image, 0, 0, 0, 0, $width, $height);
            
            // Save as JPEG with high quality (90%)
            $success = imagejpeg($jpegImage, $filePath, 90);
            
            // Clean up memory
            imagedestroy($image);
            imagedestroy($jpegImage);
            
            if (!$success) {
                throw new \Exception('Failed to save JPEG image');
            }
            
        } catch (\Exception $e) {
            // Fallback: save the original image data if conversion fails
            file_put_contents($filePath, base64_decode($base64Data));
            
            // Log the error for debugging
            Log::warning('Blog image conversion to JPEG failed, saved original format', [
                'error' => $e->getMessage(),
                'original_type' => $originalType,
                'file_path' => $filePath
            ]);
        }
    }

    /**
     * Process content to replace image paths with proper URLs
     */
    public static function processContentUrls($content)
    {
        if (!$content) {
            return null;
        }

        // Replace relative image paths with full URLs while preserving all img tag attributes
        $content = preg_replace_callback('/<img([^>]*?)src="([^"]+)"([^>]*?)>/i', function($matches) {
            $beforeSrc = $matches[1];
            $src = $matches[2];
            $afterSrc = $matches[3];
            
            // If it's a relative path to our blogs folder, make it absolute
            if (strpos($src, '/blogs/content-images/') === 0) {
                $newSrc = asset($src);
            } else {
                $newSrc = $src;
            }
            
            // Reconstruct the full img tag with all attributes preserved
            return '<img' . $beforeSrc . 'src="' . $newSrc . '"' . $afterSrc;
        }, $content);

        return $content;
    }
}

