<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\Property;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\QuoteCalculationTrait as TraitsQuoteCalculationTrait;
use App\Http\Controllers\API\Traits\EmailTrait;
use App\Models\Quote;

class BookingController extends Controller
{
    use JsonResponseTrait, TraitsQuoteCalculationTrait, EmailTrait;

    /**
     * Store booking (create or update) by UUID
     */
    public function store(Request $request)
    {
        $uuid = $request->uuid;
        if (!$uuid) {
            return $this->error('UUID is required', 422);
        }
        $pickUpProperty = Property::where('uuid', $uuid)->where('type', 'pick_up')->first();
        $dropOffProperty = Property::where('uuid', $uuid)->where('type', 'drop_off')->first();
        
        if (!$pickUpProperty || !$dropOffProperty) {
            return $this->error('Properties not found for this UUID', 404);
        }
        $booking = Booking::where('uuid', $uuid)->first();
        $isUpdate = $booking !== null;
        $syncableFields = [
            'user_id',
            'user_type',
            'name',
            'email',
            'phone',
            'status',
            'time_preference',
            'custom_time',
            'reason_to_move',
            'booking_item_ids',
            'difficult_items_notes',
            'truck_id',
            'truck_booking_hours',
            'is_completed'
        ];
        $bookingData = [];
        foreach ($syncableFields as $field) {
            if ($request->has($field)) {
                // Field exists in payload - use the value (even if null or empty array)
                $value = $request->input($field);
                if ($field === 'booking_item_ids' && is_array($value)) {
                    $bookingData[$field] = array_unique($value);
                } else {
                    $bookingData[$field] = $value;
                }
            } else {
                // Field not in payload
                if ($isUpdate) {
                    continue;
                } else {
                    if ($field === 'status') {
                        $bookingData[$field] = 'pending';
                    } elseif ($field === 'is_completed') {
                        $bookingData[$field] = false;
                    } else {
                        $bookingData[$field] = null;
                    }
                }
            }
        }
        $bookingData['pick_up_property_id'] = $pickUpProperty->id;
        $bookingData['drop_off_property_id'] = $dropOffProperty->id;
        $bookingData['move_date'] = $pickUpProperty->move_date;
        $bookingData['uuid'] = $uuid;
        
        // Check if request has name, email, and phone - if so, only update those fields
        $hasNameEmailPhone = $request->has('name') && $request->has('email') && $request->has('phone');
        
        if ($hasNameEmailPhone) {
            // Only update name, email, phone in user and quote - skip everything else
            $name = $request->input('name');
            $email = $request->input('email');
            $phone = $request->input('phone');
            
            // Get existing booking and quote
            $quote = Quote::where('uuid', $uuid)->first();
            $userId = null;
            
            // SIMPLIFIED: If request has user_id, it's an existing user - use it directly
            if ($request->has('user_id') && $request->user_id) {
                $user = User::find($request->user_id);
                if (!$user) {
                    return $this->error('User not found', 404);
                }
                // Use user's email from database (not request email)
                $email = $user->email;
                $userId = $user->id;
                
                // Update name and phone if provided
                $userUpdateData = [];
                if ($name && $user->name !== $name) {
                    $userUpdateData['name'] = $name;
                }
                if ($phone && $user->phone !== $phone) {
                    $userUpdateData['phone'] = $phone;
                }
                if (!empty($userUpdateData)) {
                    $user->update($userUpdateData);
                }
            }
            // Update user if booking exists and has user_id (and request doesn't have user_id)
            elseif ($booking && $booking->user_id) {
                $user = User::find($booking->user_id);
                if ($user) {
                    $userUpdateData = [];
                    
                    // Check if email is being changed
                    if ($email && $user->email !== $email) {
                        // Check if new email already exists for another user
                        $emailExists = User::where('email', $email)
                            ->where('id', '!=', $user->id)
                            ->exists();
                        
                        if ($emailExists) {
                            return $this->error('Email already exists for another user', 422);
                        }
                        $userUpdateData['email'] = $email;
                    }
                    
                    // Update name and phone
                    if ($name && $user->name !== $name) {
                        $userUpdateData['name'] = $name;
                    }
                    if ($phone && $user->phone !== $phone) {
                        $userUpdateData['phone'] = $phone;
                    }
                    
                    // Update user if there are changes
                    if (!empty($userUpdateData)) {
                        try {
                            $user->update($userUpdateData);
                        } catch (\Illuminate\Database\QueryException $e) {
                            if ($e->getCode() == 23000) {
                                return $this->error('Email already exists. Please use a different email address.', 422);
                            }
                            throw $e;
                        }
                    }
                    // Use user's email from database
                    $email = $user->email;
                    $userId = $user->id;
                }
            } else {
                // No existing user - create or find user
                $existingUser = User::where('email', $email)->first();
                
                if ($existingUser) {
                    // Update existing user
                    $updateData = [];
                    if ($name && $existingUser->name !== $name) {
                        $updateData['name'] = $name;
                    }
                    if ($phone && $existingUser->phone !== $phone) {
                        $updateData['phone'] = $phone;
                    }
                    if (!empty($updateData)) {
                        $existingUser->update($updateData);
                    }
                    $userId = $existingUser->id;
                    $email = $existingUser->email; // Use user's email
                } else {
                    // Create new user
                    try {
                        $randomPassword = $this->generateRandomPassword();
                        $user = User::create([
                            'name' => $name,
                            'email' => $email,
                            'phone' => $phone,
                            'password' => Hash::make($randomPassword),
                            'role_id' => 2,
                            'profile_image' => '/user.jpg',
                            'email_verified_at' => now(),
                            'active' => 1,
                        ]);
                        $userId = $user->id;
                        $this->sendBookingUserLoginEmail($user, $randomPassword);
                       
                    } catch (\Illuminate\Database\QueryException $e) {
                        if ($e->getCode() == 23000) {
                            return $this->error('Email already exists. Please use a different email address.', 422);
                        }
                        throw $e;
                    }
                }
            }
            
            // Update booking with user_id
            if ($booking && $userId) {
                $booking->user_id = $userId;
            }
            
            // Update quote raw_data with name, email, phone
            if ($quote) {
                $rawData = $quote->raw_data ?? [];
                $rawData['name'] = $name;
                $rawData['email'] = $email;
                $rawData['phone'] = $phone;
                $quote->raw_data = $rawData;
                $quote->save();
            }
            
            // Update booking name, email, phone - always use user's email
            if ($booking) {
                if ($userId) {
                    $user = User::find($userId);
                    if ($user) {
                        $booking->name = $user->name;
                        $booking->email = $user->email; // Always use user's email
                        $booking->phone = $user->phone;
                    }
                } else {
                    $booking->name = $name;
                    $booking->email = $email;
                    $booking->phone = $phone;
                }
                $booking->save();
            }
            
            // Return early - don't process anything else
            return $this->message('User information updated successfully');
        }
        
        // SIMPLIFIED: Handle user_id logic
        // If request has user_id, it's an existing user - use it directly
        if (!empty($bookingData['user_id'])) {
            // Existing user - verify user exists and use their data
            $existingUser = User::find($bookingData['user_id']);
            if (!$existingUser) {
                return $this->error('User not found', 404);
            }
            
            // Use user's email, name, phone in booking (not request data)
            $bookingData['email'] = $existingUser->email;
            $bookingData['name'] = $existingUser->name;
            $bookingData['phone'] = $existingUser->phone;
            
            // If booking exists and has different user_id, update it
            if ($booking && $booking->user_id != $bookingData['user_id']) {
                // User changed - update booking with new user
                $bookingData['user_id'] = $existingUser->id;
            }
        } 
        // If no user_id but has email, find or create user
        elseif (!empty($bookingData['email']) && !empty($bookingData['name']) && !empty($bookingData['phone'])) {
            // Check if user with this email already exists
            $existingUser = User::where('email', $bookingData['email'])->first();
            
            if ($existingUser) {
                // User exists - use their ID and update name/phone if needed
                $updateData = [];
                if ($existingUser->name !== $bookingData['name']) {
                    $updateData['name'] = $bookingData['name'];
                }
                if ($existingUser->phone !== $bookingData['phone']) {
                    $updateData['phone'] = $bookingData['phone'];
                }
                if (!empty($updateData)) {
                    $existingUser->update($updateData);
                }
                $bookingData['user_id'] = $existingUser->id;
                // Use user's email (not request email)
                $bookingData['email'] = $existingUser->email;
            } else {
                // Create new user
                try {
                    $randomPassword = $this->generateRandomPassword();
                    $user = User::create([
                        'name' => $bookingData['name'],
                        'email' => $bookingData['email'],
                        'phone' => $bookingData['phone'],
                        'password' => Hash::make($randomPassword),
                        'role_id' => 2,
                        'profile_image' => '',
                        'email_verified_at' => now(),
                        'active' => 1,
                    ]);
                    
                    $bookingData['user_id'] = $user->id;
                    
                    // Send login details email to new user
                    try {
                        $this->sendBookingUserLoginEmail($user, $randomPassword);
                    } catch (\Exception $e) {
                        \Illuminate\Support\Facades\Log::error('Failed to send booking user login email: ' . $e->getMessage());
                    }
                } catch (\Illuminate\Database\QueryException $e) {
                    if ($e->getCode() == 23000) {
                        return $this->error('Email already exists. Please use a different email address.', 422);
                    }
                    throw $e;
                }
            }
        }
        // If booking exists and has user_id, preserve it
        elseif ($booking && $booking->user_id) {
            $existingUser = User::find($booking->user_id);
            if ($existingUser) {
                $bookingData['user_id'] = $existingUser->id;
                $bookingData['email'] = $existingUser->email;
                $bookingData['name'] = $existingUser->name;
                $bookingData['phone'] = $existingUser->phone;
            }
        }
        
        // If request has required_truck_cubic_meter, it's a truck-only booking
        // Remove booking_item_ids from booking if they exist
        if ($request->has('required_truck_cubic_meter') && $request->filled('required_truck_cubic_meter')) {
            // Truck-only booking: clear booking_item_ids
            $bookingData['booking_item_ids'] = [];
            BookingItem::where('uuid', $uuid)->get()->each(function ($item) {
                $item->delete();
            });
        }
        
        // Create or update booking
        if ($booking) {
            $booking->update($bookingData);
            $booking->refresh();
        } else {
            $booking = Booking::create($bookingData);
        }
        if ($booking->truck_id) {
            // Refresh booking to ensure we have the latest data including removed items
            $booking->refresh();
            // Clear the relationship cache and reload with fresh data
            $booking->unsetRelation('bookingItems');
            $booking->load(['bookingItems.entity', 'truck']);
            
            $additionalTimeMinutes = $request->get('additional_time_minutes', 0);
            
            // Check if quote already exists
            $quote = Quote::where('uuid', $uuid)->first();
            
            // Create a request object with job_type and other calculation parameters
            // Priority: Use request values if available, otherwise use values from raw_data
            $requestForCalculation = new \Illuminate\Http\Request();
            
            // Job type: use from request if available, otherwise from raw_data
            if ($request->has('job_type') && !empty($request->job_type)) {
                $requestForCalculation->merge(['job_type' => $request->job_type]);
            } elseif ($quote && isset($quote->raw_data['job_type']) && !empty($quote->raw_data['job_type'])) {
                $requestForCalculation->merge(['job_type' => $quote->raw_data['job_type']]);
            } elseif ($quote && $quote->job_type && !empty($quote->job_type)) {
                $requestForCalculation->merge(['job_type' => $quote->job_type]);
            }
            
            // Extra movers: use from request if available, otherwise from raw_data
            if ($request->has('extra_movers')) {
                $requestForCalculation->merge(['extra_movers' => $request->extra_movers]);
            } elseif ($quote && isset($quote->raw_data['extra_movers'])) {
                $requestForCalculation->merge(['extra_movers' => $quote->raw_data['extra_movers']]);
            }
            
            // Callout fee times: use from request if available, otherwise from raw_data
            if ($request->has('base_to_pickup_time')) {
                $requestForCalculation->merge(['base_to_pickup_time' => $request->base_to_pickup_time]);
            } elseif ($quote && isset($quote->raw_data['base_to_pickup_time'])) {
                $requestForCalculation->merge(['base_to_pickup_time' => $quote->raw_data['base_to_pickup_time']]);
            }
            
            if ($request->has('dropof_to_base')) {
                $requestForCalculation->merge(['dropof_to_base' => $request->dropof_to_base]);
            } elseif ($quote && isset($quote->raw_data['dropof_to_base'])) {
                $requestForCalculation->merge(['dropof_to_base' => $quote->raw_data['dropof_to_base']]);
            }
            
            if ($request->has('pickup_to_dropof')) {
                $requestForCalculation->merge(['pickup_to_dropof' => $request->pickup_to_dropof]);
            } elseif ($quote && isset($quote->raw_data['pickup_to_dropof'])) {
                $requestForCalculation->merge(['pickup_to_dropof' => $quote->raw_data['pickup_to_dropof']]);
            }
            
            // Required truck cubic meter: use from request if available, otherwise from quote
            if ($request->has('required_truck_cubic_meter') && $request->filled('required_truck_cubic_meter')) {
                $requestForCalculation->merge(['required_truck_cubic_meter' => $request->required_truck_cubic_meter]);
            } elseif ($quote && $quote->calculation_type === 'truck_only' && $quote->total_cubic_meters) {
                $requestForCalculation->merge(['required_truck_cubic_meter' => $quote->total_cubic_meters]);
            }
            
            // Calculate quote using trait - this will recalculate all costs including storage and assemble/disassemble
            $quoteData = $this->calculateQuote($booking, null, $additionalTimeMinutes, $requestForCalculation);
            
            if ($quoteData) {
                $quoteData['uuid'] = $uuid;
                // Set user_id from booking (should always be set by now)
                $quoteData['user_id'] = $booking->user_id;
                
                // Ensure total_cost is always included and properly calculated
                // Always recalculate to ensure storage and assemble costs are included
                $totalCost = 0;
                if (isset($quoteData['movers_cost'])) {
                    $totalCost += (float) $quoteData['movers_cost'];
                }
                // Check for callout_fee in response or raw_data
                $calloutFee = 0;
                if (isset($quoteData['callout_fee'])) {
                    $calloutFee = (float) $quoteData['callout_fee'];
                } elseif (isset($quoteData['raw_data']['breakdown']['callout_fee'])) {
                    $calloutFee = (float) $quoteData['raw_data']['breakdown']['callout_fee'];
                }
                $totalCost += $calloutFee;
                if (isset($quoteData['assemble_disassemble_cost'])) {
                    $totalCost += (float) $quoteData['assemble_disassemble_cost'];
                }
                if (isset($quoteData['storage_items_cost'])) {
                    $totalCost += (float) $quoteData['storage_items_cost'];
                }
                // Always set total_cost to ensure it includes all costs
                $quoteData['total_cost'] = round($totalCost, 2);

                // Get existing discount information from quote if it exists
                $existingDiscount = null;
                $existingDiscountType = null;
                $applyOnRemaining = false;
                if ($quote) {
                    $existingRawData = $quote->raw_data ?? [];
                    if (isset($existingRawData['discount']) && $existingRawData['discount'] > 0) {
                        $existingDiscount = (float) $existingRawData['discount'];
                        $existingDiscountType = $existingRawData['discount_type'] ?? 'percentage';
                        $applyOnRemaining = isset($existingRawData['breakdown']['discount_on_remaining']) 
                            && $existingRawData['breakdown']['discount_on_remaining'];
                    }
                }

                // Apply discount if existing discount exists (reapply after recalculation)
                if ($quoteData['total_cost'] > 0 && $existingDiscount !== null && $existingDiscount > 0) {
                    $discountValue = $existingDiscount;
                    $discountType = $existingDiscountType;
                    $depositAmount = (float) ($quote->deposit_amount ?? 0);
                    
                    // Determine if discount should be on remaining amount
                    $willApplyOnRemaining = ($quote && $quote->deposit_payment_status === 'succeeded' 
                        && $quote->remaining_amount > 0 && $applyOnRemaining);
                    
                    if ($willApplyOnRemaining) {
                        // Apply discount on remaining amount (after deposit payment)
                        // Calculate original remaining: new total cost - deposit
                        $newOriginalTotalCost = $quoteData['total_cost'];
                        $newOriginalRemaining = $newOriginalTotalCost - $depositAmount;
                        
                        // Calculate discount amount on remaining
                        $discountAmount = 0;
                        if ($discountType === 'percentage') {
                            $discountAmount = ($newOriginalRemaining * $discountValue) / 100;
                        } else {
                            // Fixed amount - cannot exceed remaining amount
                            $discountAmount = min($discountValue, $newOriginalRemaining);
                        }
                        
                        // Calculate new remaining amount
                        $newRemainingAmount = round($newOriginalRemaining - $discountAmount, 2);
                        
                        // Calculate new total cost: deposit + discounted remaining
                        $quoteData['total_cost'] = round($depositAmount + $newRemainingAmount, 2);
                        
                        // Update raw_data
                        if (!isset($quoteData['raw_data']['breakdown'])) {
                            $quoteData['raw_data']['breakdown'] = [];
                        }
                        $quoteData['raw_data']['original_total_cost'] = $newOriginalTotalCost;
                        $quoteData['raw_data']['original_remaining_amount'] = $newOriginalRemaining;
                        $quoteData['raw_data']['discount'] = $discountValue;
                        $quoteData['raw_data']['discount_type'] = $discountType;
                        $quoteData['raw_data']['discount_on_remaining'] = $discountValue;
                        $quoteData['raw_data']['breakdown']['discount_amount'] = round($discountAmount, 2);
                        $quoteData['raw_data']['breakdown']['discount_on_remaining'] = true;
                        if ($discountType === 'percentage') {
                            $quoteData['raw_data']['breakdown']['discount_percent'] = $discountValue;
                        } else {
                            unset($quoteData['raw_data']['breakdown']['discount_percent']);
                        }
                    } else {
                        // Apply discount on total cost (original behavior)
                        $originalTotalCost = $quoteData['total_cost'];
                        $depositAmount = (float) ($quote->deposit_amount ?? 0);
                        $originalRemainingAmount = $originalTotalCost - $depositAmount;
                        
                        // Calculate discount amount
                        $discountAmount = 0;
                        if ($discountType === 'percentage') {
                            if ($discountValue <= 100) {
                                $discountAmount = ($originalTotalCost * $discountValue) / 100;
                            }
                        } else {
                            // Fixed amount - cannot exceed total cost
                            $discountAmount = min($discountValue, $originalTotalCost);
                        }
                        
                        if ($discountAmount > 0) {
                            $quoteData['total_cost'] = round($originalTotalCost - $discountAmount, 2);
                            
                            // Store discount information in breakdown
                            if (!isset($quoteData['raw_data']['breakdown'])) {
                                $quoteData['raw_data']['breakdown'] = [];
                            }
                            $quoteData['raw_data']['original_total_cost'] = $originalTotalCost;
                            $quoteData['raw_data']['original_remaining_amount'] = max(0, round($originalRemainingAmount, 2));
                            $quoteData['raw_data']['discount'] = $discountValue;
                            $quoteData['raw_data']['discount_type'] = $discountType;
                            unset($quoteData['raw_data']['discount_on_remaining']);
                            $quoteData['raw_data']['breakdown']['discount_amount'] = round($discountAmount, 2);
                            unset($quoteData['raw_data']['breakdown']['discount_on_remaining']);
                            if ($discountType === 'percentage') {
                                $quoteData['raw_data']['breakdown']['discount_percent'] = $discountValue;
                            } else {
                                unset($quoteData['raw_data']['breakdown']['discount_percent']);
                            }
                        }
                    }
                } else {
                    // No discount to apply, but still update original_remaining_amount if quote exists
                    // This ensures original_remaining_amount is updated when items are edited/removed
                    if ($quote) {
                        $depositAmount = (float) ($quote->deposit_amount ?? 0);
                        $originalTotalCost = $quoteData['total_cost'];
                        $originalRemainingAmount = $originalTotalCost - $depositAmount;
                        
                        // Initialize raw_data if not set
                        if (!isset($quoteData['raw_data'])) {
                            $quoteData['raw_data'] = [];
                        }
                        if (!isset($quoteData['raw_data']['breakdown'])) {
                            $quoteData['raw_data']['breakdown'] = [];
                        }
                        
                        // Update original_total_cost and original_remaining_amount
                        // Only update if they don't exist or if there's no discount (to preserve discount context)
                        $hasDiscount = isset($quote->raw_data['discount']) && $quote->raw_data['discount'] > 0;
                        if (!$hasDiscount) {
                            // No discount exists, so update original values
                            $quoteData['raw_data']['original_total_cost'] = $originalTotalCost;
                            $quoteData['raw_data']['original_remaining_amount'] = max(0, round($originalRemainingAmount, 2));
                        } else {
                            // Discount exists, but we still want to update original_remaining_amount if it's not set
                            // or if the total cost changed significantly (items were edited)
                            if (!isset($quoteData['raw_data']['original_total_cost'])) {
                                $quoteData['raw_data']['original_total_cost'] = $originalTotalCost;
                            }
                            if (!isset($quoteData['raw_data']['original_remaining_amount'])) {
                                $quoteData['raw_data']['original_remaining_amount'] = max(0, round($originalRemainingAmount, 2));
                            }
                        }
                    }
                }
                
                if ($quote) {
                    // Update existing quote, but preserve quote_number, order_number, and status if they exist
                    if (!$quote->quote_number) {
                        $quoteData['quote_number'] = Quote::generateQuoteNumber();
                    }
                    // Preserve existing status if not being updated
                    if (!$request->has('status')) {
                        unset($quoteData['status']);
                    }
                    
                    // Handle extra_movers: if explicitly in request (even if 0), use calculated value; otherwise preserve existing
                    if ($request->has('extra_movers')) {
                        // extra_movers is in request - use calculated value (will be 0 if sent as 0)
                        // Ensure extra_movers is set in quoteData at top level and in raw_data
                        $requestExtraMovers = (int)$request->input('extra_movers');
                        $calculatedExtraMovers = isset($quoteData['extra_movers']) ? $quoteData['extra_movers'] : $requestExtraMovers;
                        
                        // Set at top level
                        $quoteData['extra_movers'] = max(0, min(3, $calculatedExtraMovers));
                        $quoteData['extra_movers_cost'] = isset($quoteData['extra_movers_cost']) ? $quoteData['extra_movers_cost'] : 0;
                        
                        // Ensure it's in raw_data (should already be there from calculation, but ensure it)
                        if (isset($quoteData['raw_data'])) {
                            $quoteData['raw_data']['extra_movers'] = $quoteData['extra_movers'];
                            $quoteData['raw_data']['extra_movers_cost'] = $quoteData['extra_movers_cost'];
                        }
                    } else {
                        // extra_movers not in request - preserve existing value
                        if (isset($quote->raw_data['extra_movers']) && !isset($quoteData['raw_data']['extra_movers'])) {
                            $quoteData['raw_data']['extra_movers'] = $quote->raw_data['extra_movers'];
                        }
                        // Also preserve extra_movers_cost if it exists
                        if (isset($quote->raw_data['extra_movers_cost']) && !isset($quoteData['raw_data']['extra_movers_cost'])) {
                            $quoteData['raw_data']['extra_movers_cost'] = $quote->raw_data['extra_movers_cost'];
                        }
                        // Preserve top-level fields if they exist
                        if (isset($quote->extra_movers) && !isset($quoteData['extra_movers'])) {
                            $quoteData['extra_movers'] = $quote->extra_movers;
                        }
                        if (isset($quote->extra_movers_cost) && !isset($quoteData['extra_movers_cost'])) {
                            $quoteData['extra_movers_cost'] = $quote->extra_movers_cost;
                        }
                    }
                    
                    // Ensure user_id is set from booking
                    if (!$quoteData['user_id'] && $booking->user_id) {
                        $quoteData['user_id'] = $booking->user_id;
                    }
                    
                    // Ensure storage_items_cost and assemble_disassemble_cost are always in breakdown
                    // Always set them from quoteData to ensure they match the calculated values
                    if (isset($quoteData['raw_data']) && is_array($quoteData['raw_data'])) {
                        if (!isset($quoteData['raw_data']['breakdown'])) {
                            $quoteData['raw_data']['breakdown'] = [];
                        }
                        $quoteData['raw_data']['breakdown']['storage_items_cost'] = round($quoteData['storage_items_cost'] ?? 0, 2);
                        $quoteData['raw_data']['breakdown']['assemble_disassemble_cost'] = round($quoteData['assemble_disassemble_cost'] ?? 0, 2);
                    }
                    
                    // Explicitly update total_cost to ensure it's saved
                    $quote->update($quoteData);
                    $quote->refresh();
                    
                    // Update remaining_amount - calculate from discounted total_cost
                    $depositAmount = (float) ($quote->deposit_amount ?? 0);
                    // Calculate remaining amount: discounted total_cost - deposit_amount
                    // If discount was applied on remaining, total_cost already includes deposit + discounted remaining
                    // So remaining_amount = total_cost - deposit_amount will give the correct discounted remaining
                    $discountedTotalCost = (float) $quoteData['total_cost'];
                    $quote->remaining_amount = max(0, round($discountedTotalCost - $depositAmount, 2));
                    $quote->save();
                } else {
                    // Create new quote with unique quote_number
                    $quoteData['quote_number'] = Quote::generateQuoteNumber();
                    // Ensure user_id is set from booking
                    if (!$quoteData['user_id'] && $booking->user_id) {
                        $quoteData['user_id'] = $booking->user_id;
                    }
                    $quote = Quote::create($quoteData);
                }
            }
        }
        
        $result['booking'] = $booking->load(['bookingItems.entity', 'quote']);
        $result['uuid'] = $uuid;
        
        return $this->message('Booking saved successfully');
    }

    public function getByUuid(string $uuid)
    {
        $booking = Booking::where('uuid', $uuid)
            ->with(['bookingItems.entity' ,'quote'])
            ->first();
        if (!$booking) {
            return $this->error('Booking not found', 404);
        }

        // Get booking with truck pricing accessor
        $response = $booking->toArray();
        $response['truck_pricing'] = $booking->truck_pricing;
        
        // Remove the truck relationship if present
        unset($response['truck']);
        
        return $this->success($response, 'Booking fetched successfully');
    }

    /**
     * Generate random password
     */
    private function generateRandomPassword($length = 12)
    {
        $uppercase = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $lowercase = 'abcdefghijklmnopqrstuvwxyz';
        $numbers = '0123456789';
        $special = '@#$%&*';

        $allChars = $uppercase . $lowercase . $numbers . $special;

        $password = '';
        $password .= $uppercase[rand(0, strlen($uppercase) - 1)];
        $password .= $lowercase[rand(0, strlen($lowercase) - 1)];
        $password .= $numbers[rand(0, strlen($numbers) - 1)];
        $password .= $special[rand(0, strlen($special) - 1)];

        for ($i = 4; $i < $length; $i++) {
            $password .= $allChars[rand(0, strlen($allChars) - 1)];
        }

        return str_shuffle($password);
    }
}
