<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\BookingItem;
use App\Models\ChildCategory;
use App\Models\AssembleDisassembleCategory;
use Illuminate\Http\Request;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\QuoteCalculationTrait;
use App\Models\Booking;
use App\Models\Category;
use App\Models\Quote;
use App\Models\StorageItem;

class BookingItemController extends Controller
{
    use JsonResponseTrait, QuoteCalculationTrait;

    /**
     * Store booking items (create or update) by UUID - handles multiple items
     */
    public function store(Request $request)
    {
        $uuid = $request->uuid;
        if (!$uuid) {
            return $this->error('UUID is required', 422);
        }
        $createdRegularItems = [];
        $createdAssembleItems = [];
        $createdStorageItems = [];
        $result = [];
        $payloadRegularItemIds = [];
        $payloadAssembleItemIds = [];
        $payloadStorageItemIds = [];
        
        if ($request->has('items') && is_array($request->items)) {
            foreach ($request->items as $item) {
                if (!isset($item['category_id'])) {
                    continue;
                }
                $categoryId = $item['category_id'];
                $payloadRegularItemIds[] = $categoryId;
                
                $bookingItem = BookingItem::where('uuid', $uuid)
                    ->where('entity_type', ChildCategory::class)
                    ->where('entity_id', $categoryId)
                    ->first();
                $itemData = [
                    'uuid' => $uuid,
                    'entity_type' => ChildCategory::class,
                    'entity_id' => $categoryId,
                    'is_assemble_disassemble' => false,
                    'quantity' => $item['quantity'] ?? 1,
                ];
                if ($bookingItem) {
                    $fillableFields = [
                        'entity_type',
                        'entity_id',
                        'is_assemble_disassemble',
                        'quantity'
                    ];
                    foreach ($fillableFields as $field) {
                        if (!array_key_exists($field, $itemData)) {
                            $itemData[$field] = null;
                        }
                    }
                    $bookingItem->update($itemData);
                    $bookingItem->refresh();
                } else {
                    $bookingItem = BookingItem::create($itemData);
                }
                $createdRegularItems[] = $bookingItem->load(['entity']);
            }
        }
        
        // Delete regular items that are not in the payload (only if 'items' key exists in request)
        if ($request->has('items')) {
            $query = BookingItem::where('uuid', $uuid)
                ->where('entity_type', ChildCategory::class)
                ->where('is_assemble_disassemble', false);
            
            // If payload has items, delete those not in the list; if empty array, delete all
            if (!empty($payloadRegularItemIds)) {
                $query->whereNotIn('entity_id', $payloadRegularItemIds);
            }
            // If empty array, the query will delete all regular items (no whereNotIn condition)
            $query->delete();
        }

        // Handle assemble/disassemble items
        if ($request->has('assemble_items') && is_array($request->assemble_items)) {
            foreach ($request->assemble_items as $item) {
                if (!isset($item['assemble_disassemble_category_id'])) {
                    continue;
                }
                $assembleCategoryId = $item['assemble_disassemble_category_id'];
                $payloadAssembleItemIds[] = $assembleCategoryId;
                
                $bookingItem = BookingItem::where('uuid', $uuid)
                    ->where('entity_type', AssembleDisassembleCategory::class)
                    ->where('entity_id', $assembleCategoryId)
                    ->first();
                $itemData = [
                    'uuid' => $uuid,
                    'entity_type' => AssembleDisassembleCategory::class,
                    'entity_id' => $assembleCategoryId,
                    'is_assemble_disassemble' => true,
                    'assemble_disassemble_type' => $item['assemble_disassemble_type'] ?? 'null',
                    'quantity' => $item['quantity'] ?? 1,
                ];
                if ($bookingItem) {
                    // If updating, set missing fields to null
                    $fillableFields = [
                        'entity_type',
                        'entity_id',
                        'is_assemble_disassemble',
                        'assemble_disassemble_type',
                        'quantity'
                    ];
                    foreach ($fillableFields as $field) {
                        if (!array_key_exists($field, $itemData)) {
                            $itemData[$field] = null;
                        }
                    }
                    $bookingItem->update($itemData);
                    $bookingItem->refresh();
                } else {
                    $bookingItem = BookingItem::create($itemData);
                }
                
                // Format assemble items - only include selected pricing, not all entity fields
                $bookingItem->load(['entity']);
                $assembleItem = [
                    'id' => $bookingItem->id,
                    'uuid' => $bookingItem->uuid,
                    'entity_type' => $bookingItem->entity_type,
                    'entity_id' => $bookingItem->entity_id,
                    'quantity' => $bookingItem->quantity,
                    'is_assemble_disassemble' => $bookingItem->is_assemble_disassemble,
                    'assemble_disassemble_type' => $bookingItem->assemble_disassemble_type,
                    'created_at' => $bookingItem->created_at,
                    'updated_at' => $bookingItem->updated_at,
                    'entity' => [
                        'id' => $bookingItem->entity->id ?? null,
                        'name' => $bookingItem->entity->name ?? null,
                        'description' => $bookingItem->entity->description ?? null,
                        'image' => $bookingItem->entity->image ?? null,
                    ],
                    'selected_pricing' => $bookingItem->selected_pricing,
                ];
                $createdAssembleItems[] = $assembleItem;
            }
        }

        if ($request->has('storage_items') && is_array($request->storage_items)) {
            foreach ($request->storage_items as $item) {
                if (!isset($item['storage_item_id'])) {
                    continue;
                }

                $storageItemId = $item['storage_item_id'];
                $payloadStorageItemIds[] = $storageItemId;

                $bookingItem = BookingItem::where('uuid', $uuid)
                    ->where('entity_type', StorageItem::class)
                    ->where('entity_id', $storageItemId)
                    ->first();

                $itemData = [
                    'uuid' => $uuid,
                    'entity_type' => StorageItem::class,
                    'entity_id' => $storageItemId,
                    'is_assemble_disassemble' => false,
                    'assemble_disassemble_type' => null,
                    'quantity' => $item['quantity'] ?? 1,
                ];

                if ($bookingItem) {
                    $fillableFields = [
                        'entity_type',
                        'entity_id',
                        'is_assemble_disassemble',
                        'assemble_disassemble_type',
                        'quantity',
                    ];

                    foreach ($fillableFields as $field) {
                        if (!array_key_exists($field, $itemData)) {
                            $itemData[$field] = null;
                        }
                    }

                    $bookingItem->update($itemData);
                    $bookingItem->refresh();
                } else {
                    $bookingItem = BookingItem::create($itemData);
                }

                $createdStorageItems[] = $bookingItem->load(['entity']);
            }
        }
        
        // Delete assemble/disassemble items that are not in the payload
        // If 'assemble_items' key exists in request, sync based on payload
        // If 'assemble_items' key doesn't exist but 'items' exists, delete all assemble items
        if ($request->has('assemble_items')) {
            $queryAssemble = BookingItem::where('uuid', $uuid)
                ->where('entity_type', AssembleDisassembleCategory::class)
                ->where('is_assemble_disassemble', true);
            
            // If payload has items, delete those not in the list; if empty array, delete all
            if (!empty($payloadAssembleItemIds)) {
                $queryAssemble->whereNotIn('entity_id', $payloadAssembleItemIds);
            }
            // If empty array, the query will delete all assemble items (no whereNotIn condition)
            $deletedAssembleCount = $queryAssemble->delete();
        } elseif ($request->has('items')) {
            // If only 'items' is provided (no 'assemble_items'), delete all assemble items
            // This ensures the database matches exactly what's in the payload
            BookingItem::where('uuid', $uuid)
                ->where('entity_type', AssembleDisassembleCategory::class)
                ->where('is_assemble_disassemble', true)
                ->delete();
        }

        if ($request->has('storage_items')) {
            $storageDeleteQuery = BookingItem::where('uuid', $uuid)
                ->where('entity_type', StorageItem::class)
                ->where('is_assemble_disassemble', false);

            if (!empty($payloadStorageItemIds)) {
                $storageDeleteQuery->whereNotIn('entity_id', $payloadStorageItemIds);
            }

            $storageDeleteQuery->delete();
        } elseif ($request->has('items')) {
            BookingItem::where('uuid', $uuid)
                ->where('entity_type', StorageItem::class)
                ->where('is_assemble_disassemble', false)
                ->delete();
        }

        $result['booking_items'] = array_merge($createdRegularItems, $createdStorageItems);
        $result['items'] = $createdRegularItems;
        $result['assemble_items'] = $createdAssembleItems;
        $result['storage_items'] = $createdStorageItems;
        $result['uuid'] = $uuid;
        
        // Recalculate quote after items are added/removed
        $booking = Booking::where('uuid', $uuid)->first();
        if ($booking && $booking->truck_id) {
            // Refresh booking to get latest items
            $booking->refresh();
            // Clear relationship cache to ensure fresh data
            $booking->unsetRelation('bookingItems');
            $booking->load(['bookingItems.entity', 'truck']);
            
            // Get existing quote to preserve distance and other fields
            $quote = Quote::where('uuid', $uuid)->first();
            $distanceKm = $quote ? $quote->distance_km : null;
            
            // Create a request object with job_type and other calculation parameters if they exist in the quote
            $requestForCalculation = new Request();
            if ($quote && isset($quote->raw_data['job_type'])) {
                $requestForCalculation->merge(['job_type' => $quote->raw_data['job_type']]);
            }
            if ($quote && isset($quote->raw_data['extra_movers'])) {
                $requestForCalculation->merge(['extra_movers' => $quote->raw_data['extra_movers']]);
            }
            // Add callout fee times if they exist in raw_data
            if ($quote && isset($quote->raw_data['base_to_pickup_time'])) {
                $requestForCalculation->merge([
                    'base_to_pickup_time' => $quote->raw_data['base_to_pickup_time'],
                ]);
            }
            if ($quote && isset($quote->raw_data['dropof_to_base'])) {
                $requestForCalculation->merge([
                    'dropof_to_base' => $quote->raw_data['dropof_to_base'],
                ]);
            }
            if ($quote && isset($quote->raw_data['pickup_to_dropof'])) {
                $requestForCalculation->merge([
                    'pickup_to_dropof' => $quote->raw_data['pickup_to_dropof'],
                ]);
            }
            
            // Calculate quote using trait - this will recalculate all costs including storage and assemble/disassemble
            $quoteData = $this->calculateQuote($booking, $distanceKm, 0, $requestForCalculation);
            
            if ($quoteData) {
                $quoteData['uuid'] = $uuid;
                $quoteData['user_id'] = $booking->user_id;
                $quoteData['distance_km'] = $distanceKm;
                
                // Ensure total_cost is always included and properly calculated
                if (!isset($quoteData['total_cost']) || $quoteData['total_cost'] === null) {
                    // If total_cost is missing, calculate it from the breakdown
                    $totalCost = 0;
                    if (isset($quoteData['movers_cost'])) {
                        $totalCost += (float) $quoteData['movers_cost'];
                    }
                    // Check for callout_fee in response or raw_data
                    $calloutFee = 0;
                    if (isset($quoteData['callout_fee'])) {
                        $calloutFee = (float) $quoteData['callout_fee'];
                    } elseif (isset($quoteData['raw_data']['breakdown']['callout_fee'])) {
                        $calloutFee = (float) $quoteData['raw_data']['breakdown']['callout_fee'];
                    }
                    $totalCost += $calloutFee;
                    if (isset($quoteData['assemble_disassemble_cost'])) {
                        $totalCost += (float) $quoteData['assemble_disassemble_cost'];
                    }
                    if (isset($quoteData['storage_items_cost'])) {
                        $totalCost += (float) $quoteData['storage_items_cost'];
                    }
                    $quoteData['total_cost'] = round($totalCost, 2);
                } else {
                    $quoteData['total_cost'] = round((float) $quoteData['total_cost'], 2);
                }
                
                // Ensure user_id is always set
                if (!isset($quoteData['user_id']) || !$quoteData['user_id']) {
                    if ($booking && $booking->user_id) {
                        $quoteData['user_id'] = $booking->user_id;
                    } elseif ($quote && $quote->user_id) {
                        // Preserve existing user_id if quote exists
                        $quoteData['user_id'] = $quote->user_id;
                    } elseif ($quote && $quote->booking_id) {
                        // Try to get from booking relationship
                        $bookingForUser = Booking::find($quote->booking_id);
                        if ($bookingForUser && $bookingForUser->user_id) {
                            $quoteData['user_id'] = $bookingForUser->user_id;
                        }
                    }
                }

                if ($quote) {
                    // Update existing quote, but preserve quote_number, order_number, and status
                    if (!$quote->quote_number) {
                        $quoteData['quote_number'] = Quote::generateQuoteNumber();
                    }
                    // Preserve existing status
                    unset($quoteData['status']);
                    // Preserve distance_km
                    $quoteData['distance_km'] = $distanceKm;
                    
                    // Ensure user_id is preserved if not in quoteData
                    if (!isset($quoteData['user_id']) && $quote->user_id) {
                        $quoteData['user_id'] = $quote->user_id;
                    }
                    
                    // Update quote with recalculated costs
                    $quote->update($quoteData);
                    $quote->refresh();
                    
                    // Final check: if user_id is still null, try to set it from booking
                    if (!$quote->user_id && $booking && $booking->user_id) {
                        $quote->user_id = $booking->user_id;
                        $quote->save();
                    }
                } else {
                    // Create new quote with unique quote_number
                    $quoteData['quote_number'] = Quote::generateQuoteNumber();
                    // Ensure user_id is set before creating
                    if (!isset($quoteData['user_id']) || !$quoteData['user_id']) {
                        if ($booking && $booking->user_id) {
                            $quoteData['user_id'] = $booking->user_id;
                        }
                    }
                    $quote = Quote::create($quoteData);
                    
                    // Final check: if user_id is still null after creation, try to set it
                    if (!$quote->user_id && $booking && $booking->user_id) {
                        $quote->user_id = $booking->user_id;
                        $quote->save();
                    }
                }
                
                // Add quote to result
                $result['quote'] = $quote->load(['user']);
            }
        }
        return $this->success($result, 'Booking items saved successfully');
    }

    /**
     * Get all booking items by UUID - returns both category items and assemble items
     */
    public function getByUuid(string $uuid)
    {
        $bookingItems = BookingItem::where('uuid', $uuid)
            ->with(['entity'])
            ->get();
        if ($bookingItems->isEmpty()) {
            return $this->error('Booking items not found', 404);
        }
        // Separate items by type
        $result = [
            'items' => [],
            'assemble_items' => [],
            'storage_items' => [],
        ];
        foreach ($bookingItems as $item) {
            if ($item->is_assemble_disassemble) {
                // Format assemble items - only include selected pricing, not all entity fields
                $assembleItem = [
                    'id' => $item->id,
                    'uuid' => $item->uuid,
                    'entity_type' => $item->entity_type,
                    'entity_id' => $item->entity_id,
                    'quantity' => $item->quantity,
                    'is_assemble_disassemble' => $item->is_assemble_disassemble,
                    'assemble_disassemble_type' => $item->assemble_disassemble_type,
                    'created_at' => $item->created_at,
                    'updated_at' => $item->updated_at,
                    'entity' => [
                        'id' => $item->entity->id ?? null,
                        'name' => $item->entity->name ?? null,
                        'description' => $item->entity->description ?? null,
                        'image' => $item->entity->image ?? null,
                    ],
                    'selected_pricing' => $item->selected_pricing,
                ];
                $result['assemble_items'][] = $assembleItem;
            } elseif ($item->entity_type === StorageItem::class) {
                $result['storage_items'][] = $item;
            } else {
                $result['items'][] = $item;
            }
        }
        $response = [
            'uuid' => $uuid,
            'items' => $result['items'],
            'assemble_items' => $result['assemble_items'],
            'storage_items' => $result['storage_items'],
        ];
        return $this->success($response, 'Booking items fetched successfully');
    }

    public function getAllItems(Request $request)
    {
        $search = $request->get('search');
        $parentCategoriesQuery = Category::active()->ordered()->with(['childCategories' => function($query) {
            $query->active()->ordered();
        }]);
        $assembleCategoriesQuery = AssembleDisassembleCategory::active()->ordered();
        $storageItemsQuery = StorageItem::active()->ordered();
        
        if ($search) {
            $parentCategoriesQuery->where('title', 'like', "%{$search}%");
            $assembleCategoriesQuery->where('name', 'like', "%{$search}%");
            $storageItemsQuery->where('title', 'like', "%{$search}%");
        }
        
        $parentCategories = $parentCategoriesQuery->get();
        
        // If search is provided, filter child categories within parent categories
        if ($search) {
            $parentCategories->each(function($parent) use ($search) {
                $parent->childCategories = $parent->childCategories->filter(function($child) use ($search) {
                    return stripos($child->title, $search) !== false;
                })->values();
            });
        }
        
        $assembleDisassembleCategories = $assembleCategoriesQuery->get();
        $storageItems = $storageItemsQuery->get();
        $frequentItems = $this->getFrequentItems();
        
        // Structure parent categories with nested child categories
        $parentCategoriesWithChildren = $parentCategories->map(function($parent) {
            return [
                'id' => $parent->id,
                'title' => $parent->title,
                'sku' => $parent->sku,
                'description' => $parent->description,
                'image' => $parent->image,
                'is_active' => $parent->is_active,
                'sort_order' => $parent->sort_order,
                'created_at' => $parent->created_at,
                'updated_at' => $parent->updated_at,
                'child_categories' => $parent->childCategories->map(function($child) {
                    return [
                        'id' => $child->id,
                        'main_parent_category' => $child->main_parent_category,
                        'title' => $child->title,
                        'volumn' => $child->volumn,
                        'description' => $child->description,
                        'image' => $child->image,
                        'is_active' => $child->is_active,
                        'sort_order' => $child->sort_order,
                        'created_at' => $child->created_at,
                        'updated_at' => $child->updated_at,
                    ];
                })->values()
            ];
        });
        
        $response = [
            'frequent_items' => $frequentItems,
            'parent_categories' => $parentCategoriesWithChildren,
            'assemble_disassemble_categories' => $assembleDisassembleCategories,
            'storage_items' => $storageItems
        ];
        return $this->success($response, 'All items fetched successfully');
    }
    
    /**
     * Get frequently used items from booking items mixed with top child categories
     * Always shows at least 6 items if available:
     * - Shows booking items (if any, even if less than 5)
     * - Fills remaining slots with top child categories by sort order
     * - Minimum 6 items total if available in database
     * Returns parent categories with their frequent child categories nested inside
     */
    private function getFrequentItems($limit = 10)
    {
        $frequentChildCategories = [];
        $minItems = 6; // Minimum items to show
        $targetLimit = max($limit, $minItems); // Use the larger of limit or minItems
        
        // Step 1: Get ALL booking items (no limit) - even if less than 5, we'll show them and mix with categories
        $bookingItemCounts = BookingItem::selectRaw('booking_items.entity_id, COUNT(*) as usage_count')
            ->where('booking_items.entity_type', ChildCategory::class)
            ->where('booking_items.is_assemble_disassemble', false)
            ->join('child_categories', 'booking_items.entity_id', '=', 'child_categories.id')
            ->where('child_categories.is_active', true)
            ->groupBy('booking_items.entity_id')
            ->orderBy('usage_count', 'desc')
            ->get(); // No limit here - get all booking items to mix with categories
        
        $bookingItemIds = [];
        if ($bookingItemCounts->count() > 0) {
            // Load all booking item entities
            $entityIds = $bookingItemCounts->pluck('entity_id')->toArray();
            $childCategories = ChildCategory::with('mainParentCategory')
                ->whereIn('id', $entityIds)
                ->where('is_active', true)
                ->get()
                ->keyBy('id');
            
            // Create a map of usage counts
            $usageCounts = $bookingItemCounts->keyBy('entity_id');
            
            foreach ($bookingItemCounts as $item) {
                $entity = $childCategories->get($item->entity_id);
                
                // Only add if entity exists, is active, and has a valid parent category
                if ($entity && $entity->is_active && $entity->main_parent_category) {
                    $bookingItemIds[] = $entity->id;
                    $frequentChildCategories[] = [
                        'child_category' => [
                            'id' => $entity->id,
                            'main_parent_category' => $entity->main_parent_category,
                            'title' => $entity->title,
                            'description' => $entity->description ?? null,
                            'volumn' => $entity->volumn ?? 1,
                            'usage_count' => (int) $item->usage_count,
                            'is_from_booking' => true // Mark as from booking items
                        ],
                        'parent_category_id' => $entity->main_parent_category
                    ];
                }
            }
        }
        
        // Step 2: Fill remaining slots with top child categories (sorted by sort_order)
        // Always ensure we have at least minItems (6) total if available
        $currentCount = count($frequentChildCategories);
        $remainingSlots = $targetLimit - $currentCount;
        
        if ($remainingSlots > 0) {
            $childCategories = ChildCategory::active()
                ->ordered()
                ->with('mainParentCategory')
                ->whereNotNull('main_parent_category')
                ->when(!empty($bookingItemIds), function($query) use ($bookingItemIds) {
                    // Exclude items already in booking items list
                    return $query->whereNotIn('id', $bookingItemIds);
                })
                ->limit($remainingSlots)
                ->get();
            
            foreach ($childCategories as $category) {
                if ($category->main_parent_category) {
                    $frequentChildCategories[] = [
                        'child_category' => [
                            'id' => $category->id,
                            'main_parent_category' => $category->main_parent_category,
                            'title' => $category->title,
                            'description' => $category->description ?? null,
                            'volumn' => $category->volumn ?? 1,
                            'usage_count' => 0,
                            'is_from_booking' => false // Mark as from top categories
                        ],
                        'parent_category_id' => $category->main_parent_category
                    ];
                }
            }
        }
        
        // Step 3: Group frequent items by parent category and filter out invalid parents
        $parentCategoriesMap = [];
        foreach ($frequentChildCategories as $item) {
            $parentId = $item['parent_category_id'];
            
            // Skip if parent_id is null or invalid
            if (!$parentId) {
                continue;
            }
            
            if (!isset($parentCategoriesMap[$parentId])) {
                $parentCategory = Category::where('id', $parentId)
                    ->where('is_active', true)
                    ->first();
                    
                if ($parentCategory) {
                    $parentCategoriesMap[$parentId] = [
                        'id' => $parentCategory->id,
                        'title' => $parentCategory->title,
                        'sku' => $parentCategory->sku ?? null,
                        'description' => $parentCategory->description ?? null,
                        'image' => $parentCategory->image ?? null,
                        'is_active' => $parentCategory->is_active,
                        'sort_order' => $parentCategory->sort_order ?? 999,
                        'created_at' => $parentCategory->created_at,
                        'updated_at' => $parentCategory->updated_at,
                        'child_categories' => []
                    ];
                } else {
                    // Skip if parent category doesn't exist or is inactive
                    continue;
                }
            }
            
            if (isset($parentCategoriesMap[$parentId])) {
                $parentCategoriesMap[$parentId]['child_categories'][] = $item['child_category'];
            }
        }
        
        // Step 4: Convert to array and sort by parent category sort_order
        $result = collect($parentCategoriesMap)
            ->values()
            ->sortBy('sort_order')
            ->values()
            ->all();
        
        // Return empty array structure if no valid items found (instead of null)
        return $result ?: [];
    }

    /**
     * Add a single booking item (for admin panel)
     */
    public function addItem(Request $request)
    {
        $request->validate([
            'uuid' => 'required|string',
            'entity_type' => 'required|string',
            'entity_id' => 'required|integer',
            'quantity' => 'required|integer|min:1',
            'is_assemble_disassemble' => 'nullable|boolean',
            'assemble_disassemble_type' => 'nullable|string|in:assemble,disassemble,both',
        ]);

        $uuid = $request->uuid;
        $booking = Booking::where('uuid', $uuid)->first();
        
        if (!$booking) {
            return $this->error('Booking not found', 404);
        }

        $entityType = $request->entity_type;
        $entityId = $request->entity_id;
        
        // Validate entity type
        $allowedTypes = [ChildCategory::class, AssembleDisassembleCategory::class, StorageItem::class];
        if (!in_array($entityType, $allowedTypes)) {
            return $this->error('Invalid entity type', 422);
        }

        // Check if entity exists
        $entity = $entityType::find($entityId);
        if (!$entity) {
            return $this->error('Entity not found', 404);
        }

        // Create booking item
        $bookingItem = BookingItem::create([
            'uuid' => $uuid,
            'entity_type' => $entityType,
            'entity_id' => $entityId,
            'quantity' => $request->quantity,
            'is_assemble_disassemble' => $request->is_assemble_disassemble ?? false,
            'assemble_disassemble_type' => $request->assemble_disassemble_type ?? null,
        ]);

        // Recalculate quote
        $quote = Quote::where('uuid', $uuid)->first();
        if ($quote && $booking->truck_id) {
            $booking->refresh();
            $booking->unsetRelation('bookingItems');
            $booking->load(['bookingItems.entity', 'truck']);
            
            $distanceKm = $quote->distance_km;
            $calculationRequest = new Request();
            if ($quote->raw_data && isset($quote->raw_data['job_type'])) {
                $calculationRequest->merge(['job_type' => $quote->raw_data['job_type']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['extra_movers'])) {
                $calculationRequest->merge(['extra_movers' => $quote->raw_data['extra_movers']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['base_to_pickup_time'])) {
                $calculationRequest->merge(['base_to_pickup_time' => $quote->raw_data['base_to_pickup_time']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['dropof_to_base'])) {
                $calculationRequest->merge(['dropof_to_base' => $quote->raw_data['dropof_to_base']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['pickup_to_dropof'])) {
                $calculationRequest->merge(['pickup_to_dropof' => $quote->raw_data['pickup_to_dropof']]);
            }
            
            $quoteData = $this->calculateQuote($booking, $distanceKm, 0, $calculationRequest);
            
            if ($quoteData) {
                $quote->update([
                    'movers_cost' => round($quoteData['movers_cost'] ?? 0, 2),
                    'callout_fee' => round($quoteData['callout_fee'] ?? 0, 2),
                    'total_cost' => round($quoteData['total_cost'] ?? 0, 2),
                    'total_cubic_meters' => round($quoteData['total_cubic_meters'] ?? 0, 2),
                    'raw_data' => $quoteData['raw_data'] ?? [],
                ]);
            }
        }

        return $this->success($bookingItem->load('entity'), 'Booking item added successfully');
    }

    /**
     * Remove a booking item (for admin panel)
     */
    public function removeItem(Request $request, $id)
    {
        $bookingItem = BookingItem::find($id);
        
        if (!$bookingItem) {
            return $this->error('Booking item not found', 404);
        }

        $uuid = $bookingItem->uuid;
        $bookingItem->delete();

        // Recalculate quote
        $booking = Booking::where('uuid', $uuid)->first();
        $quote = Quote::where('uuid', $uuid)->first();
        
        if ($quote && $booking && $booking->truck_id) {
            $booking->refresh();
            $booking->unsetRelation('bookingItems');
            $booking->load(['bookingItems.entity', 'truck']);
            
            $distanceKm = $quote->distance_km;
            $calculationRequest = new Request();
            if ($quote->raw_data && isset($quote->raw_data['job_type'])) {
                $calculationRequest->merge(['job_type' => $quote->raw_data['job_type']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['extra_movers'])) {
                $calculationRequest->merge(['extra_movers' => $quote->raw_data['extra_movers']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['base_to_pickup_time'])) {
                $calculationRequest->merge(['base_to_pickup_time' => $quote->raw_data['base_to_pickup_time']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['dropof_to_base'])) {
                $calculationRequest->merge(['dropof_to_base' => $quote->raw_data['dropof_to_base']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['pickup_to_dropof'])) {
                $calculationRequest->merge(['pickup_to_dropof' => $quote->raw_data['pickup_to_dropof']]);
            }
            
            $quoteData = $this->calculateQuote($booking, $distanceKm, 0, $calculationRequest);
            
            if ($quoteData) {
                $quote->update([
                    'movers_cost' => round($quoteData['movers_cost'] ?? 0, 2),
                    'callout_fee' => round($quoteData['callout_fee'] ?? 0, 2),
                    'total_cost' => round($quoteData['total_cost'] ?? 0, 2),
                    'total_cubic_meters' => round($quoteData['total_cubic_meters'] ?? 0, 2),
                    'raw_data' => $quoteData['raw_data'] ?? [],
                ]);
            }
        }

        return $this->success(null, 'Booking item removed successfully');
    }

    /**
     * Update a booking item (for admin panel)
     */
    public function updateItem(Request $request, $id)
    {
        $bookingItem = BookingItem::find($id);
        
        if (!$bookingItem) {
            return $this->error('Booking item not found', 404);
        }

        $request->validate([
            'quantity' => 'nullable|integer|min:1',
            'is_assemble_disassemble' => 'nullable|boolean',
            'assemble_disassemble_type' => 'nullable|string|in:assemble,disassemble,both',
        ]);

        if ($request->has('quantity')) {
            $bookingItem->quantity = $request->quantity;
        }
        if ($request->has('is_assemble_disassemble')) {
            $bookingItem->is_assemble_disassemble = $request->is_assemble_disassemble;
        }
        if ($request->has('assemble_disassemble_type')) {
            $bookingItem->assemble_disassemble_type = $request->assemble_disassemble_type;
        }
        
        $bookingItem->save();

        // Recalculate quote
        $uuid = $bookingItem->uuid;
        $booking = Booking::where('uuid', $uuid)->first();
        $quote = Quote::where('uuid', $uuid)->first();
        
        if ($quote && $booking && $booking->truck_id) {
            $booking->refresh();
            $booking->unsetRelation('bookingItems');
            $booking->load(['bookingItems.entity', 'truck']);
            
            $distanceKm = $quote->distance_km;
            $calculationRequest = new Request();
            if ($quote->raw_data && isset($quote->raw_data['job_type'])) {
                $calculationRequest->merge(['job_type' => $quote->raw_data['job_type']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['extra_movers'])) {
                $calculationRequest->merge(['extra_movers' => $quote->raw_data['extra_movers']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['base_to_pickup_time'])) {
                $calculationRequest->merge(['base_to_pickup_time' => $quote->raw_data['base_to_pickup_time']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['dropof_to_base'])) {
                $calculationRequest->merge(['dropof_to_base' => $quote->raw_data['dropof_to_base']]);
            }
            if ($quote->raw_data && isset($quote->raw_data['pickup_to_dropof'])) {
                $calculationRequest->merge(['pickup_to_dropof' => $quote->raw_data['pickup_to_dropof']]);
            }
            
            $quoteData = $this->calculateQuote($booking, $distanceKm, 0, $calculationRequest);
            
            if ($quoteData) {
                $quote->update([
                    'movers_cost' => round($quoteData['movers_cost'] ?? 0, 2),
                    'callout_fee' => round($quoteData['callout_fee'] ?? 0, 2),
                    'total_cost' => round($quoteData['total_cost'] ?? 0, 2),
                    'total_cubic_meters' => round($quoteData['total_cubic_meters'] ?? 0, 2),
                    'raw_data' => $quoteData['raw_data'] ?? [],
                ]);
            }
        }

        return $this->success($bookingItem->load('entity'), 'Booking item updated successfully');
    }


    
   
}

