<?php

namespace App\Http\Controllers\API;

use App\Models\Job;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Models\JobApplication;
use App\Http\Controllers\API\Traits\ValidationTrait;
use App\Models\User;

class JobApplicationController extends Controller
{
    use ValidationTrait;

    public function index()
    {
        $jobApplications = Job::active()->valid()->ordered()->get();
        return $this->success($jobApplications, 'Job applications fetched successfully');
    }

    public function getJobByPermalink($permalink)
    {
        $job = Job::active()->valid()->where('permalink', $permalink)->first();
        if (!$job) {
            return $this->error('Job not found', 404);
        }
        return $this->success($job, 'Job fetched successfully');
    }

    public function apply(Request $request, $jobId)
    {
        $job = Job::active()->valid()->find($jobId);
        if (!$job) {
            return $this->message('Job not found', 404);
        }
        if ($job->applications()->where('email', $request->email)->exists()) {
            return $this->error('You have already applied for this job');
        }
        $validator = Validator::make($request->all(), [
            'full_name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'cover_letter' => 'required|string|min:50',
            'cv_file' => 'required|file|mimes:pdf,doc,docx|max:5120', // 5MB max
        ]);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $cvFile = $request->file('cv_file');
        $cvFileName = time() . '_' . $cvFile->getClientOriginalName();
        $directoryPath = public_path('job_applications');
        if (!file_exists($directoryPath)) {
            mkdir($directoryPath, 0755, true);
        }
        $cvFile->move($directoryPath, $cvFileName);
        JobApplication::create([
            'job_id' => $job->id,
            'full_name' => $request->full_name,
            'email' => $request->email,
            'phone' => $request->phone,
            'cover_letter' => $request->cover_letter,
            'cv_file' => $cvFileName,
            'status' => 'pending',
        ]);
        return $this->message('Your application has been submitted successfully! We will review it and get back to you soon.');
    }

    public function getUserJobApplications()
    {
        $user = auth()->user();
        $jobApplications = JobApplication::where('user_id', $user->id)->with('job')->get();
        if ($jobApplications->isEmpty()) {
            return $this->message('No job applications found', 404);
        }
        return $this->success($jobApplications, 'Job applications fetched successfully');
    }
}
