<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\PropertyDetail;
use Illuminate\Http\Request;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\ValidationTrait;

class PropertyDetailController extends Controller
{
    use JsonResponseTrait, ValidationTrait;

    /**
     * Store property details (create or update)
     */
    public function store(Request $request)
    {
        $uuid = $request->uuid;
        if (!$uuid) {
            return $this->error('UUID is required', 422);
        }
        $result = [];
        if ($request->has('pickup_details') && !empty($request->pickup_details)) {
            $pickupProperty = Property::where('uuid', $uuid)
                ->where('type', 'pick_up')
                ->first();
            if (!$pickupProperty) {
                return $this->error('Pickup property not found for this UUID', 404);
            }
            $pickupDetails = PropertyDetail::where('property_id', $pickupProperty->id)
                ->where('property_type', 'pick_up')
                ->first();
            $fillableFields = ['property_category', 'commercial_property_type', 'flights_of_stairs', 'access_required', 'lift_booking_info', 
                               'internal_levels', 'levels_in_house', 'truck_parking', 
                               'storage_name', 'storage_units', 'walk_distance', 'comments'];
            
            $requestData = $request->pickup_details;
            $pickupDetailsData = [];            
            // Only include fields that exist in the payload
            // If field is null in payload, it will be set to null (nullifying existing values)
            // If field has a value, save that value
            // If field is not in payload, don't include it (keeps existing value)
            foreach ($fillableFields as $field) {
                if (array_key_exists($field, $requestData)) {
                    $pickupDetailsData[$field] = $requestData[$field];
                }
            }
            
            // Always set these fields
            $pickupDetailsData['property_id'] = $pickupProperty->id;
            $pickupDetailsData['property_type'] = 'pick_up';
            
            if ($pickupDetails) {
                $pickupDetails->update($pickupDetailsData);
                $pickupDetails->refresh();
            } else {
                $pickupDetails = PropertyDetail::create($pickupDetailsData);
            }
            $result['pickup_details'] = $pickupDetails;
        }
        if ($request->has('dropoff_details') && !empty($request->dropoff_details)) {
            $dropoffProperty = Property::where('uuid', $uuid)
                ->where('type', 'drop_off')
                ->first();
            if (!$dropoffProperty) {
                return $this->error('Dropoff property not found for this UUID', 404);
            }
            $dropoffDetails = PropertyDetail::where('property_id', $dropoffProperty->id)
                ->where('property_type', 'drop_off')
                ->first();
            // Get all fillable fields that can be updated (excluding property_id and property_type)
            $fillableFields = ['property_category', 'commercial_property_type', 'flights_of_stairs', 'access_required', 'lift_booking_info', 
                               'internal_levels', 'levels_in_house', 'truck_parking', 
                               'storage_name', 'storage_units', 'walk_distance', 'comments'];
            
            $requestData = $request->dropoff_details;
            $dropoffDetailsData = [];
            foreach ($fillableFields as $field) {
                if (array_key_exists($field, $requestData)) {
                    $dropoffDetailsData[$field] = $requestData[$field];
                }
            }
            // Always set these fields
            $dropoffDetailsData['property_id'] = $dropoffProperty->id;
            $dropoffDetailsData['property_type'] = 'drop_off';
            
            if ($dropoffDetails) {
                $dropoffDetails->update($dropoffDetailsData);
                $dropoffDetails->refresh();
            } else {
                $dropoffDetails = PropertyDetail::create($dropoffDetailsData);
            }
            $result['dropoff_details'] = $dropoffDetails;
        }
        $result['uuid'] = $uuid;
        return $this->success($result, 'Property details saved successfully');
    }

    /**
     * Get property details by UUID
     */
    public function getByUuid(string $uuid)
    {
        $pickupProperty = Property::where('uuid', $uuid)
            ->where('type', 'pick_up')
            ->first();
        $dropoffProperty = Property::where('uuid', $uuid)
            ->where('type', 'drop_off')
            ->first();
        if (!$pickupProperty && !$dropoffProperty) {
            return $this->error('Properties not found', 404);
        }
        $result = [
            'uuid' => $uuid,
            'pickup_details' => null,
            'dropoff_details' => null,
        ];
        if ($pickupProperty) {
            $result['pickup_details'] = PropertyDetail::where('property_id', $pickupProperty->id)
                ->where('property_type', 'pick_up')
                ->first();
        }
        if ($dropoffProperty) {
            $result['dropoff_details'] = PropertyDetail::where('property_id', $dropoffProperty->id)
                ->where('property_type', 'drop_off')
                ->first();
        }
        $result['pickup_location'] = $pickupProperty;
        $result['dropoff_location'] = $dropoffProperty;
        return $this->success($result, 'Property details fetched successfully');
    }
}
