<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\Quote;
use Illuminate\Http\Request;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\QuoteCalculationTrait as TraitsQuoteCalculationTrait;
use App\Http\Controllers\API\Traits\StripeTrait;
use App\Http\Controllers\API\Traits\EmailTrait;
use App\Models\Property;
use App\Models\PropertyDetail;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class QuoteController extends Controller
{
    use JsonResponseTrait, TraitsQuoteCalculationTrait, StripeTrait, EmailTrait;

    // /**
    //  * Store or update quote for a booking by UUID
    //  */
    // public function store(Request $request)
    // {
    //     $uuid = $request->uuid;
    //     if (!$uuid) {
    //         return $this->error('UUID is required', 422);
    //     }
    //     $booking = Booking::where('uuid', $uuid)->first();
    //     if (!$booking) {
    //         return $this->error('Booking not found', 404);
    //     }

    //                                 // Refresh booking to ensure we have the latest data including all items
    //     $booking->refresh();
    //     // Clear the relationship cache and reload with fresh data
    //     $booking->unsetRelation('bookingItems');
    //     $booking->load(['bookingItems.entity', 'truck']);

    //     // Get distance from request - convert from meters to kilometers if provided
    //     $distanceKm = null;
    //     if ($request->has('distance_km') && $request->filled('distance_km')) {
    //         // Frontend sends distance in meters, convert to kilometers
    //         $distanceMeters = $request->get('distance_km');
    //         $distanceKm = is_numeric($distanceMeters) ? ($distanceMeters / 1000) : null;
    //     }

    //     // Check if quote already exists to preserve existing distance if not provided
    //     $quote = Quote::where('uuid', $uuid)->first();
    //     if ($quote && $distanceKm === null) {
    //         $distanceKm = $quote->distance_km; // Preserve existing distance
    //     }

    //     $additionalTimeMinutes = $request->get('additional_time_minutes', 0);
    //     // Calculate quote using trait - this will recalculate all costs including storage and assemble/disassemble
    //     $quoteData = $this->calculateQuote($booking, $distanceKm, $additionalTimeMinutes, $request);
    //     if (!$quoteData) {
    //         return $this->error('Unable to calculate quote. Missing truck or pricing information.', 400);
    //     }
    //     $quoteData['uuid'] = $uuid;
    //     $quoteData['user_id'] = $booking->user_id;
    //     $quoteData['distance_km'] = $distanceKm; // Ensure distance is set

    //     // Ensure total_cost is always included and properly calculated
    //     if (!isset($quoteData['total_cost']) || $quoteData['total_cost'] === null) {
    //         // If total_cost is missing, calculate it from the breakdown
    //         $totalCost = 0;
    //         if (isset($quoteData['movers_cost'])) {
    //             $totalCost += (float) $quoteData['movers_cost'];
    //         }
    //         // Check for callout_fee in response or raw_data
    //         $calloutFee = 0;
    //         if (isset($quoteData['callout_fee'])) {
    //             $calloutFee = (float) $quoteData['callout_fee'];
    //         } elseif (isset($quoteData['raw_data']['breakdown']['callout_fee'])) {
    //             $calloutFee = (float) $quoteData['raw_data']['breakdown']['callout_fee'];
    //         }
    //         $totalCost += $calloutFee;
    //         if (isset($quoteData['assemble_disassemble_cost'])) {
    //             $totalCost += (float) $quoteData['assemble_disassemble_cost'];
    //         }
    //         if (isset($quoteData['storage_items_cost'])) {
    //             $totalCost += (float) $quoteData['storage_items_cost'];
    //         }
    //         $quoteData['total_cost'] = round($totalCost, 2);
    //     } else {
    //         $quoteData['total_cost'] = round((float) $quoteData['total_cost'], 2);
    //     }

    //     if ($quote) {
    //         // Update existing quote, but preserve quote_number, order_number if they exist
    //         if (!$quote->quote_number) {
    //             $quoteData['quote_number'] = Quote::generateQuoteNumber();
    //         }
    //         // Preserve existing status and quote_type if not being updated
    //         if (!$request->has('status')) {
    //             unset($quoteData['status']);
    //         }

    //         // Explicitly update total_cost to ensure it's saved
    //         $quote->update($quoteData);
    //         $quote->refresh();
    //     } else {
    //         $quoteData['quote_number'] = Quote::generateQuoteNumber();
    //         $quote = Quote::create($quoteData);
    //     }
    //     $quote->load(['user']);
    //     return $this->success($quote, 'Quote calculated and saved successfully');
    // }

    /**
     * Get quote by UUID
     */
    public function getByUuid(string $uuid)
    {
        $quote = Quote::where('uuid', $uuid)
            ->with(['user', 'truck'])
            ->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        $pickUpProperty = Property::where('uuid', $uuid)->where('type', 'pick_up')->first();
        $dropOffProperty = Property::where('uuid', $uuid)->where('type', 'drop_off')->first();
        $quote['pickup_property'] = $pickUpProperty;
        $quote['dropoff_property'] = $dropOffProperty;
        // Get property details with only id and property_category
        $propertyDetailsPickUp = null;
        $propertyDetailsDropOff = null;
        if ($pickUpProperty) {
            $propertyDetailsPickUp = PropertyDetail::where('property_id', $pickUpProperty->id)
                ->where('property_type', 'pick_up')
                ->select('id', 'property_category')
                ->first();
        }

        if ($dropOffProperty) {
            $propertyDetailsDropOff = PropertyDetail::where('property_id', $dropOffProperty->id)
                ->where('property_type', 'drop_off')
                ->select('id', 'property_category')
                ->first();
        }
        $quote['property_details_pickup'] = $propertyDetailsPickUp;
        $quote['property_details_dropoff'] = $propertyDetailsDropOff;
        return $this->success($quote, 'Quote fetched successfully');
    }

    /**
     * Update quote and process Stripe payment (30% deposit or full amount)
     * Simplified one-step flow: If payment_method_id is provided, payment is charged immediately
     */
    public function update(Request $request)
    {
        $quote = Quote::where('uuid', $request->uuid)->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        $updateData = $request->only(['status']);

        // Check if payment processing is requested
        if ($request->has('payment_type')) {
            try {
                $paymentType = $request->payment_type; // 'deposit' or 'full'
                $paymentMethodId = $request->input('payment_method_id'); // From frontend Stripe Card Element
                // Validate payment type
                if (!in_array($paymentType, ['deposit', 'full'])) {
                    return $this->error('Invalid payment type. Must be "deposit" or "full"', 422);
                }
                // Check if payment already exists
                if ($paymentType === 'deposit' && $quote->stripe_payment_intent_deposit_id) {
                    return $this->error('Deposit payment intent already exists', 422);
                }
                // Create and optionally confirm payment intent (one-step if payment_method_id provided)
                $paymentResult = $this->createPaymentIntentForQuote($quote, $quote->total_cost, $paymentType, $paymentMethodId);
                $paymentIntent = $paymentResult['payment_intent'];
                $customer = $paymentResult['customer'];

                // Update quote with Stripe information
                $updateData['stripe_customer_id'] = $customer->id;

                // Determine payment status based on payment intent status
                $paymentStatus = 'pending';
                if ($paymentIntent->status === 'succeeded') {
                    $paymentStatus = 'succeeded';
                } elseif ($paymentIntent->status === 'requires_action') {
                    $paymentStatus = 'processing'; // 3D Secure required
                } elseif ($paymentIntent->status === 'canceled' || $paymentIntent->status === 'payment_failed') {
                    $paymentStatus = 'failed';
                }

                if ($paymentType === 'deposit') {
                    $updateData['stripe_payment_intent_deposit_id'] = $paymentIntent->id;
                    $updateData['deposit_amount'] = $paymentResult['payment_amount'];
                    $updateData['remaining_amount'] = $paymentResult['remaining_amount'];
                    $updateData['deposit_payment_status'] = $paymentStatus;
                } else {
                    // Full payment - store in deposit fields for consistency
                    $updateData['stripe_payment_intent_deposit_id'] = $paymentIntent->id;
                    $updateData['deposit_amount'] = $paymentResult['payment_amount'];
                    $updateData['remaining_amount'] = 0;
                    $updateData['deposit_payment_status'] = $paymentStatus;
                    // When payment is full, set remaining_payment_status to succeeded
                    if ($paymentStatus === 'succeeded') {
                        $updateData['remaining_payment_status'] = 'succeeded';
                    }
                }

                // Convert to order if not already
                if ($quote->quote_type !== 'order') {
                    $updateData['quote_type'] = 'order';
                    if (!$quote->order_number) {
                        $updateData['order_number'] = Quote::generateOrderNumber();
                    }
                }

                // Update quote
                $quote->update($updateData);
                $quote->refresh();
                $quote->load(['user']);

                // Prepare response data for frontend
                $responseData = [
                    'quote' => $quote,
                    'payment_status' => $paymentStatus,
                    'amount_charged' => $paymentResult['payment_amount'],
                    'remaining_amount' => $paymentResult['remaining_amount'],
                    'total_amount' => $paymentResult['total_amount'],
                ];

                $responseData['pickup_property'] = Property::where('uuid', $quote->uuid)->where('type', 'pick_up')->first();
                $responseData['dropoff_property'] = Property::where('uuid', $quote->uuid)->where('type', 'drop_off')->first();

                // If deposit succeeded, also create and return remaining payment link
                if ($paymentType === 'deposit' && $paymentStatus === 'succeeded') {
                    try {
                        $paymentLinkResult = $this->createRemainingPaymentLink($quote);
                        $responseData['remaining_payment_link'] = $paymentLinkResult['payment_link_url'];
                        // Update quote with payment link
                        $quote->update(['remaining_payment_url' => $paymentLinkResult['payment_link_url']]);
                    } catch (\Exception $e) {
                        Log::warning('Could not create remaining payment link: ' . $e->getMessage());
                        // Don't fail the request if payment link creation fails
                    }
                }

                // Note: Order confirmation emails are sent via Stripe webhook (payment_intent.succeeded event)
                // This ensures emails are only sent once when payment is actually confirmed by Stripe
                // Do not send emails here to avoid duplicate emails

                // if ($paymentStatus === 'succeeded') {
                //     try {
                //         $emailResults = $this->sendOrderConfirmationEmail($quote, $paymentType, $paymentStatus, false);
                //         if (!$emailResults['customer']) {
                //             Log::warning('Failed to send order confirmation email to customer for quote ID: ' . $quote->id);
                //         }
                //         if (!$emailResults['admin']) {
                //             Log::warning('Failed to send order confirmation email to admin for quote ID: ' . $quote->id);
                //         }
                //     } catch (\Exception $e) {
                //         Log::error('Error sending order confirmation emails: ' . $e->getMessage());
                //         // Don't fail the request if email sending fails
                //     }
                // }

                $message = $paymentStatus === 'succeeded'
                    ? 'Payment charged successfully!'
                    : ($paymentStatus === 'requires_action'
                        ? 'Payment requires authentication. Use client_secret to complete 3D Secure.'
                        : 'Payment intent created. Use client_secret to confirm payment on frontend.');
                return $this->success($responseData, $message);
            } catch (\Stripe\Exception\ApiErrorException $e) {
                Log::error('Stripe API Error in quote update: ' . $e->getMessage());
                $errorMessage = $this->getStripeErrorMessage($e);
                return $this->error($errorMessage, 422);
            } catch (\Exception $e) {
                Log::error('Error processing payment: ' . $e->getMessage());
                return $this->error('Error processing payment: ' . $e->getMessage(), 500);
            }
        }

        // Regular update without payment
        $quote->update($updateData);
        $quote->refresh();
        $quote->load(['user']);

        return $this->success($quote, 'Quote updated successfully');
    }

    /**
     * Create remaining payment link (70%)
     * This is called after work completion to send payment link to customer via email
     */
    public function createRemainingPaymentLinkEndpoint(Request $request)
    {
        $quote = Quote::where('uuid', $request->uuid)->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        // Check if deposit was paid
        if ($quote->deposit_payment_status !== 'succeeded') {
            return $this->error('Deposit payment must be completed first', 422);
        }
        // Check if remaining payment already exists and is completed
        if ($quote->remaining_payment_status === 'succeeded') {
            return $this->error('Remaining payment already completed', 422);
        }
        // Check if payment link already exists
        if ($quote->remaining_payment_url) {
            return $this->success([
                'quote' => $quote->fresh(),
                'payment_link' => $quote->remaining_payment_url,
                'amount' => $quote->remaining_amount,
            ], 'Payment link already exists');
        }
        try {
            // Create remaining payment link using StripeTrait
            $paymentLinkResult = $this->createRemainingPaymentLink($quote);
            $paymentLink = $paymentLinkResult['payment_link'];
            $paymentLinkUrl = $paymentLinkResult['payment_link_url'];
            $customer = $paymentLinkResult['customer'];
            // Ensure customer ID is saved
            if (!$quote->stripe_customer_id) {
                $quote->update(['stripe_customer_id' => $customer->id]);
            }
            // Update quote with remaining payment link
            $quote->update([
                'remaining_payment_url' => $paymentLinkUrl,
                'remaining_payment_status' => 'pending',
            ]);
            return $this->success([
                'quote' => $quote->fresh(),
                'payment_link' => $paymentLinkUrl,
                'amount' => $paymentLinkResult['remaining_amount'],
            ], 'Payment link created successfully. Send this link to customer via email.');
        } catch (\Stripe\Exception\ApiErrorException $e) {
            Log::error('Stripe API Error creating payment link: ' . $e->getMessage());
            $errorMessage = $this->getStripeErrorMessage($e);
            return $this->error($errorMessage, 422);
        } catch (\Exception $e) {
            Log::error('Error creating payment link: ' . $e->getMessage());
            return $this->error('Error creating payment link: ' . $e->getMessage(), 500);
        }
    }


    /**
     * Get payment intent status (for checking payment status)
     */
    public function getPaymentIntentStatus(Request $request)
    {
        $request->validate([
            'payment_intent_id' => 'required|string',
        ]);

        try {
            $this->initializeStripe();
            $paymentIntent = \Stripe\PaymentIntent::retrieve($request->payment_intent_id);

            return $this->success([
                'payment_intent_id' => $paymentIntent->id,
                'status' => $paymentIntent->status,
                'amount' => $paymentIntent->amount / 100,
                'currency' => $paymentIntent->currency,
            ], 'Payment intent status retrieved');
        } catch (\Exception $e) {
            Log::error('Error retrieving payment intent status: ' . $e->getMessage());
            return $this->error('Error retrieving payment status: ' . $e->getMessage(), 500);
        }
    }

    public function getUserQuotes(Request $request)
    {
        $user = $request->user();
        $booking = Booking::where('user_id', $user->id)->get();
        $quotes = Quote::whereIn('booking_id', $booking->pluck('id'))
            ->orderByDesc('created_at')
            ->get();

        // Get all UUIDs from quotes
        $uuids = $quotes->pluck('uuid')->filter()->unique();

        // Fetch all properties for these quotes (both pick_up and drop_off)
        $properties = Property::whereIn('uuid', $uuids)
            ->get()
            ->groupBy('uuid');

        // Format quotes with their related properties
        $formatted = $quotes->map(function (Quote $quote) use ($properties) {
            $data = $quote->toArray();

            // Get properties for this quote's UUID
            $quoteProperties = $properties->get($quote->uuid, collect());

            // Separate pick_up and drop_off properties
            $pickUpProperty = $quoteProperties->where('type', 'pick_up')->first();
            $dropOffProperty = $quoteProperties->where('type', 'drop_off')->first();

            // Add properties to the quote data
            $data['pickup_property'] = $pickUpProperty ? $pickUpProperty->toArray() : null;
            $data['dropoff_property'] = $dropOffProperty ? $dropOffProperty->toArray() : null;

            return $data;
        });

        return $this->success($formatted, 'User quotes fetched successfully');
    }

    public function getUserQuotesUuid(string $uuid)
    {
        $quotes = Quote::with([
            'user',
            'truck',
            'booking.bookingItems.entity',
            'booking.pickUpProperty.propertyDetails',
            'booking.dropOffProperty.propertyDetails',
            'booking.truck',
        ])
            ->where('uuid', $uuid)
            ->orderByDesc('created_at')
            ->get();

        // Get all UUIDs from quotes
        $uuids = $quotes->pluck('uuid')->filter()->unique();

        // Fetch all properties for these quotes (both pick_up and drop_off) matched by UUID
        $properties = Property::whereIn('uuid', $uuids)
            ->with('propertyDetails')
            ->get()
            ->groupBy('uuid');

        // Format quotes with all related data
        $formatted = $quotes->map(function (Quote $quote) use ($properties) {
            $data = $quote->toArray();

            // Get properties for this quote's UUID (matched by UUID, not through booking)
            $quoteProperties = $properties->get($quote->uuid, collect());

            // Separate pick_up and drop_off properties matched by UUID
            $pickUpProperty = $quoteProperties->where('type', 'pick_up')->first();
            $dropOffProperty = $quoteProperties->where('type', 'drop_off')->first();

            // Add properties matched by UUID to the quote data
            $data['pickup_property'] = $pickUpProperty ? $pickUpProperty->toArray() : null;
            $data['dropoff_property'] = $dropOffProperty ? $dropOffProperty->toArray() : null;

            // Get property details for UUID-matched properties
            $propertyDetailsPickUp = null;
            $propertyDetailsDropOff = null;
            if ($pickUpProperty && $pickUpProperty->propertyDetails) {
                $propertyDetailsPickUp = $pickUpProperty->propertyDetails->toArray();
            }
            if ($dropOffProperty && $dropOffProperty->propertyDetails) {
                $propertyDetailsDropOff = $dropOffProperty->propertyDetails->toArray();
            }
            $data['property_details_pickup'] = $propertyDetailsPickUp;
            $data['property_details_dropoff'] = $propertyDetailsDropOff;

            return $data;
        });
        return $this->success($formatted, 'User quotes fetched successfully');
    }
}
