<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Quote;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\Webhook;
use Stripe\Exception\SignatureVerificationException;
use App\Models\GeneralSettings;
use App\Http\Controllers\API\Traits\EmailTrait;

class StripeWebhookController extends Controller
{
    use EmailTrait;
    /**
     * Handle Stripe webhook events
     */
    public function handleWebhook(Request $request)
    {
        try {
            // Initialize Stripe from config
            $activeStripeEnvironment = GeneralSettings::where('option', '_active_stripe_environment')->first();
            $environment = $activeStripeEnvironment ? $activeStripeEnvironment->value : 'sandbox';
            // Get secret key from config based on environment
            if ($environment === 'sandbox') {
                $secretKey = config('services.stripe.sandbox_secret');
            } else {
                $secretKey = config('services.stripe.live_secret') ?? config('services.stripe.secret');
            }
            // Fallback to GeneralSettings if config doesn't have the key
            if (!$secretKey) {
                $stripeApiKey = $environment === 'sandbox' ? '_stripe_sandbox_key' : '_stripe_live_key';
                $stripeSecret = GeneralSettings::where('option', $stripeApiKey)->first();
                if ($stripeSecret) {
                    $secretKey = $stripeSecret->value;
                }
            }
            if (!$secretKey) {
                Log::error('Stripe secret key not found in config or GeneralSettings');
                return response()->json(['error' => 'Stripe configuration not found'], 500);
            }
            Stripe::setApiKey($secretKey);
            // Get webhook secret from environment or config
            $webhookSecret = config('services.stripe.webhook_secret');
            // Fallback to GeneralSettings if config doesn't have the secret
            if (!$webhookSecret) {
                $webhookSecretSetting = GeneralSettings::where('option', '_stripe_webhook_secret')->first();
                if ($webhookSecretSetting) {
                    $webhookSecret = $webhookSecretSetting->value;
                }
            }
            if (!$webhookSecret) {
                Log::error('Stripe webhook secret not configured');
                return response()->json(['error' => 'Webhook secret not configured'], 500);
            }
            // Get the raw payload - MUST use php://input for webhook signature verification
            // Laravel's getContent() should work, but if it doesn't, use php://input directly
            $payload = $request->getContent();
            if (empty($payload)) {
                // Fallback: get raw input stream
                $payload = @file_get_contents('php://input');
            }
            if (empty($payload)) {
                Log::error('Stripe webhook payload is empty');
                return response()->json(['error' => 'Empty payload'], 400);
            }
            $sigHeader = $request->header('Stripe-Signature');
            if (!$sigHeader) {
                Log::error('Stripe webhook signature header is missing');
                return response()->json(['error' => 'Missing signature header'], 400);
            }
            try {
                $event = Webhook::constructEvent($payload, $sigHeader, $webhookSecret);
            } catch (SignatureVerificationException $e) {
                Log::error('Stripe webhook signature verification failed: ' . $e->getMessage());
                return response()->json(['error' => 'Invalid signature'], 400);
            }
            // Handle the event
            switch ($event->type) {
                case 'payment_intent.succeeded':
                    $this->handlePaymentIntentSucceeded($event->data->object);
                    break;

                case 'payment_intent.payment_failed':
                    $this->handlePaymentIntentFailed($event->data->object);
                    break;

                case 'checkout.session.completed':
                    $this->handleCheckoutSessionCompleted($event->data->object);
                    break;

                case 'payment_intent.requires_action':
                    $this->handlePaymentIntentRequiresAction($event->data->object);
                    break;

                default:
                    Log::info('Unhandled Stripe webhook event: ' . $event->type);
            }
            return response()->json(['received' => true], 200);
        } catch (\Exception $e) {
            Log::error('Error handling Stripe webhook: ' . $e->getMessage());
            return response()->json(['error' => 'Webhook processing failed'], 500);
        }
    }

    /**
     * Handle successful payment intent
     */
    private function handlePaymentIntentSucceeded($paymentIntent)
    {
        try {
            // Stripe metadata is a StripeObject, access it properly
            $metadata = $paymentIntent->metadata ?? null;
            $orderId = null;
            $quoteId = null;
            $paymentType = null;
            
            if ($metadata) {
                // Access metadata as object properties
                $orderId = $metadata->order_id ?? null;
                $quoteId = $metadata->quote_id ?? null;
                $paymentType = $metadata->payment_type ?? null;
            }
            if (!$orderId && !$quoteId) {
                Log::warning('Payment intent succeeded but no order_id or quote_id in metadata', [
                    'payment_intent_id' => $paymentIntent->id
                ]);
                return;
            }
            // Find quote by order_number or quote_id
            $quote = null;
            if ($orderId) {
                $quote = Quote::where('order_number', $orderId)->first();
            }
            if (!$quote && $quoteId) {
                $quote = Quote::find($quoteId);
            }
            if (!$quote) {
                Log::warning('Quote not found for payment intent', [
                    'payment_intent_id' => $paymentIntent->id,
                    'order_id' => $orderId,
                    'quote_id' => $quoteId
                ]);
                return;
            }
            // Update quote based on payment type
            if ($paymentType === 'deposit') {
                $quote->update([
                    'deposit_payment_status' => 'succeeded',
                    'stripe_payment_intent_deposit_id' => $paymentIntent->id,
                ]);
                Log::info('Deposit payment succeeded', [
                    'quote_id' => $quote->id,
                    'payment_intent_id' => $paymentIntent->id
                ]);
                
                // Send confirmation emails to customer and admin for deposit payment
                try {
                    $emailResults = $this->sendOrderConfirmationEmail($quote, 'deposit', 'succeeded', false);
                    
                    if (!$emailResults['customer']) {
                        Log::warning('Failed to send order confirmation email to customer for quote ID: ' . $quote->id);
                    }
                    if (!$emailResults['admin']) {
                        Log::warning('Failed to send order confirmation email to admin for quote ID: ' . $quote->id);
                    }
                } catch (\Exception $e) {
                    Log::error('Error sending order confirmation emails after deposit payment: ' . $e->getMessage());
                }
            } elseif ($paymentType === 'remaining') {
                // Add remaining amount to deposit amount and set remaining amount to zero
                $remainingAmount = $quote->remaining_amount ?? 0;
                $newDepositAmount = ($quote->deposit_amount ?? 0) + $remainingAmount;
                
                $quote->update([
                    'remaining_payment_status' => 'succeeded',
                    'stripe_payment_intent_remaining_id' => $paymentIntent->id,
                    'deposit_amount' => $newDepositAmount,
                    'remaining_amount' => 0,
                    'remaining_payment_url'=>null,
                ]);
                
                Log::info('Remaining payment succeeded - Amount transferred to deposit', [
                    'quote_id' => $quote->id,
                    'payment_intent_id' => $paymentIntent->id,
                    'remaining_amount_transferred' => $remainingAmount,
                    'new_deposit_amount' => $newDepositAmount
                ]);
                
                // Refresh quote to get updated values
                $quote->refresh();
                
                // Send confirmation emails to customer and admin
                try {
                    $emailResults = $this->sendOrderConfirmationEmail($quote, 'full', 'succeeded', true);
                    
                    if (!$emailResults['customer']) {
                        Log::warning('Failed to send order confirmation email to customer for quote ID: ' . $quote->id);
                    }
                    if (!$emailResults['admin']) {
                        Log::warning('Failed to send order confirmation email to admin for quote ID: ' . $quote->id);
                    }
                } catch (\Exception $e) {
                    Log::error('Error sending order confirmation emails after remaining payment: ' . $e->getMessage());
                }
            }

        } catch (\Exception $e) {
            Log::error('Error handling payment intent succeeded: ' . $e->getMessage());
        }
    }

    /**
     * Handle failed payment intent
     */
    private function handlePaymentIntentFailed($paymentIntent)
    {
        try {
            // Stripe metadata is a StripeObject, access it properly
            $metadata = $paymentIntent->metadata ?? null;
            $orderId = null;
            $quoteId = null;
            $paymentType = null;
            
            if ($metadata) {
                // Access metadata as object properties
                $orderId = $metadata->order_id ?? null;
                $quoteId = $metadata->quote_id ?? null;
                $paymentType = $metadata->payment_type ?? null;
            }

            if (!$orderId && !$quoteId) {
                return;
            }

            $quote = null;
            if ($orderId) {
                $quote = Quote::where('order_number', $orderId)->first();
            }
            if (!$quote && $quoteId) {
                $quote = Quote::find($quoteId);
            }

            if (!$quote) {
                return;
            }
            // Update quote based on payment type
            if ($paymentType === 'deposit') {
                $quote->update([
                    'deposit_payment_status' => 'failed',
                ]);
            } elseif ($paymentType === 'remaining') {
                $quote->update([
                    'remaining_payment_status' => 'failed',
                ]);
            }

            Log::warning('Payment intent failed', [
                'quote_id' => $quote->id,
                'payment_intent_id' => $paymentIntent->id,
                'payment_type' => $paymentType
            ]);

        } catch (\Exception $e) {
            Log::error('Error handling payment intent failed: ' . $e->getMessage());
        }
    }

    /**
     * Handle completed checkout session (for remaining payment)
     */
    private function handleCheckoutSessionCompleted($session)
    {
        try {
            // Stripe metadata is a StripeObject, access it properly
            $metadata = $session->metadata ?? null;
            $orderId = null;
            $quoteId = null;
            
            if ($metadata) {
                // Access metadata as object properties
                $orderId = $metadata->order_id ?? null;
                $quoteId = $metadata->quote_id ?? null;
            }

            if (!$orderId && !$quoteId) {
                return;
            }

            $quote = null;
            if ($orderId) {
                $quote = Quote::where('order_number', $orderId)->first();
            }
            if (!$quote && $quoteId) {
                $quote = Quote::find($quoteId);
            }

            if (!$quote) {
                return;
            }

            // Add remaining amount to deposit amount and set remaining amount to zero
            $remainingAmount = $quote->remaining_amount ?? 0;
            $newDepositAmount = ($quote->deposit_amount ?? 0) + $remainingAmount;
            
            // Update quote with remaining payment status
            $quote->update([
                'remaining_payment_status' => 'succeeded',
                'stripe_payment_intent_remaining_id' => $session->id,
                'deposit_amount' => $newDepositAmount,
                'remaining_amount' => 0,
                'remaining_payment_url'=>null,
            ]);

            Log::info('Checkout session completed for remaining payment - Amount transferred to deposit', [
                'quote_id' => $quote->id,
                'session_id' => $session->id,
                'remaining_amount_transferred' => $remainingAmount,
                'new_deposit_amount' => $newDepositAmount
            ]);

            // Refresh quote to get updated values
            $quote->refresh();
            
            // Send confirmation emails to customer and admin
            try {
                $emailResults = $this->sendOrderConfirmationEmail($quote, 'full', 'succeeded', true);
                
                if (!$emailResults['customer']) {
                    Log::warning('Failed to send order confirmation email to customer for quote ID: ' . $quote->id);
                }
                if (!$emailResults['admin']) {
                    Log::warning('Failed to send order confirmation email to admin for quote ID: ' . $quote->id);
                }
            } catch (\Exception $e) {
                Log::error('Error sending order confirmation emails after checkout session completed: ' . $e->getMessage());
            }

        } catch (\Exception $e) {
            Log::error('Error handling checkout session completed: ' . $e->getMessage());
        }
    }

    /**
     * Handle payment intent requiring action (3D Secure)
     */
    private function handlePaymentIntentRequiresAction($paymentIntent)
    {
        try {
            // Stripe metadata is a StripeObject, access it properly
            $metadata = $paymentIntent->metadata ?? null;
            $orderId = null;
            $quoteId = null;
            $paymentType = null;
            
            if ($metadata) {
                // Access metadata as object properties
                $orderId = $metadata->order_id ?? null;
                $quoteId = $metadata->quote_id ?? null;
                $paymentType = $metadata->payment_type ?? null;
            }

            if (!$orderId && !$quoteId) {
                return;
            }

            $quote = null;
            if ($orderId) {
                $quote = Quote::where('order_number', $orderId)->first();
            }
            if (!$quote && $quoteId) {
                $quote = Quote::find($quoteId);
            }

            if (!$quote) {
                return;
            }

            // Update quote payment status to processing
            if ($paymentType === 'deposit') {
                $quote->update([
                    'deposit_payment_status' => 'processing',
                ]);
            } elseif ($paymentType === 'remaining') {
                $quote->update([
                    'remaining_payment_status' => 'processing',
                ]);
            }

            Log::info('Payment intent requires action', [
                'quote_id' => $quote->id,
                'payment_intent_id' => $paymentIntent->id,
                'payment_type' => $paymentType
            ]);

        } catch (\Exception $e) {
            Log::error('Error handling payment intent requires action: ' . $e->getMessage());
        }
    }
}
