<?php

namespace App\Http\Controllers\API\Traits;

use App\Models\GeneralSettings;
use SendGrid;
use SendGrid\Mail\Mail;
use Illuminate\Support\Facades\Log;

trait EmailTrait
{
    public function SendInstantEmail(array $params)
    {
        try {
            // Validate required parameters
            if (empty($params['to']) || empty($params['subject']) || empty($params['msg'])) {
                Log::error("SendInstantEmail: Missing required parameters", [
                    'params' => array_keys($params),
                    'has_to' => !empty($params['to']),
                    'has_subject' => !empty($params['subject']),
                    'has_msg' => !empty($params['msg'])
                ]);
                return false;
            }

            // Validate email address
            if (!filter_var($params['to'], FILTER_VALIDATE_EMAIL)) {
                Log::error("SendInstantEmail: Invalid email address", ['to' => $params['to']]);
                return false;
            }

            // Get email and password from database
            $settings = GeneralSettings::get()->pluck('value', 'option')->toArray();
            $mailFromAddress = $settings['_mail_from_address'] ?? env('MAIL_FROM_ADDRESS');
            $mailPassword = $settings['_mail_password'] ?? env('MAIL_PASSWORD');
            
            // Validate email configuration
            if (empty($mailFromAddress)) {
                Log::error("SendInstantEmail: Mail from address is not configured");
                return false;
            }
            
            if (empty($mailPassword)) {
                Log::error("SendInstantEmail: Mail password/API key is not configured");
                return false;
            }
            
            // Get business name for the from name
            $businessName = $settings['_b_name'] ?? 'Speedy Move';

            $email = new Mail();
            $email->setFrom($mailFromAddress, $businessName);
            $email->setSubject($params['subject']);
            $email->addTo($params['to']);
            $email->addContent("text/plain", strip_tags($params['msg'])); // Plain text version
            $email->addContent("text/html", $params['msg']); // HTML version

            $sendgrid = new SendGrid($mailPassword); // Use API key from database
            $response = $sendgrid->send($email);

            if ($response->statusCode() >= 200 && $response->statusCode() < 300) {
                Log::info("Email sent successfully", [
                    'to' => $params['to'],
                    'subject' => $params['subject'],
                    'status_code' => $response->statusCode()
                ]);
                return true;
            } else {
                $responseBody = $response->body();
                Log::error("SendGrid email failed", [
                    'status_code' => $response->statusCode(),
                    'response_body' => $responseBody,
                    'to' => $params['to'],
                    'subject' => $params['subject']
                ]);
                return false;
            }
        } catch (\Exception $e) {
            Log::error("Email sending failed: " . $e->getMessage(), [
                'to' => $params['to'] ?? 'unknown',
                'subject' => $params['subject'] ?? 'unknown',
                'exception' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            return false;
        }
    }

    /**
     * Get all general settings data for email templates
     */
    protected function getGeneralSettingsData()
    {
        $settings = GeneralSettings::get()->pluck('value', 'option')->toArray();
        
        return [
            'logo' => $settings['_header_logo'] ?? '',
            'favicon' => $settings['_fav_icon'] ?? '',
            'business_name' => $settings['_b_name'] ?? 'Speedy Move',
            'business_email' => $settings['_b_email'] ?? '',
            'business_phone' => $settings['_b_number'] ?? '',
            'business_address' => $settings['_b_street_address'] ?? '',
            'business_area' => $settings['_b_area_address'] ?? '',
            'tagline' => $settings['_tagline'] ?? '',
            'facebook_link' => $settings['_facebook_link'] ?? '#',
            'instagram_link' => $settings['_instagram_link'] ?? '#',
            'twitter_link' => $settings['_twitter_link'] ?? '#',
            'youtube_link' => $settings['_youtube_link'] ?? '#',
            'linkedin_link' => $settings['_linke_din'] ?? '#',
            'bank_name' => $settings['_bank_name'] ?? '',
            'abn' => $settings['_abn'] ?? '',
            'bsb' => $settings['_bsb'] ?? '',
            'account_number' => $settings['_acc'] ?? '',
            'bank_details' => $settings['_bank_details'] ?? '',
            'gst_percentage' => $settings['_gst_percentage'] ?? '10',
            'image_url' => url('/'),
        ];
    }

    /**
     * Send account creation email with credentials
     */
    public function sendAccountCreationEmail($user, $password)
    {
        $settings = $this->getGeneralSettingsData();
        
        $params = [
            'subject' => 'Account Created Successfully | ' . $settings['business_name'],
            'to' => $user->email,
            'msg' => view('Admin.Emails.account-login-details', [
                'name' => $user->name,
                'email' => $user->email,
                'password' => $password,
                'settings' => $settings,
            ])->render(),
        ];

        return $this->SendInstantEmail($params);
    }

    /**
     * Send newsletter subscription email
     */
    public function sendNewsletterEmail($email)
    {
        try {
            // Validate email address
            if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                Log::error("sendNewsletterEmail: Invalid email address", ['email' => $email]);
                return false;
            }

            $settings = $this->getGeneralSettingsData();
            
            $params = [
                'subject' => 'Newsletter Subscription | ' . $settings['business_name'],
                'to' => $email,
                'msg' => view('Admin.Emails.news-letter', [
                    'settings' => $settings,
                ])->render(),
            ];

            $result = $this->SendInstantEmail($params);
            
            if ($result) {
                Log::info("Newsletter subscription email sent successfully", ['to' => $email]);
            } else {
                Log::error("Newsletter subscription email failed to send", ['to' => $email]);
            }
            
            return $result;
        } catch (\Exception $e) {
            Log::error("sendNewsletterEmail exception: " . $e->getMessage(), [
                'email' => $email,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    } 
    
    public function contactUsEmail($data)
    {
        try {
            $settings = $this->getGeneralSettingsData();
            // Get admin email from general settings (business_email)
            $adminEmail = $settings['business_email']; // Fallback to hardcoded if not set
            // Validate admin email address
            if (empty($adminEmail) || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
                Log::error("contactUsEmail: Invalid admin email address", ['admin_email' => $adminEmail]);
                return false;
            }
            
            $params = [
                'subject' => 'Contact Us | ' . $settings['business_name'],
                'to' => $adminEmail,
                'msg' => view('Admin.Emails.contact', [
                    'settings' => $settings,
                    'data' => $data
                ])->render(),
            ];

            $result = $this->SendInstantEmail($params);
            
            if ($result) {
                Log::info("Contact us email sent successfully", [
                    'to' => $adminEmail,
                    'from' => $data->email ?? 'unknown'
                ]);
            } else {
                Log::error("Contact us email failed to send", [
                    'to' => $adminEmail,
                    'from' => $data->email ?? 'unknown'
                ]);
            }
            
            return $result;
        } catch (\Exception $e) {
            Log::error("contactUsEmail exception: " . $e->getMessage(), [
                'data' => $data ? (is_object($data) ? $data->toArray() : $data) : null,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send password reset email
     */
    public function sendPasswordResetEmail($user, $resetUrl)
    {
        $settings = $this->getGeneralSettingsData();
        
        $params = [
            'subject' => 'Password Reset Link | ' . $settings['business_name'],
            'to' => $user->email,
            'msg' => view('Admin.Emails.reset-password', [
                'user' => $user,
                'resetUrl' => $resetUrl,
                'settings' => $settings,
            ])->render(),
        ];

        return $this->SendInstantEmail($params);
    }

    /**
     * Send OTP email
     */
    public function sendOtpEmail($user, $otp)
    {
        $settings = $this->getGeneralSettingsData();
        
        $params = [
            'subject' => 'Password Reset OTP | ' . $settings['business_name'],
            'to' => $user->email,
            'msg' => view('Admin.Emails.send-otp', [
                'user' => $user,
                'otp' => $otp,
                'settings' => $settings,
            ])->render(),
        ];

        return $this->SendInstantEmail($params);
    }

    /**
     * Send booking user login details email
     * 
     * @param \App\Models\User $user The user object
     * @param string $password The plain text password
     * @return bool Returns true if email sent successfully, false otherwise
     */
    public function sendBookingUserLoginEmail($user, $password)
    {
        $settings = $this->getGeneralSettingsData();
        
        $params = [
            'subject' => 'Your Account Login Details | ' . $settings['business_name'],
            'to' => $user->email,
            'msg' => view('Admin.Emails.booking-user-login-details', [
                'name' => $user->name,
                'email' => $user->email,
                'password' => $password,
                'settings' => $settings,
            ])->render(),
        ];

        return $this->SendInstantEmail($params);
    }

    /**
     * Send order confirmation email to customer and admin
     * 
     * @param \App\Models\Quote $quote The quote/order object
     * @param string $paymentType 'deposit' or 'full'
     * @param string $paymentStatus Payment status (succeeded, processing, etc.)
     * @param bool $isRemainingPayment Whether this is for remaining payment completion
     * @return array Returns array with 'customer' and 'admin' keys indicating success
     */
    public function sendOrderConfirmationEmail($quote, $paymentType, $paymentStatus, $isRemainingPayment = false)
    {
        $settings = $this->getGeneralSettingsData();
        $user = $quote->user;
        
        if (!$user) {
            Log::error('Cannot send order confirmation email: User not found for quote ID ' . $quote->id);
            return ['customer' => false, 'admin' => false];
        }

        // Get deposit settings from general settings
        $allSettings = GeneralSettings::get()->pluck('value', 'option')->toArray();
        $jobType = $quote->job_type ?? ($quote->raw_data['job_type'] ?? 'local_job');
        
        $depositSettings = [
            'job_type' => $jobType,
            'local_job_deposit_amount' => (float)($allSettings['_local_job_deposit_amount'] ?? 100),
            'interstate_job_deposit_percentage' => (float)($allSettings['_interstate_job_deposit_percentage'] ?? 20),
        ];

        $results = ['customer' => false, 'admin' => false];

        // Determine email subjects based on context
        if ($isRemainingPayment) {
            $customerSubject = 'Remaining Amount Payment Completed | ' . $settings['business_name'];
            $adminSubject = 'Remaining Amount Payment Received | ' . $settings['business_name'];
        } else {
            $customerSubject = 'Order Confirmation | ' . $settings['business_name'];
            $adminSubject = 'New Order Received | ' . $settings['business_name'];
        }

        // Send email to customer
        try {
            $customerParams = [
                'subject' => $customerSubject,
                'to' => $user->email,
                'msg' => view('Admin.Emails.order-confirmation-customer', [
                    'quote' => $quote,
                    'user' => $user,
                    'settings' => $settings,
                    'paymentType' => $paymentType,
                    'paymentStatus' => $paymentStatus,
                    'isRemainingPayment' => $isRemainingPayment,
                    'depositSettings' => $depositSettings,
                ])->render(),
            ];

            $results['customer'] = $this->SendInstantEmail($customerParams);
        } catch (\Exception $e) {
            Log::error('Failed to send order confirmation email to customer: ' . $e->getMessage());
            $results['customer'] = false;
        }

        // Send email to admin
        try {
            // Get admin email from general settings (business_email)
            $adminEmail = $settings['business_email'] ?? 'support@g-techsoltions.com.au'; // Fallback to hardcoded if not set
            // Only send admin email if admin email is configured
            if (!empty($adminEmail)) {
                $adminParams = [
                    'subject' => $adminSubject,
                    'to' => $adminEmail,
                    'msg' => view('Admin.Emails.order-confirmation-admin', [
                        'quote' => $quote,
                        'user' => $user,
                        'settings' => $settings,
                        'paymentType' => $paymentType,
                        'paymentStatus' => $paymentStatus,
                        'isRemainingPayment' => $isRemainingPayment,
                        'depositSettings' => $depositSettings,
                    ])->render(),
                ];

                $results['admin'] = $this->SendInstantEmail($adminParams);
            } else {
                Log::warning('Admin email not configured in general settings (business_email). Skipping admin notification.');
                $results['admin'] = false;
            }
        } catch (\Exception $e) {
            Log::error('Failed to send order confirmation email to admin: ' . $e->getMessage());
            $results['admin'] = false;
        }

        return $results;
    }
}
