<?php

namespace App\Http\Controllers\API\Traits;

use Stripe\Stripe;
use Stripe\Customer;
use Stripe\PaymentIntent;
use App\Models\GeneralSettings;
use App\General\GeneralSettingsClass;
use Illuminate\Support\Facades\Log;



trait StripeTrait
{
    use JsonResponseTrait;
    
    /**
     * Initialize Stripe using config file (preferred method)
     */
    public function initializeStripe()
    {
        try {
            // Get active environment from GeneralSettings or default to sandbox
            $activeStripeEnvironment = GeneralSettings::where('option', '_active_stripe_environment')->first();
            $environment = $activeStripeEnvironment ? $activeStripeEnvironment->value : 'sandbox';
            
            // Get secret key from config based on environment
            if ($environment === 'sandbox') {
                $secretKey = config('services.stripe.sandbox_secret');
            } else {
                $secretKey = config('services.stripe.live_secret') ?? config('services.stripe.secret');
            }
            if (!$secretKey) {
                if (method_exists($this, 'error')) {
                    return $this->error('Stripe secret key not configured in config/services.php', 422);
                }
                throw new \Exception('Stripe secret key not configured in config/services.php');
            }
            
            // Trim any whitespace from the key
            $secretKey = trim($secretKey);
            
            // Validate key format - Stripe keys should start with sk_test_ or sk_live_
            if (!preg_match('/^sk_(test|live)_[a-zA-Z0-9]{24,}$/', $secretKey)) {
                throw new \Exception('Invalid Stripe API key format. Key must start with sk_test_ or sk_live_');
            }
            
            // Set the API key (Stripe::setApiKey() returns void/null, but the key IS set)
            Stripe::setApiKey($secretKey);
            
            // Verify the key works by making a test API call
            try {
                // Test the key by retrieving account info (lightweight call)
                $account = \Stripe\Account::retrieve();
            } catch (\Stripe\Exception\AuthenticationException $e) {
                throw new \Exception('Invalid Stripe API key. The key may be expired or incorrect. Please verify your key in the Stripe dashboard.');
            } catch (\Exception $e) {
                // If account retrieval fails for other reasons, log but don't fail
                // (some keys might not have account access)
            }
        } catch (\Exception $e) {
            if (method_exists($this, 'error')) {
                return $this->error('Stripe initialization failed: ' . $e->getMessage(), 422);
            }
            throw $e;
        }
    }

    /**
     * Create or retrieve Stripe customer for quote
     */
    public function getOrCreateStripeCustomerForQuote($email, $name, $quote)
    {
        try {
            $this->initializeStripe();
            
            // If customer already exists, retrieve it
            if ($quote->stripe_customer_id) {
                try {
                    $customer = Customer::retrieve($quote->stripe_customer_id);
                    return $customer;
                } catch (\Stripe\Exception\InvalidRequestException $e) {
                    throw new \Exception('Stripe customer not found, creating new one');
                    // Customer doesn't exist in Stripe, create new one
                }
            }
            
            // Create new customer
            // Ensure Stripe is initialized (in case it wasn't called properly)
            $this->initializeStripe();
            
            try {
                $customer = Customer::create([
                    'email' => $email,
                    'name' => $name,
                ]);                
                
                return $customer;
            } catch (\Stripe\Exception\AuthenticationException $e) {
              
                throw new \Exception('Invalid Stripe API key. Please check your configuration.');
            }
        } catch (\Exception $e) {
            Log::error('Error creating/retrieving Stripe customer: ' . $e->getMessage());
            throw $e;
        }
    }


    /**
     * Retrieve payment intent status
     */
    public function getPaymentIntentStatus($paymentIntentId)
    {
        try {
            $this->initializeStripe();
            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);
            return $paymentIntent->status;
        } catch (\Exception $e) {
            Log::error('Error retrieving payment intent status: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Create and optionally confirm payment intent - supports both 30% deposit and full amount
     * If payment_method_id is provided, payment is charged immediately (one-step)
     * If no payment_method_id, creates unconfirmed payment intent (for frontend to add payment method)
     */
    public function createPaymentIntentForQuote($quote, $amount, $paymentType = 'deposit', $paymentMethodId = null)
    {
        try {
            $this->initializeStripe();

            // Get user information - try multiple sources
            $user = $quote->user;
            
            // If no user from quote, try to get from booking
            if (!$user && $quote->booking_id) {
                $booking = \App\Models\Booking::with('user')->find($quote->booking_id);
                if ($booking && $booking->user) {
                    $user = $booking->user;
                    // Update quote with user_id for future reference
                    if (!$quote->user_id) {
                        $quote->user_id = $booking->user_id;
                        $quote->save();
                    }
                }
            }
            
            // If still no user, try to get from raw_data
            if (!$user && $quote->raw_data) {
                $rawData = $quote->raw_data;
                if (isset($rawData['email'])) {
                    $user = \App\Models\User::where('email', $rawData['email'])->first();
                    if ($user && !$quote->user_id) {
                        $quote->user_id = $user->id;
                        $quote->save();
                    }
                }
            }
            
            if (!$user) {
                throw new \Exception('User not found for this quote. Please ensure the quote has a valid user_id or booking with user information.');
            }

            // Get or create Stripe customer
            $customer = $this->getOrCreateStripeCustomerForQuote($user->email, $user->name ?? 'Customer', $quote);
            
            // Calculate amounts
            $totalAmount = (float) $amount; // Amount in dollars from frontend
            $totalAmountCents = (int) ($totalAmount * 100); // Convert to cents
            
            $paymentAmountCents = 0;
            $remainingAmountCents = 0;

            // Get job type from quote (check raw_data first, then job_type field)
            $jobType = null;
            if ($quote->raw_data && isset($quote->raw_data['job_type'])) {
                $jobType = $quote->raw_data['job_type'];
            } elseif ($quote->job_type) {
                $jobType = $quote->job_type;
            }
            
            $depositType = null; // Track how deposit was calculated
            $depositValue = null; // Track the deposit setting value used
            
            if ($paymentType === 'deposit') {
                // Get deposit settings from GeneralSettings
                $generalSettings = GeneralSettingsClass::getAllSettings();
                
                if ($jobType === 'local_job') {
                    // Local job: fixed dollar amount deposit
                    $depositAmount = (float) ($generalSettings['_local_job_deposit_amount'] ?? 100);
                    $paymentAmountCents = (int) round($depositAmount * 100);
                    $remainingAmountCents = $totalAmountCents - $paymentAmountCents;
                    $depositType = 'fixed_amount';
                    $depositValue = $depositAmount;
                } elseif ($jobType === 'interstate_job') {
                    // Interstate job: percentage deposit
                    $depositPercentage = (float) ($generalSettings['_interstate_job_deposit_percentage'] ?? 20);
                    $paymentAmountCents = (int) round($totalAmountCents * ($depositPercentage / 100));
                    $remainingAmountCents = $totalAmountCents - $paymentAmountCents;
                    $depositType = 'percentage';
                    $depositValue = $depositPercentage;
                } else {
                    // Fallback: 30% deposit if job type is not set
                    $paymentAmountCents = (int) round($totalAmountCents * 0.30);
                    $remainingAmountCents = $totalAmountCents - $paymentAmountCents;
                    $depositType = 'percentage';
                    $depositValue = 30;
                }
            } else {
                // Full amount (100%)
                $paymentAmountCents = $totalAmountCents;
                $remainingAmountCents = 0;
            }

            // Build Payment Intent data
            $paymentIntentData = [
                'amount' => $paymentAmountCents,
                'currency' => 'AUD',
                'customer' => $customer->id,
                'capture_method' => 'automatic',
                'metadata' => [
                    'order_id' => $quote->order_number ?? $quote->quote_number,
                    'quote_id' => (string) $quote->id,
                    'uuid' => $quote->uuid,
                    'total_amount' => (string) $totalAmountCents,
                    'payment_amount' => (string) $paymentAmountCents,
                    'remaining_amount' => (string) $remainingAmountCents,
                    'customer_email' => $user->email,
                    'payment_type' => $paymentType,
                    'payment_stage' => $paymentType === 'deposit' ? 'first_payment' : 'full_payment',
                    'job_type' => $jobType ?? 'unknown',
                ],
                'description' => ($paymentType === 'deposit' ? 'Deposit' : 'Full payment') . ' for Order #' . ($quote->order_number ?? $quote->quote_number),
            ];
            
            // Add deposit calculation info to metadata if it's a deposit
            if ($paymentType === 'deposit' && $depositType) {
                $paymentIntentData['metadata']['deposit_type'] = $depositType;
                $paymentIntentData['metadata']['deposit_value'] = (string) $depositValue;
            }

            // If payment_method_id is provided, confirm immediately (one-step charge)
            if ($paymentMethodId) {
                $paymentIntentData['payment_method'] = $paymentMethodId;
                $paymentIntentData['confirm'] = true;
                $paymentIntentData['return_url'] = config('app.frontend_url', 'http://localhost:3000') . '/payment/return';
            } else {
                // No payment method - create unconfirmed payment intent (frontend will add payment method)
                $paymentIntentData['confirmation_method'] = 'manual';
                $paymentIntentData['confirm'] = false;
            }

            // Create Payment Intent
            $paymentIntent = PaymentIntent::create($paymentIntentData);
            
            return [
                'payment_intent' => $paymentIntent,
                'customer' => $customer,
                'payment_amount' => $paymentAmountCents / 100,
                'remaining_amount' => $remainingAmountCents / 100,
                'total_amount' => $totalAmount,
                'payment_type' => $paymentType,
            ];
        } catch (\Stripe\Exception\ApiErrorException $e) {
            throw $e;
        } catch (\Exception $e) {
            throw $e;
        }
    }
    
    /**
     * Create Stripe Payment Link for remaining 70% amount
     * This link can be sent via email to customer
     * Customer clicks link and pays on Stripe hosted page
     */
    public function createRemainingPaymentLink($quote)
    {
        try {
            $this->initializeStripe();

            // Get user information - try multiple sources
            $user = $quote->user;
            
            // If no user from quote, try to get from booking
            if (!$user && $quote->booking_id) {
                $booking = \App\Models\Booking::with('user')->find($quote->booking_id);
                if ($booking && $booking->user) {
                    $user = $booking->user;
                    // Update quote with user_id for future reference
                    if (!$quote->user_id) {
                        $quote->user_id = $booking->user_id;
                        $quote->save();
                    }
                }
            }
            
            // If still no user, try to get from raw_data
            if (!$user && $quote->raw_data) {
                $rawData = $quote->raw_data;
                if (isset($rawData['email'])) {
                    $user = \App\Models\User::where('email', $rawData['email'])->first();
                    if ($user && !$quote->user_id) {
                        $quote->user_id = $user->id;
                        $quote->save();
                    }
                }
            }
            
            if (!$user) {
                throw new \Exception('User not found for this quote. Please ensure the quote has a valid user_id or booking with user information.');
            }

            // Get or create Stripe customer
            $customer = $this->getOrCreateStripeCustomerForQuote($user->email, $user->name ?? 'Customer', $quote);
            
            // Ensure customer ID is saved
            if (!$quote->stripe_customer_id) {
                $quote->update(['stripe_customer_id' => $customer->id]);
            }

            // Get remaining amount
            $remainingAmount = (float) $quote->remaining_amount;
            if ($remainingAmount <= 0) {
                throw new \Exception('No remaining amount to pay');
            }

            $remainingAmountCents = (int) ($remainingAmount * 100);

            // Get business name and logo from settings
            $businessName = GeneralSettingsClass::getRequiredValue('_b_name') ?: 'Speedy Move';
            $logoPath = GeneralSettingsClass::getRequiredValue('_header_logo');
            $logoUrl = $logoPath ? url('/') . '/' . $logoPath : null;

            // Create Stripe Payment Link with Speedy Move branding
            $paymentLinkData = [
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'AUD',
                        'product_data' => [
                            'name' => 'Speedy Move - Remaining Payment - Order #' . ($quote->order_number ?? $quote->quote_number),
                            'description' => 'Complete payment for your order with Speedy Move',
                        ],
                        'unit_amount' => $remainingAmountCents,
                    ],
                    'quantity' => 1,
                ]],
                'after_completion' => [
                    'type' => 'redirect',
                    'redirect' => [
                        'url' => config('app.frontend_url', 'http://localhost:3000') . '/payment/success',
                    ],
                ],
                'metadata' => [
                    'order_id' => $quote->order_number ?? $quote->quote_number,
                    'quote_id' => (string) $quote->id,
                    'uuid' => $quote->uuid,
                    'payment_type' => 'remaining',
                    'payment_stage' => 'final_payment',
                    'deposit_payment_intent_id' => $quote->stripe_payment_intent_deposit_id,
                    'business_name' => $businessName,
                ],
            ];

            // Note: Business name and logo on Stripe Payment Link page are set in Stripe Dashboard
            // Go to: Stripe Dashboard > Settings > Branding
            // Set Business name to "Speedy Move" and upload your logo there
            // The product name below will show "Speedy Move" in the payment description

            $paymentLink = \Stripe\PaymentLink::create($paymentLinkData);

            return [
                'payment_link' => $paymentLink,
                'payment_link_url' => $paymentLink->url,
                'customer' => $customer,
                'remaining_amount' => $remainingAmount,
            ];
        } catch (\Stripe\Exception\ApiErrorException $e) {
            Log::error('Stripe API Error creating payment link: ' . $e->getMessage());
            throw $e;
        } catch (\Exception $e) {
            Log::error('Error creating payment link: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get user-friendly error message from Stripe API error
     */
    public function getStripeErrorMessage(\Stripe\Exception\ApiErrorException $e)
    {
        $errorCode = $e->getError()->code ?? null;
        $errorMessage = $e->getError()->message ?? $e->getMessage();
        
        // Map common Stripe error codes to user-friendly messages
        $userFriendlyMessages = [
            'card_declined' => 'Your card was declined. Please try a different payment method.',
            'insufficient_funds' => 'Your card has insufficient funds. Please try a different payment method.',
            'expired_card' => 'Your card has expired. Please use a different payment method.',
            'incorrect_cvc' => 'Your card\'s security code is incorrect. Please check the CVC and try again.',
            'processing_error' => 'An error occurred while processing your card. Please try again.',
            'authentication_required' => 'Additional authentication is required on this card. Please try again or use a different payment method.',
            'lost_card' => 'Your card has been reported as lost. Please contact your bank or try a different payment method.',
            'stolen_card' => 'Your card has been reported as stolen. Please contact your bank or try a different payment method.',
        ];
        
        return $errorCode && isset($userFriendlyMessages[$errorCode]) 
            ? $userFriendlyMessages[$errorCode] 
            : $errorMessage;
    }
}
