<?php

namespace App\Http\Controllers\API\Traits;

use App\Models\GeneralSettings;
use App\Models\JobApplication;
use App\Models\order;
use App\Models\Quote;
use App\Models\User;
use App\Models\UserMeta;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

trait UserTrait
{
    use ValidationTrait, EmailTrait;
    public function registerUser(Request $request)
    {
        $validator = $this->userValidationRequest($request);
        if ($validator->fails()) {
            $errors = $validator->errors()->toArray();
            $message = reset($errors)[0]; // Get the first error message
            return $this->error($message, 422);
        }
        $sanitizedData = $validator->validated();
        $originalPassword = $sanitizedData['password']; // Store original password for email
        if (isset($request->created_by)) {
            $sanitizedData['created_by'] = $request->created_by;
            $sanitizedData['email_verified_at'] = now();
            $sanitizedData['active'] = 0;
        }
        $sanitizedData['password'] = Hash::make($sanitizedData['password']);
        $sanitizedData['role_id'] = 2;
        $sanitizedData['active'] = 1;
        $user = User::create($sanitizedData);
        if (!$user) {
            return $this->error('An Error Occurred, During creating user account.', 500);
        }
        $updateJobApplication = JobApplication::where('email', $request->email)->get();
        if (!$updateJobApplication->isEmpty()) {
            $updateJobApplication->each(function ($jobApplication) use ($user) {
                $jobApplication->user_id = $user->id;
                $jobApplication->save();
            });
        }
        $quotes = Quote::where('user_id', $user->id)->get();
        if (!$quotes->isEmpty()) {
            $quotes->each(function ($quote) use ($user) {
                $quote->user_id = $user->id;
                $quote->save();
            });
        }
        // Send account creation email with credentials
        $this->sendAccountCreationEmail($user, $originalPassword);
        /** @var \App\Models\User $user */
        $token = $user->createToken('authToken')->plainTextToken;
        return $this->withToken($user, $token, 'User registered successfully');
    }

    public function loginUser(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);
        $user = User::where('email', $credentials['email'])->first();
        if (!$user) {
            return $this->error('Invalid email or password.', 401);
        }
        if ($user->active == 0) {
            return $this->error('Your account is not active.', 401);
        }
        if (!Auth::attempt($credentials)) {
            return $this->error('Invalid credentials', 401);
        }
        $user = Auth::user();
        /** @var \App\Models\User $user */
        $token = $user->createToken('authToken')->plainTextToken;
        return $this->withToken($user, $token, 'User logged in successfully');
    }
    public function authUser()
    {
        return auth()->user();
    }

    public function newsLetter($model, Request $request)
    {
        $validator = $this->newLetterValidationRequest($request);
        if ($validator->fails()) {
            $errors = $validator->errors()->toArray();
            $message = reset($errors)[0]; // Get the first error message
            return $this->error($message, 422);
        }
        // $newLetter = $model->firstOrCreate($validator->validated());
        // if (!$newLetter) {
        //     return $this->error('Failed to create user address', 500);
        // }
        // Send newsletter subscription email using new method
        try {
            $emailSent = $this->sendNewsletterEmail($request->email);
            if ($emailSent) {
                return $this->message('Newsletter subscription created successfully. Confirmation email sent.');
            } else {
                // Email failed but subscription might still be processed
                \Illuminate\Support\Facades\Log::warning('Newsletter subscription email failed to send', [
                    'email' => $request->email
                ]);
                return $this->message('Newsletter subscription created successfully, but confirmation email could not be sent.');
            }
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Newsletter subscription email exception', [
                'email' => $request->email,
                'error' => $e->getMessage()
            ]);
            return $this->message('Newsletter subscription created successfully, but confirmation email could not be sent.');
        }
    }
    public function resetPasswordLink($resetUrl, $details)
    {
        return $this->sendPasswordResetEmail($details, $resetUrl);
    }

    protected function checkOldPassword($oldPassword, $currentPassword)
    {
        return Hash::check($oldPassword, $currentPassword);
    }

    protected function handleProfileImageUpload(Request $request, $user)
    {
        if (!$request->hasFile('profile_image')) {
            return;
        }

        $directory = public_path('user-images');

        if (!File::exists($directory)) {
            File::makeDirectory($directory, 0755, true);
        }

        $file = $request->file('profile_image');
        $fileName = Str::uuid()->toString() . '_' . time() . '.' . $file->getClientOriginalExtension();
        $file->move($directory, $fileName);

        $user->profile_image = 'user-images/' . $fileName;
    }

    protected function updateUserProfile(Request $request, $user)
    {
        $user->fill($request->except(['password', 'old_password']));
        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }
        if ($request->hasFile('profile_image')) {
            $this->handleProfileImageUpload($request, $user);
        }
        if ($request->filled('name')) {
            $user->name = $request->name;
        }
        if ($request->filled('email')) {
            $user->email = $request->email;
        }
        if ($request->filled('phone')) {
            $user->phone = $request->phone;
        }
        $user->save();
       
    }

    protected function getUserByEmail(Request $request)
    {
        return User::where('email', $request->email)
            ->first();
    }

    protected function generateAndStoreOtp($user)
    {
        $otp = $this->generateOtp();
        $user->otp = Hash::make($otp);
        $user->otp_created_at = Carbon::now();
        $user->save();
        return $otp;
    }

    protected function generateOtp()
    {
        return str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
    }

    protected function sendOtpEmailToUser($user, $otp)
    {
        return $this->sendOtpEmail($user, $otp);
    }

    protected function generateResetToken()
    {
        return bin2hex(random_bytes(32));
    }

    protected function isOtpValid($otp, $user)
    {
        if (!$user || !$user->otp || !$user->otp_created_at) {
            return false;
        }
        return Hash::check($otp, $user->otp) && $user->otp_created_at->diffInMinutes(Carbon::now()) <= 10;
    }

    protected function verifyOtpAndGetUser(Request $request)
    {
        $user = User::where('email', $request->email)->first();
        if (!$this->isOtpValid($request->otp, $user)) {
            return null;
        }
        return $user;
    }

    protected function verifyResetTokenAndGetUser(Request $request)
    {
        $user = User::where('email', $request->email)->first();
        if (!$user || !$user->password_reset_token || !$user->password_reset_token_created_at) {
            return null;
        }
        // Check if token is valid and not expired (24 hours)
        $tokenValid = Hash::check($request->token, $user->password_reset_token);
        $tokenExpired = $user->password_reset_token_created_at->diffInHours(Carbon::now()) > 24;
        if (!$tokenValid || $tokenExpired) {
            return null;
        }
        return $user;
    }

    protected function validateAndUpdatePassword(Request $request, $user)
    {
        $validator = $this->passwordValidationRequest($request);
        if ($validator->fails()) {
            return $this->handleValidationFailure($validator);
        }
        $user->password = Hash::make($request->password);
        $user->save();
        return true;
    }

    protected function assignSignupPlanToCustomer($customerId)
    {
        $plans = DB::table('membership_plans')->where('signup', 1)->get();
        if ($plans->isEmpty()) {
            return false;
        }
        $currentDate = date('d-m-Y H:i:s');
        foreach ($plans as $plan) {
            $expiration = '';
            if ($plan->duration == 'Un-Limited') {
                $expiration = $this->addDaysToCurrentDate(999999);
            } else {
                $expiration = $this->addDaysToCurrentDate($plan->duration);
            }
            DB::table('membership_subscriptions')->insert([
                'plan_id' => $plan->id,
                'customer_id' => $customerId,
                'created_by' => 0,
                'starting_date' => $currentDate,
                'ending_date' => $expiration,
                'expired' => 0,
            ]);
        }
        return true;
    }

    protected function addDaysToCurrentDate($numberOfDays)
    {
        $currentDate = date('d-m-Y H:i:s');
        $newDate = date('d-m-Y', strtotime($currentDate . ' + ' . $numberOfDays . ' days'));
        return $newDate;
    }

    protected function getSubscriptionDetails($userId)
    {
        $userSubscriptions = DB::table('membership_subscriptions')
            ->join('membership_plans', 'membership_subscriptions.plan_id', '=', 'membership_plans.id')
            ->where('membership_subscriptions.expired', 0)
            ->where('membership_subscriptions.customer_id', $userId)
            ->select('membership_subscriptions.*', 'membership_plans.*');
        $signupPlans = DB::table('membership_plans')
            ->leftJoin('membership_subscriptions', function ($join) use ($userId) {
                $join->on('membership_subscriptions.plan_id', '=', 'membership_plans.id')
                    ->where('membership_subscriptions.customer_id', $userId);
            })
            ->where('membership_plans.signup', 1)
            ->whereNull('membership_subscriptions.id')
            ->select('membership_plans.*', 'membership_subscriptions.*');
        $subscriptionPlans = $userSubscriptions->union($signupPlans)->distinct()->get();
        return $subscriptionPlans;
    }
}
