<?php

namespace App\Http\Controllers\API;


use App\Http\Controllers\Controller;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\Quote;
use App\Models\Truck;
use Illuminate\Http\Request;

class TrcukController extends Controller
{
    use JsonResponseTrait;

    public function index(Request $request)
    {
        $trucks = Truck::active()->ordered()->get();
        if ($trucks->isEmpty()) {
            return $this->error('No trucks found', 404);
        }
        $quote = null;
        $quoteData = null;
        $uuid = $request->input('uuid');
        if($request->input('item_list_empty')) {
            $booking = Booking::where('uuid', $uuid)->first();
            if ($booking) {
                // Get all booking items for this uuid
                $bookingItems = BookingItem::where('uuid', $uuid)->get();
                
                // Delete all booking items
                if ($bookingItems->isNotEmpty()) {
                    foreach ($bookingItems as $item) {
                        $item->delete();
                    }
                }
                
                // Set booking_item_ids to null in booking table
                $booking->booking_item_ids = null;
                $booking->save();
            }
        }
        if (!$request->input('volume')) {
            if ($uuid) {
                $quote = Quote::where('uuid', $uuid)->first();
                if ($quote) {
                    $bookingItems = BookingItem::where('uuid', $uuid)->get();
                    $quoteData = [
                        'extra_movers' => $quote->raw_data['extra_movers'] ?? 0,
                    ];
                    if ($bookingItems->isEmpty()) {
                        $quoteData['total_cubic_meters'] = $quote->total_cubic_meters ?? 0;
                    } else {
                        $quoteData['total_cubic_meters'] = null;
                        $quote->total_cubic_meters = null;
                        $quote->save();
                    }
                }
            }
        }
        // Get parameters from request
        $requiredCubicMeter = $request->input('volume');
        $totalVolume = 0;
        if ($requiredCubicMeter && $requiredCubicMeter > 0) {
            $totalVolume = (float)$requiredCubicMeter;
        }
        // Priority 2: If UUID is provided and volume is not, calculate from booking items
        elseif ($uuid) {
            $bookingItems = BookingItem::where('uuid', $uuid)->with('entity')->get();
            if (!$bookingItems->isEmpty()) {
                foreach ($bookingItems as $item) {
                    $totalVolume += $item->getTotalVolume();
                }
            }
        }
        // If no volume determined (neither volume nor UUID with items), return all trucks
        if ($totalVolume <= 0) {
            $response = $this->success($trucks, 'Active trucks fetched successfully');
            $responseData = $response->getData(true);
            if ($quoteData) {
                $responseData['quote'] = $quoteData;
            }
            return response()->json($responseData, $response->getStatusCode());
        }
        // Filter trucks that can fit the volume (capacity >= totalVolume)
        $fittingTrucks = $trucks->filter(function ($t) use ($totalVolume) {
            return (float)($t->cubic_capacity ?? 0) >= (float)$totalVolume;
        });
        // If no trucks fit, return message
        if ($fittingTrucks->isEmpty()) {
            $response = $this->message('No trucks available with sufficient capacity. Please fill the form and we can arrange trucks according to your volume requirements.', 404);
            $responseData = $response->getData(true);
            if ($quoteData) {
                $responseData['quote'] = $quoteData;
            }
            return response()->json($responseData, $response->getStatusCode());
        }
        // Find the smallest truck that fits (exact or closest match)
        $suggestedTruck = $fittingTrucks->sortBy(function ($t) {
            return (float)($t->cubic_capacity ?? 0);
        })->first();
        // Return only the suggested truck (smallest that fulfills the capacity)
        $capacity = (float)($suggestedTruck->cubic_capacity ?? 0);
        $truckWithMeta = array_merge($suggestedTruck->toArray(), [
            'fits' => true,
            'capacity_remaining' => $capacity - (float)$totalVolume,
            'suggested' => true,
        ]);
        $response = $this->success([$truckWithMeta], 'Truck that fulfills capacity requirement returned');
        $responseData = $response->getData(true);
        if ($quoteData) {
            $responseData['quote'] = $quoteData;
        }
        return response()->json($responseData, $response->getStatusCode());
    }
}
