<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\ChildCategory;
use App\Models\ChildCategoryMeta;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class ChildCategoryController extends Controller
{
    public function index(Request $request)
    {
        $query = ChildCategory::with('mainParentCategory');

        // Keyword filter
        if ($request->filled('keyword')) {
            $query->where('title', 'like', '%' . $request->keyword . '%');
        }

        // Date filters
        if ($request->filled('status')) {
            $today = now();
            switch ($request->status) {
                case 'today':
                    $query->whereDate('created_at', $today);
                    break;
                case 'yesterday':
                    $query->whereDate('created_at', $today->subDay());
                    break;
                case '7days':
                    $query->whereDate('created_at', '>=', now()->subDays(7));
                    break;
                case '30days':
                    $query->whereDate('created_at', '>=', now()->subDays(30));
                    break;
                case 'this_month':
                    $query->whereMonth('created_at', now()->month)
                        ->whereYear('created_at', now()->year);
                    break;
                case 'last_month':
                    $query->whereMonth('created_at', now()->subMonth()->month)
                        ->whereYear('created_at', now()->subMonth()->year);
                    break;
            }
        }

        // Parent category filter
        if ($request->filled('parent_category_id')) {
            $query->where('main_parent_category', $request->parent_category_id);
        }

        $childCategories = $query->ordered()->get();
        $categories = Category::active()->ordered()->get();

        return view('Admin.ChildCategory.index', compact('childCategories', 'categories'));
    }

    public function store(Request $request)
    {
        $isUpdate = $request->filled('child_category_id');
        
        // Validate input data
        $validatedData = $request->validate([
            'main_parent_category' => 'required|integer',
            'title' => 'required|string|max:255',
            'volumn' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        // Auto-assign sort order if not provided or is 0
        if (!isset($validatedData['sort_order']) || $validatedData['sort_order'] == 0) {
            $validatedData['sort_order'] = ChildCategory::getNextSortOrder();
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_child.' . $image->getClientOriginalExtension();
            $image->move(public_path('categories'), $imageName);
            $validatedData['image'] = 'categories/' . $imageName;
        }

        // Create or update the main child category
        if ($isUpdate) {
            $childCategory = ChildCategory::find($request->child_category_id);
            if (!$childCategory) {
                return redirect()->route('child.category.index')->with('error', 'Child category not found.');
            }
            $childCategory->update($validatedData);
            $message = 'Child Category Updated Successfully.';
        } else {
            $childCategory = ChildCategory::create($validatedData);
            $message = 'Child Category Added Successfully.';
        }

        return redirect()->route('child.category.index')->with('success', $message);
    }

    public function getNextSortOrder()
    {
        $nextSortOrder = ChildCategory::getNextSortOrder();
        return response()->json(['next_sort_order' => $nextSortOrder]);
    }

    public function delete(Request $request)
    {
        $category = ChildCategory::find($request->child_category);
        if ($category) {
            // Delete image if it exists
            if (!empty($category->image) && File::exists(public_path($category->image))) {
                File::delete(public_path($category->image));
            }

            // Delete related metas first
            ChildCategoryMeta::where('child_category_id', $category->id)->delete();
            $category->delete();
            return redirect()->route('child.category.index')->with('success', 'Child Category deleted successfully.');
        }
        return redirect()->route('child.category.index')->with('error', 'Something went wrong.');
    }

    public function edit(Request $request)
    {
        $id = $request->get('id');
        $childCategory = ChildCategory::with('childCategoryMeta')
            ->join('categories', 'categories.id', '=', 'child_categories.main_parent_category')
            ->select(
                'categories.title as parent_title',
                'child_categories.id',
                'child_categories.title',
                'child_categories.volumn',
                'child_categories.description',
                'child_categories.image',
                'child_categories.is_active',
                'child_categories.sort_order',
                'child_categories.main_parent_category',
                'child_categories.created_at',
                'child_categories.updated_at'
            )
            ->where('child_categories.id', $id)
            ->first();
        return response()->json($childCategory, 200, [], JSON_UNESCAPED_SLASHES);
    }

    public function update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'is_active' => 'boolean',
        ]);
        $childCategory = ChildCategory::findOrFail($id);
        $childCategory->update($validatedData);
        return response()->json(['status' => 'success', 'message' => 'Child Category Updated Successfully.']);
    }

    public function show(Request $request)
    {
        $id = $request->id;
        $child = Category::join('child_categories', 'categories.id', '=', 'child_categories.main_parent_category')
            ->select('categories.title', 'child_categories.*')
            ->where('child_categories.id', $id)
            ->get();
        return response()->json($child);
    }
}