<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Contactu;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ContactUsController extends Controller
{
    /**
     * Display a listing of the contact us messages.
     */
    public function index()
    {
        $contacts = Contactu::orderBy('created_at', 'desc')->get();
        return view('Admin.ContactUs.index', compact('contacts'));
    }

    /**
     * Display the specified contact us message.
     */
    public function show(Contactu $contact)
    {
        return view('Admin.ContactUs.show', compact('contact'));
    }

    /**
     * Show the form for editing the specified contact us message.
     */
    public function edit(Contactu $contact)
    {
        return view('Admin.ContactUs.edit', compact('contact'));
    }

    /**
     * Update the specified contact us message.
     */
    public function update(Request $request, Contactu $contact)
    {
        $request->validate([
            'f_name' => 'nullable|string|max:255',
            'l_name' => 'nullable|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'subject' => 'nullable|string|max:255',
            'message' => 'nullable|string',
            'order_id' => 'nullable|string|max:255',
        ]);
        $contact->update($request->all());
        return redirect()->route('admin.contact-us.index')
            ->with('success', 'Contact message updated successfully.');
    }

    /**
     * Remove the specified contact us message.
     */
    public function destroy(Contactu $contact)
    {
        // Delete associated file if exists
        if ($contact->file_path && Storage::disk('public')->exists($contact->file_path)) {
            Storage::disk('public')->delete($contact->file_path);
        }
        $contact->delete();
        return redirect()->route('admin.contact-us.index')
            ->with('success', 'Contact message deleted successfully.');
    }

    /**
     * Download the attached file.
     */
    public function downloadFile(Contactu $contact)
    {
        if (!$contact->file_path || !Storage::disk('public')->exists($contact->file_path)) {
            return redirect()->back()->with('error', 'File not found.');
        }
        $filePath = storage_path('app/public/' . $contact->file_path);
        return response()->download($filePath);
    }

    /**
     * Bulk delete contact messages.
     */
    public function bulkDelete(Request $request)
    {
        $request->validate([
            'contact_ids' => 'required|array',
            'contact_ids.*' => 'exists:contactus,id'
        ]);
        $contacts = Contactu::whereIn('id', $request->contact_ids)->get();
        foreach ($contacts as $contact) {
            // Delete associated file if exists
            if ($contact->file_path && Storage::disk('public')->exists($contact->file_path)) {
                Storage::disk('public')->delete($contact->file_path);
            }
            $contact->delete();
        }
        return redirect()->route('admin.contact-us.index')
            ->with('success', 'Selected contact messages deleted successfully.');
    }
}
