<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\FAQ;
use App\Models\FAQQuestion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class FAQController extends Controller
{
    /**
     * Display a listing of FAQs
     */
    public function index(Request $request)
    {
        $query = FAQ::query();
        if ($request->has('status') && $request->status !== '') {
            $query->where('status', $request->status);
        }
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('permalink', 'like', "%{$search}%")
                  ->orWhere('page_url', 'like', "%{$search}%");
            });
        }
        $faqs = $query->orderBy('sort_order', 'asc')
                     ->orderBy('created_at', 'desc')
                     ->get();
        return view('Admin.FAQ.index', compact('faqs'));
    }

    /**
     * Show the form for creating a new FAQ
     */
    public function create()
    {
        return view('Admin.FAQ.create');
    }

    /**
     * Store a newly created FAQ
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('faqs', 'permalink')->whereNull('deleted_at'),
            ],
            'page_url' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'questions' => 'required|array|min:1',
            'questions.*.question' => 'required|string|max:500',
            'questions.*.answer' => 'required|string',
            'questions.*.sort_order' => 'nullable|integer|min:0',
            'questions.*.status' => 'nullable|integer|in:0,1',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|integer|in:0,1',
        ]);
        $data = $request->except(['questions']);
        $data['created_by'] = Auth::id();
        $data['sort_order'] = $request->sort_order ?? 0;
        $faq = FAQ::create($data);
        foreach ($request->questions as $index => $questionData) {
            FAQQuestion::create([
                'faq_id' => $faq->id,
                'question' => $questionData['question'],
                'answer' => $questionData['answer'],
                'sort_order' => $questionData['sort_order'] ?? $index + 1,
                'status' => $questionData['status'] ?? 1,
                'created_by' => Auth::id(),
            ]);
        }
        $message = 'FAQ created successfully.';
        if ($request->has('stay_on_page')) {
            return redirect()->route('faqs.create')->with('success', $message);
        }
        return redirect()->route('faqs.index')->with('success', $message);
    }

    /**
     * Display the specified FAQ
     */
    public function show(FAQ $faq)
    {
        return view('Admin.FAQ.show', compact('faq'));
    }

    /**
     * Show the form for editing the specified FAQ
     */
    public function edit(FAQ $faq)
    {
        return view('Admin.FAQ.edit', compact('faq'));
    }

    /**
     * Update the specified FAQ
     */
    public function update(Request $request, FAQ $faq)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('faqs', 'permalink')->whereNull('deleted_at')->ignore($faq->id),
            ],
            'page_url' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'questions' => 'required|array|min:1',
            'questions.*.question' => 'required|string|max:500',
            'questions.*.answer' => 'required|string',
            'questions.*.sort_order' => 'nullable|integer|min:0',
            'questions.*.status' => 'nullable|integer|in:0,1',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|integer|in:0,1',
        ]);
        $data = $request->except(['questions']);
        $data['updated_by'] = Auth::id();
        $data['sort_order'] = $request->sort_order ?? 0;
        $faq->update($data);
        $faq->questions()->delete();
        foreach ($request->questions as $index => $questionData) {
            FAQQuestion::create([
                'faq_id' => $faq->id,
                'question' => $questionData['question'],
                'answer' => $questionData['answer'],
                'sort_order' => $questionData['sort_order'] ?? $index + 1,
                'status' => $questionData['status'] ?? 1,
                'created_by' => Auth::id(),
            ]);
        }
        return redirect()->route('faqs.index')->with('success', 'FAQ updated successfully.');
    }

    /**
     * Remove the specified FAQ from storage
     */
    public function destroy(FAQ $faq)
    {
        $faq->delete();
        return redirect()->route('faqs.index')->with('success', 'FAQ deleted successfully.');
    }

    /**
     * Update FAQ status
     */
    public function updateStatus(Request $request, FAQ $faq)
    {
        $request->validate([
            'status' => 'required|integer|in:0,1'
        ]);
        $faq->update(['status' => $request->status]);
        return response()->json([
            'success' => true,
            'message' => 'FAQ status updated successfully.',
            'status' => $faq->getStatusText()
        ]);
    }

    /**
     * Check permalink availability
     */
    public function checkPermalink(Request $request)
    {
        $permalink = $request->permalink;
        if (empty($permalink)) {
            return response()->json([
                'available' => false,
                'message' => 'Permalink is required.'
            ]);
        }
        $query = FAQ::where('permalink', $permalink);
        $exists = $query->whereNull('deleted_at')->exists();
        return response()->json([
            'available' => !$exists,
            'message' => $exists ? 'Permalink already exists.' : 'Permalink is available.'
        ]);
    }

    /**
     * Generate permalink from title
     */
    public function generatePermalink(Request $request)
    {
        $title = $request->title;
        if (empty($title)) {
            return response()->json(['permalink' => '']);
        }
        $permalink = strtolower($title);
        $permalink = preg_replace('/[^a-z0-9\s-]/', '', $permalink);
        $permalink = preg_replace('/\s+/', '-', $permalink);
        $permalink = preg_replace('/-+/', '-', $permalink);
        $permalink = trim($permalink, '-');
        $originalPermalink = $permalink;
        $counter = 1;
        while (FAQ::where('permalink', $permalink)->exists()) {
            $permalink = $originalPermalink . '-' . $counter;
            $counter++;
        }
        return response()->json(['permalink' => $permalink]);
    }
}