<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Job;
use App\Models\JobApplication;
use Carbon\Carbon;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class JobController extends Controller
{
    public function index(Request $request)
    {
        $query = Job::withCount('applications');

        // Keyword filter
        if ($request->filled('keyword')) {
            $query->where('title', 'like', '%' . $request->keyword . '%')
                  ->orWhere('description', 'like', '%' . $request->keyword . '%')
                  ->orWhere('location', 'like', '%' . $request->keyword . '%');
        }

        // Status filter
        if ($request->filled('status')) {
            switch ($request->status) {
                case 'active':
                    $query->where('is_active', true)
                          ->where('valid_from', '<=', now())
                          ->where('valid_until', '>=', now());
                    break;
                case 'inactive':
                    $query->where('is_active', false);
                    break;
                case 'expired':
                    $query->where('valid_until', '<', now());
                    break;
                case 'upcoming':
                    $query->where('valid_from', '>', now());
                    break;
                case 'today':
                    $query->whereDate('created_at', Carbon::today());
                    break;
                case 'yesterday':
                    $query->whereDate('created_at', Carbon::yesterday());
                    break;
                case '7days':
                    $query->where('created_at', '>=', Carbon::now()->subDays(7));
                    break;
                case '30days':
                    $query->where('created_at', '>=', Carbon::now()->subDays(30));
                    break;
                case 'this_month':
                    $query->whereMonth('created_at', Carbon::now()->month);
                    break;
                case 'last_month':
                    $query->whereMonth('created_at', Carbon::now()->subMonth()->month);
                    break;
            }
        }

        // User type filter
        if ($request->filled('user_type')) {
            $query->where('user_type', $request->user_type);
        }

        // Employment type filter
        if ($request->filled('employment_type')) {
            $query->where('employment_type', $request->employment_type);
        }

        $jobs = $query->ordered()->get(); // Get all for DataTables
        return view('Admin.Job.index', compact('jobs'));
    }

    public function create()
    {
        return view('Admin.Job.create');
    }

    public function store(Request $request)
    {
        $isUpdate = $request->filled('job_id');
        $jobId = $request->input('job_id');
        
        $rules = [
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('jobs', 'permalink'),
            ],
            'description' => 'nullable|string',
            'user_type' => 'nullable|in:driver,mover,supervisor,manager,admin,other',
            'employment_type' => 'required|in:full_time,part_time,contract,temporary,internship',
            'location' => 'nullable|string|max:255',
            'salary_min' => 'nullable|numeric|min:0',
            'salary_max' => 'nullable|numeric|min:0|gte:salary_min',
            'valid_from' => 'required|date|after_or_equal:today',
            'valid_until' => 'required|date|after:valid_from',
            'is_active' => 'nullable',
            'sort_order' => 'integer|min:0',
        ];
        
        $validated_data = $request->validate($rules);
        
        // Convert is_active to proper boolean
        $validated_data['is_active'] = (bool) $validated_data['is_active'];
        
        // Auto-assign sort order if not provided or is 0
        if (!isset($validated_data['sort_order']) || $validated_data['sort_order'] == 0) {
            $validated_data['sort_order'] = Job::getNextSortOrder();
        }
        
        // Save or update job
        if ($isUpdate) {
            $job = Job::findOrFail($jobId);
            $job->update($validated_data);
            $message = 'Job updated successfully.';
        } else {
            $job = Job::create($validated_data);
            $message = 'Job created successfully.';
        }
        
        return response()->json([
            'status' => 'success',
            'message' => $message
        ]);
    }

    public function delete(Request $request)
    {
        $id = $request->id;
        $job = Job::find($id);
        if ($job) {
            // Delete all related job applications
            JobApplication::where('job_id', $job->id)->delete();

            // Delete the job
            $job->delete();

            return redirect()->route('admin.jobs.index')->with('success', 'Job deleted successfully.');
        }

        return redirect()->route('admin.jobs.index')->with('error', 'Something went wrong.');
    }

    public function edit(Request $request, $id = null)
    {
        // Handle both old route (with request id) and new route (with parameter id)
        $jobId = $id ?? $request->id;
        $job = Job::find($jobId);
        
        if (!$job) {
            return redirect()->route('admin.jobs.index')->with('error', 'Job not found.');
        }

        return view('Admin.Job.create', compact('job'));
    }

    public function show(Request $request)
    {
        $job = Job::withCount('applications')->find($request->id);

        if (!$job) {
            return response()->json(['error' => 'Job not found'], 404);
        }

        return response()->json([
            'id' => $job->id,
            'title' => $job->title,
            'description' => $job->description,
            'user_type' => $job->user_type,
            'employment_type' => $job->employment_type,
            'location' => $job->location,
            'salary_min' => $job->salary_min,
            'salary_max' => $job->salary_max,
            'valid_from' => $job->valid_from->format('Y-m-d'),
            'valid_until' => $job->valid_until->format('Y-m-d'),
            'is_active' => $job->is_active,
            'sort_order' => $job->sort_order,
            'applications_count' => $job->applications_count,
        ]);
    }

    public function update(Request $request, $id)
    {
        // Validate the incoming request data
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('jobs', 'permalink')->ignore($id),
            ],
            'description' => 'nullable|string',
            'user_type' => 'nullable|in:driver,mover,supervisor,manager,admin,other',
            'employment_type' => 'required|in:full_time,part_time,contract,temporary,internship',
            'location' => 'nullable|string|max:255',
            'salary_min' => 'nullable|numeric|min:0',
            'salary_max' => 'nullable|numeric|min:0|gte:salary_min',
            'valid_from' => 'required|date|after_or_equal:today',
            'valid_until' => 'required|date|after:valid_from',
            'is_active' => 'nullable',
            'sort_order' => 'integer|min:0',
        ]);
        
        // Check for validation errors
        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'message' => $validator->errors()], 422);
        }
        
        // Get validated data and convert is_active to boolean
        $validated_data = $validator->validated();
        $validated_data['is_active'] = (bool) $validated_data['is_active'];
        
        // Find the job to update
        $job = Job::find($id);
        if (!$job) {
            return response()->json(['status' => 'error', 'message' => 'Job not found.'], 404);
        }
        
        // Update the job
        $job->update($validated_data);
        
        return response()->json(['status' => 'success', 'message' => 'Job updated successfully.']);
    }

    public function changeStatus(Request $request)
    {
        $job = Job::find($request->id);
        if ($job) {
            $job->is_active = !$job->is_active;
            $job->save();
            
            return response()->json([
                'status' => 'success',
                'message' => 'Job status updated successfully.',
                'is_active' => $job->is_active
            ]);
        }
        
        return response()->json(['status' => 'error', 'message' => 'Job not found.'], 404);
    }

    public function applications(Request $request, $jobId)
    {
        $job = Job::findOrFail($jobId);
        $query = $job->applications();

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Keyword filter
        if ($request->filled('keyword')) {
            $query->where('full_name', 'like', '%' . $request->keyword . '%')
                  ->orWhere('email', 'like', '%' . $request->keyword . '%');
        }

        $applications = $query->ordered()->get();
        
        return view('Admin.Job.applications', compact('job', 'applications'));
    }

    public function updateApplicationStatus(Request $request, $applicationId)
    {
        $application = JobApplication::find($applicationId);
        if ($application) {
            $application->status = $request->status;
            $application->notes = $request->notes;
            $application->save();
            
            return response()->json([
                'status' => 'success',
                'message' => 'Application status updated successfully.'
            ]);
        }
        
        return response()->json(['status' => 'error', 'message' => 'Application not found.'], 404);
    }

    public function allApplications(Request $request)
    {
        $query = JobApplication::with('job');

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Job filter
        if ($request->filled('job_id')) {
            $query->where('job_id', $request->job_id);
        }

        // Keyword filter
        if ($request->filled('keyword')) {
            $query->where('full_name', 'like', '%' . $request->keyword . '%')
                  ->orWhere('email', 'like', '%' . $request->keyword . '%');
        }

        // Date filter
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $applications = $query->latest()->get();
        $jobs = Job::active()->ordered()->get();

        return view('Admin.Job.allApplications', compact('applications', 'jobs'));
    }

    public function showApplication(Request $request)
    {
        $application = JobApplication::with('job')->find($request->id);
        
        if (!$application) {
            return response()->json(['status' => 'error', 'message' => 'Application not found.'], 404);
        }

        $html = '
        <div class="row">
            <div class="col-md-6">
                <h6>Applicant Information</h6>
                <table class="table table-sm">
                    <tr><td><strong>Name:</strong></td><td>' . $application->full_name . '</td></tr>
                    <tr><td><strong>Email:</strong></td><td>' . $application->email . '</td></tr>
                    <tr><td><strong>Phone:</strong></td><td>' . ($application->phone ?? 'Not provided') . '</td></tr>
                    <tr><td><strong>Applied Date:</strong></td><td>' . $application->created_at->format('M d, Y H:i') . '</td></tr>
                    <tr><td><strong>Status:</strong></td><td><span class="badge bg-info">' . ucfirst($application->status) . '</span></td></tr>
                </table>
            </div>
            <div class="col-md-6">
                <h6>Job Information</h6>
                <table class="table table-sm">
                    <tr><td><strong>Job Title:</strong></td><td>' . $application->job->title . '</td></tr>
                    <tr><td><strong>Location:</strong></td><td>' . ($application->job->location ?? 'Not specified') . '</td></tr>
                    <tr><td><strong>Employment Type:</strong></td><td>' . ucfirst(str_replace('_', ' ', $application->job->employment_type)) . '</td></tr>
                    <tr><td><strong>User Type:</strong></td><td>' . ucfirst($application->job->user_type) . '</td></tr>
                </table>
            </div>
        </div>
        
        <div class="row mt-3">
            <div class="col-12">
                <h6>Cover Letter</h6>
                <div class="border p-3 rounded" style="max-height: 200px; overflow-y: auto;">
                    ' . nl2br(e($application->cover_letter)) . '
                </div>
            </div>
        </div>
        
        <div class="row mt-3">
            <div class="col-12">
                <h6>CV File</h6>
                ' . ($application->cv_file ? 
                    '<a href="' . asset('job_applications/' . $application->cv_file) . '" target="_blank" class="btn btn-primary btn-sm">
                        <i class="ri-download-line"></i> Download CV
                    </a>' : 
                    '<span class="text-muted">No CV file uploaded</span>') . '
            </div>
        </div>
        
        <div class="row mt-3">
            <div class="col-12">
                <div class="alert alert-info">
                    <i class="fa-solid fa-info-circle me-2"></i>
                    <strong>Note:</strong> To update the application status, use the Actions dropdown menu in the table.
                </div>
            </div>
        </div>
        ';

        return response($html);
    }

    public function getNextSortOrder()
    {
        $nextSortOrder = Job::getNextSortOrder();
        return response()->json(['next_sort_order' => $nextSortOrder]);
    }

    public function checkPermalink(Request $request)
    {
        $permalink = $request->input('permalink');
        $excludeId = $request->input('exclude_id');
        
        if (empty($permalink)) {
            return response()->json([
                'available' => false,
                'message' => 'Permalink is required.'
            ]);
        }
        
        $query = Job::where('permalink', $permalink);
        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }
        
        $exists = $query->exists();
        
        return response()->json([
            'available' => !$exists,
            'message' => $exists ? 'Permalink already exists.' : 'Permalink is available.'
        ]);
    }

    public function generatePermalink(Request $request)
    {
        $title = $request->title;
        if (empty($title)) {
            return response()->json(['permalink' => '']);
        }
        
        $permalink = strtolower($title);
        $permalink = preg_replace('/[^a-z0-9\s-]/', '', $permalink);
        $permalink = preg_replace('/\s+/', '-', $permalink);
        $permalink = preg_replace('/-+/', '-', $permalink);
        $permalink = trim($permalink, '-');
        
        $originalPermalink = $permalink;
        $counter = 1;
        while (Job::where('permalink', $permalink)->exists()) {
            $permalink = $originalPermalink . '-' . $counter;
            $counter++;
        }
        
        return response()->json(['permalink' => $permalink]);
    }
}