<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Service;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rule;

class ServiceController extends Controller
{
    public function index(Request $request)
    {
        $query = Service::query();
        if ($request->filled('status')) {
            $query->where('active', $request->status);
        }
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('description', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('content', 'LIKE', "%{$searchTerm}%");
            });
        }
        if ($request->filled('date_range')) {
            $dateRange = $request->date_range;
            $now = Carbon::now();
            switch ($dateRange) {
                case 'today':
                    $query->whereDate('created_at', $now->toDateString());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [
                        $now->startOfWeek()->toDateTimeString(),
                        $now->endOfWeek()->toDateTimeString()
                    ]);
                    break;
                case 'month':
                    $query->whereMonth('created_at', $now->month)
                        ->whereYear('created_at', $now->year);
                    break;
                case 'year':
                    $query->whereYear('created_at', $now->year);
                    break;
            }
        }
        $services = $query->orderBy('created_at', 'desc')->get();
        return view('Admin.Service.index', compact('services'));
    }

    public function create()
    {
        return view('Admin.Service.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('services', 'permalink')->whereNull('deleted_at'),
            ],
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'featured_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'active' => 'required|integer|in:0,1,2,3',
            'fimage_alt_text' => 'nullable|string',
            '_seo_title' => 'nullable|string|max:255',
            '_seo_tags' => 'nullable|string',
            '_seo_description' => 'nullable|string',
            '_canonical_url' => 'nullable|url',
            '_schema_markup' => 'nullable|string',
            '_page_index' => 'nullable|integer',
        ]);
        $data = $request->except(['cover_image', 'featured_images']);
        $data['created_by'] = Auth::id();
        $coverDir = public_path('services/covers');
        $featuredDir = public_path('services/featured');
        if (!File::exists($coverDir)) {
            File::makeDirectory($coverDir, 0755, true);
        }
        if (!File::exists($featuredDir)) {
            File::makeDirectory($featuredDir, 0755, true);
        }
        if ($request->hasFile('cover_image')) {
            $coverImage = $request->file('cover_image');
            $coverImageName = 'service_cover_' . time() . '_' . uniqid() . '.' . $coverImage->getClientOriginalExtension();
            $coverImage->move($coverDir, $coverImageName);
            $data['cover_image'] = 'services/covers/' . $coverImageName;
        }
        $featuredImagePaths = [];
        if ($request->hasFile('featured_images')) {
            foreach ($request->file('featured_images') as $index => $image) {
                $imageName = 'service_featured_' . time() . '_' . $index . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $image->move($featuredDir, $imageName);
                $featuredImagePaths[] = 'services/featured/' . $imageName;
            }
        }
        $data['featured_images'] = implode(',', $featuredImagePaths);
        Service::create($data);
        $message = 'Service created successfully.';
        if ($request->has('stay_on_page')) {
            return redirect()->route('services.create')->with('success', $message);
        }
        return redirect()->route('services.index')->with('success', $message);
    }

    public function show(Service $service)
    {
        return view('Admin.Service.show', compact('service'));
    }

    public function edit(Service $service)
    {
        return view('Admin.Service.edit', compact('service'));
    }

    public function update(Request $request, Service $service)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('services', 'permalink')->whereNull('deleted_at')->ignore($service->id),
            ],
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'featured_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'active' => 'required|integer|in:0,1,2,3',
            'fimage_alt_text' => 'nullable|string',
            '_seo_title' => 'nullable|string|max:255',
            '_seo_tags' => 'nullable|string',
            '_seo_description' => 'nullable|string',
            '_canonical_url' => 'nullable|url',
            '_schema_markup' => 'nullable|string',
            '_page_index' => 'nullable|integer',
        ]);

        $data = $request->except(['cover_image', 'featured_images']);
        $data['updated_by'] = Auth::id();
        $coverDir = public_path('services/covers');
        $featuredDir = public_path('services/featured');
        if (!File::exists($coverDir)) {
            File::makeDirectory($coverDir, 0755, true);
        }
        if (!File::exists($featuredDir)) {
            File::makeDirectory($featuredDir, 0755, true);
        }
        if ($request->hasFile('cover_image')) {
            if ($service->cover_image && File::exists(public_path($service->cover_image))) {
                File::delete(public_path($service->cover_image));
            }
            $coverImage = $request->file('cover_image');
            $coverImageName = 'service_cover_' . time() . '_' . uniqid() . '.' . $coverImage->getClientOriginalExtension();
            $coverImage->move($coverDir, $coverImageName);
            $data['cover_image'] = 'services/covers/' . $coverImageName;
        }
        if ($request->hasFile('featured_images')) {
            if ($service->featured_images) {
                $oldImages = explode(',', $service->featured_images);
                foreach ($oldImages as $oldImage) {
                    if (File::exists(public_path($oldImage))) {
                        File::delete(public_path($oldImage));
                    }
                }
            }
            $featuredImagePaths = [];
            foreach ($request->file('featured_images') as $index => $image) {
                $imageName = 'service_featured_' . time() . '_' . $index . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $image->move($featuredDir, $imageName);
                $featuredImagePaths[] = 'services/featured/' . $imageName;
            }
            $data['featured_images'] = implode(',', $featuredImagePaths);
        }
        $service->update($data);
        $message = 'Service updated successfully.';
        if ($request->has('stay_on_page')) {
            return redirect()->route('services.edit', $service)->with('success', $message);
        }
        return redirect()->route('services.index')->with('success', $message);
    }

    public function destroy(Service $service)
    {
        // Delete associated images
        if ($service->cover_image && File::exists(public_path($service->cover_image))) {
            File::delete(public_path($service->cover_image));
        }
        if ($service->featured_images) {
            $images = explode(',', $service->featured_images);
            foreach ($images as $image) {
                if (File::exists(public_path($image))) {
                    File::delete(public_path($image));
                }
            }
        }
        $service->delete();
        return redirect()->route('services.index')->with('success', 'Service moved to trash successfully.');
    }

    public function updateStatus(Request $request, Service $service)
    {
        $request->validate([
            'active' => 'required|integer|in:0,1,2,3'
        ]);
        $service->update(['active' => $request->active]);
        return response()->json([
            'success' => true,
            'message' => 'Service status updated successfully.',
            'status' => $service->getStatusText()
        ]);
    }

   

    public function restore($id)
    {
        $service = Service::withTrashed()->findOrFail($id);
        $service->restore();
        return redirect()->route('services.trash')->with('success', 'Service restored successfully.');
    }

    public function checkPermalink(Request $request)
    {
        $permalink = $request->input('permalink');
        
        if (empty($permalink)) {
            return response()->json([
                'available' => false,
                'message' => 'Permalink is required.'
            ]);
        }
        $query = Service::where('permalink', $permalink);
        $exists = $query->whereNull('deleted_at')->exists();
        return response()->json([
            'available' => !$exists,
            'message' => $exists ? 'Permalink already exists' : 'Permalink is available'
        ]);
    }
}