<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\EmailController;
use App\Http\Controllers\Controller;
use App\Http\Controllers\API\Traits\EmailTrait;
use App\Models\GeneralSettings;
use App\Models\User;
use App\Models\UserMeta;
use App\Models\Quote;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rule;


class UserController extends Controller
{
    use EmailTrait;
    // Removed index() - customers() is the main listing method

    public function edit(User $User)
    {
        // Only allow editing customers (role_id = 2)
        if ($User->role_id != 2) {
            abort(403, 'Only customers can be edited here.');
        }
        $User->load('userMeta', 'quotes', 'orders');
        return view('Admin.Users.edit', compact('User'));
    }

    public function update(Request $request, $id)
    {
        $user = User::find($request->user_id);
        if (!$user || $user->role_id != 2) {
            return response()->json([
                'status' => 404,
                'message' => 'Customer not found.',
            ]);
        }

        $sanitizedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'email',
                Rule::unique('users')->ignore($id),
            ],
            'phone' => 'required|string|max:20',
        ]);

        $sanitizedData['profile_image'] = '/user.jpg';

        if (isset($request->created_by)) {
            $sanitizedData['created_by'] = $request->created_by;
            $sanitizedData['email_verified_at'] = now();
            $sanitizedData['active'] = 1;
        }

        // Handle password update if provided
        $plainPassword = null;
        if ($request->has('password') && !empty($request->password)) {
            $plainPassword = $request->password; // Store plain password for email
            $sanitizedData['password'] = Hash::make($request->password);
        }

        $user->update($sanitizedData);
        $metaDescriptions = $request->only([
            '_b_name',
            '_b_email',
            '_b_number',
            '_b_street_address',
            '_b_area_address',
        ]);

        $userMeta = [];
        foreach ($metaDescriptions as $option => $value) {
            if ($value !== null) {
                $userMeta[] = [
                    'user_id' => $user->id,
                    'option' => $option,
                    'value' => $value,
                ];
            }
        }

        # Delete previous user meta descriptions
        DB::table('user_metas')->where('user_id', $user->id)->delete();
        DB::table('user_metas')->insert($userMeta);

        $data = GeneralSettings::where('option', '_header_logo')->orWhere('option', '_b_email')->orWhere('option', '_instagram_link')->orWhere('option', '_facebook_link')->orWhere('option', '_twitter_link')->orWhere('option', '_youtube_link')->get()->toArray();

        $header_logo_index = array_search('_header_logo', array_column($data, 'option'));
        $logo = $data[$header_logo_index]['value'];

        if ($user) {
            if (isset($request->confirmation_email_sending)) {
                // Send account update email using EmailTrait with account-login-details template
                try {
                    $settings = $this->getGeneralSettingsData();
                    // Get password - use the plain password if it was updated, otherwise show message
                    $password = $plainPassword ?? '[Your existing password remains unchanged]';
                    
                    // Get the email address to send to (prefer _b_email from userMeta, fallback to user email)
                    $emailTo = $user->email ?? $request->_b_email;
                    
                    // Use the same account-login-details template as create
                    $params = [
                        'subject' => 'Account Updated Successfully | ' . $settings['business_name'],
                        'to' => $emailTo,
                        'msg' => view('Admin.Emails.account-login-details', [
                            'name' => $user->name,
                            'email' => $user->email,
                            'password' => $password,
                            'settings' => $settings,
                        ])->render(),
                    ];

                    $emailSent = $this->SendInstantEmail($params);

                    if ($emailSent) {
                        return response()->json([
                            'status' => 200,
                            'message' => 'Customer Updated successfully, An Email has been sent to customer email for account information.',
                        ]);
                    } else {
                        // Check logs for more details - the error is already logged in SendInstantEmail
                        \Illuminate\Support\Facades\Log::error('Account update email failed for user', [
                            'user_id' => $user->id,
                            'email' => $emailTo
                        ]);
                        
                        // Check if it's a configuration issue
                        $emailSettings = \App\Models\GeneralSettings::get()->pluck('value', 'option')->toArray();
                        $mailFromAddress = $emailSettings['_mail_from_address'] ?? null;
                        $mailPassword = $emailSettings['_mail_password'] ?? null;
                        
                        $errorMsg = 'Customer Updated successfully, but failed to send email. ';
                        if (empty($mailFromAddress) || empty($mailPassword)) {
                            $errorMsg .= 'Please configure Mail From Address and Mail Password in General Settings.';
                        } else {
                            $errorMsg .= 'The SendGrid API key in General Settings (Mail Password field) appears to be expired, revoked, or invalid. Please update it with a valid SendGrid API key from your SendGrid account.';
                        }
                        
                        return response()->json([
                            'status' => 500,
                            'message' => $errorMsg,
                        ]);
                    }
                } catch (\Exception $e) {
                    \Illuminate\Support\Facades\Log::error('Exception while sending account update email', [
                        'user_id' => $user->id,
                        'email' => $user->email,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    return response()->json([
                        'status' => 500,
                        'message' => 'Customer Updated successfully, but an error occurred while sending the email: ' . $e->getMessage(),
                    ]);
                }
            }

            return response()->json([
                'status' => 200,
                'message' => 'Customer Updated successfully.',
            ]);
        } else {
            return response()->json([
                'status' => 500,
                'message' => 'An Error Occurred, During Updated customer account.',
            ]);
        }
    }

    public function customers()
    {
        // Get all users for the created_by lookup (admin users)
        $adminUsers = User::where('role_id', '!=', 2)->get();
        
        // Get customers (role_id = 2) with quotes and orders count
        $customers = User::where('role_id', 2)
            ->withCount(['quotes', 'orders'])
            ->latest()
            ->get();
        
        return view('Admin.Users.customer', compact('adminUsers', 'customers'));
    }

    public function create()
    {
        // Get admin users for created_by dropdown
        $adminUsers = User::where('role_id', '!=', 2)->get();
        return view('Admin.Users.create', compact('adminUsers'));
    }

    public function show(User $user)
    {
        // Only show customers (role_id = 2)
        if ($user->role_id != 2) {
            return response()->json([
                'status' => 404,
                'message' => 'Customer not found.',
            ], 404);
        }

        $adminUsers = User::where('role_id', '!=', 2)->get();
        $UserMeta = UserMeta::where('user_id', $user->id)->get();
        
        // Get quotes and orders for this customer
        $quotes = Quote::where('user_id', $user->id)
            ->with(['truck', 'booking'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Separate quotes and orders
        $customerQuotes = $quotes->where('quote_type', '!=', 'order');
        $customerOrders = $quotes->where('quote_type', 'order');
        
        return response()->json([
            'status' => 200,
            'user_details' => $user,
            'users' => $adminUsers,
            'user_meta' => $UserMeta,
            'quotes' => $customerQuotes->values(),
            'orders' => $customerOrders->values(),
            'quotes_count' => $customerQuotes->count(),
            'orders_count' => $customerOrders->count(),
        ]);
    }

    public function store(Request $request)
    {
        $sanitizedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                Rule::unique('users'),
            ],
            'password' => 'required|min:8',
            'phone' => 'required|string|max:20',
        ]);

        $sanitizedData['profile_image'] = '/user.jpg';
        $sanitizedData['role_id'] = 2; // Always set to customer (role_id = 2)

        if (isset($request->created_by)) {
            $sanitizedData['created_by'] = $request->created_by;
            $sanitizedData['email_verified_at'] = now();
            $sanitizedData['active'] = 1;
        }

        $sanitizedData['password'] = Hash::make($sanitizedData['password']);

        $user = User::create($sanitizedData);
        $metaDescriptions = $request->only([
            '_b_name',
            '_b_email',
            '_b_number',
            '_b_street_address',
            '_b_area_address',
        ]);

        $userMeta = [];
        foreach ($metaDescriptions as $option => $value) {
            if ($value !== null) {
                $userMeta[] = [
                    'user_id' => $user->id,
                    'option' => $option,
                    'value' => $value,
                ];
            }
        }

        # Insert user meta descriptions in bulk
        DB::table('user_metas')->insert($userMeta);

        $data = GeneralSettings::where('option', '_header_logo')->orWhere('option', '_b_email')->orWhere('option', '_instagram_link')->orWhere('option', '_facebook_link')->orWhere('option', '_twitter_link')->orWhere('option', '_youtube_link')->get()->toArray();

        $admin_email_index = array_search('_b_email', array_column($data, 'option'));
        $header_logo_index = array_search('_header_logo', array_column($data, 'option'));
        $insta_link_index = array_search('_instagram_link', array_column($data, 'option'));
        $fb_link_index = array_search('_facebook_link', array_column($data, 'option'));
        $twitter_link_index = array_search('_twitter_link', array_column($data, 'option'));
        $youtube_link_index = array_search('_youtube_link', array_column($data, 'option'));

        $admin_email = $data[$admin_email_index]['value'];
        $logo = $data[$header_logo_index]['value'];
        $instagram_link = $data[$insta_link_index]['value'];
        $facebook_link = $data[$fb_link_index]['value'];
        $twitter_link = $data[$twitter_link_index]['value'];
        $youtube_link = $data[$youtube_link_index]['value'];

        if ($user) {
            if (isset($request->confirmation_email_sending)) {
                // Send account creation email using EmailTrait
                try {
                    $emailSent = $this->sendAccountCreationEmail($user, $request->password);
                    if ($emailSent) {
                        return response()->json([
                            'status' => 200,
                            'message' => 'Customer created successfully, An Email has been sent to customer email for account information.',
                        ]);
                    } else {
                        // Check logs for more details - the error is already logged in SendInstantEmail
                        \Illuminate\Support\Facades\Log::error('Account creation email failed for user', [
                            'user_id' => $user->id,
                            'email' => $user->email
                        ]);
                        
                        // Check if it's a configuration issue
                        $settings = \App\Models\GeneralSettings::get()->pluck('value', 'option')->toArray();
                        $mailFromAddress = $settings['_mail_from_address'] ?? null;
                        $mailPassword = $settings['_mail_password'] ?? null;
                        
                        $errorMsg = 'Customer created successfully, but failed to send email. ';
                        if (empty($mailFromAddress) || empty($mailPassword)) {
                            $errorMsg .= 'Please configure Mail From Address and Mail Password in General Settings.';
                        } else {
                            $errorMsg .= 'The SendGrid API key in General Settings (Mail Password field) appears to be expired, revoked, or invalid. Please update it with a valid SendGrid API key from your SendGrid account.';
                        }
                        
                        return response()->json([
                            'status' => 500,
                            'message' => $errorMsg,
                        ]);
                    }
                } catch (\Exception $e) {
                    \Illuminate\Support\Facades\Log::error('Exception while sending account creation email', [
                        'user_id' => $user->id,
                        'email' => $user->email,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    return response()->json([
                        'status' => 500,
                        'message' => 'Customer created successfully, but an error occurred while sending the email: ' . $e->getMessage(),
                    ]);
                }
            }

            return response()->json([
                'status' => 200,
                'message' => 'Customer created successfully.',
            ]);
        } else {
            return response()->json([
                'status' => 500,
                'message' => 'An Error Occurred, During creating customer account.',
            ]);
        }
    }

    public function destroy(User $user)
    {
        // Only allow deleting customers
        if ($user->role_id != 2) {
            abort(403, 'Only customers can be deleted here.');
        }

        $user->delete();

        return redirect()->route('Customer.List')->with('success', 'Customer deleted successfully.');
    }

    public function updateStatus(Request $request, User $user)
    {
        // Only allow updating customer status
        if ($user->role_id != 2) {
            return response()->json([
                'status' => 403,
                'message' => 'Only customers can be updated here.',
            ], 403);
        }

        $request->validate([
            'active' => 'required',
        ]);

        $user->active = $request->active;
        $user->save();

        return response()->json([
            'status' => 200,
            'message' => 'Customer status updated successfully.',
        ]);
    }

    public function showUserProfile()
    {
        $user = auth()->user();
        return view('Admin.Users.profile')->with(['user' => $user]);
    }

    public function updateProfile(Request $request)
    {
        $user = auth()->user();
        $updateType = $request->input('update_type', '');

        // Validate general inputs if any are required
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|max:255',
            'phone' => 'sometimes|string|max:20',
        ]);

        // Handle password update
        if ($updateType === 'password') {
            $request->validate([
                'old_password' => ['required', function ($attribute, $value, $fail) use ($user) {
                    if (!Hash::check($value, $user->password)) {
                        $fail('The current password is incorrect.');
                    }
                }],
                'new_password' => [
                    'required', 
                    'confirmed', 
                    'min:8', // Minimum length of 8 characters
                    'regex:/[a-z]/', // Requires at least one lowercase letter
                    'regex:/[A-Z]/', // Requires at least one uppercase letter
                    'regex:/[0-9]/', // Requires at least one number
                    'regex:/[@$!%*?&]/', // Requires at least one special character
                ],
            ]);
        
            // Update the user's password
            $user->update([
                'password' => Hash::make($request->new_password),
            ]);

            return redirect()->back()->with('success', 'Password updated successfully.');
        }

        // Check if the profile image is being uploaded
        if ($request->hasFile('profile_image')) {
            $request->validate([
                'profile_image' => ['required', 'image', 'max:2048'], // 2MB max size
            ]);

            // Create directory inside the public folder if it doesn't exist
            $profileImagePath = public_path('profile_images');
            if (!File::exists($profileImagePath)) {
                File::makeDirectory($profileImagePath, 0755, true); // Create directory with permissions
            }

            // Delete old image if exists
            if ($user->profile_image && File::exists(public_path($user->profile_image))) {
                File::delete(public_path($user->profile_image)); // Delete the old image
            }

            // Store new image in the public folder
            $imageName = time() . '_' . $user->id . '.' . $request->profile_image->extension();
            $request->profile_image->move($profileImagePath, $imageName); // Move image to the directory

            // Update user's profile image
            $user->update([
                'profile_image' => 'profile_images/'. $imageName,
            ]);

            return redirect()->back()->with('success', 'Profile image updated successfully.');
        }

        // Handle other updates (e.g., name, email, phone)
        $user->update([
            'name' => $request->input('name', $user->name),
            'email' => $request->input('email', $user->email),
            'phone' => $request->input('phone', $user->phone),
        ]);

        return redirect()->back()->with('success', 'Profile updated successfully.');
    }
}
