<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AssembleDisassembleCategory extends Model
{
    use HasFactory;
    
    protected $fillable = [
        'name',
        'description',
        'image',
        'assemble_price',
        'disassemble_price',
        'both_price',
        'assemble_time_minutes',
        'disassemble_time_minutes',
        'both_time_minutes',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'assemble_price' => 'decimal:2',
        'disassemble_price' => 'decimal:2',
        'both_price' => 'decimal:2',
        'assemble_time_minutes' => 'integer',
        'disassemble_time_minutes' => 'integer',
        'both_time_minutes' => 'integer',
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    // Relationships
    public function bookingItems()
    {
        return $this->morphMany(BookingItem::class, 'entity');
    }

    // Scope for active categories
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope for ordering
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order');
    }

    // Get next sort order for new categories
    public static function getNextSortOrder()
    {
        $lastCategory = self::orderBy('sort_order', 'desc')->first();
        return $lastCategory ? $lastCategory->sort_order + 1 : 1;
    }

    // Get formatted price for assemble
    public function getFormattedAssemblePriceAttribute()
    {
        return '$' . number_format($this->assemble_price, 2);
    }

    // Get formatted price for disassemble
    public function getFormattedDisassemblePriceAttribute()
    {
        return '$' . number_format($this->disassemble_price, 2);
    }

    // Get formatted price for both
    public function getFormattedBothPriceAttribute()
    {
        return '$' . number_format($this->both_price, 2);
    }

    // Get estimated time in readable format for assemble
    public function getAssembleTimeFormattedAttribute()
    {
        if (!$this->assemble_time_minutes) {
            return 'Not specified';
        }
        
        $hours = floor($this->assemble_time_minutes / 60);
        $minutes = $this->assemble_time_minutes % 60;
        
        if ($hours > 0 && $minutes > 0) {
            return "{$hours}h {$minutes}m";
        } elseif ($hours > 0) {
            return "{$hours}h";
        } else {
            return "{$minutes}m";
        }
    }

    // Get estimated time in readable format for disassemble
    public function getDisassembleTimeFormattedAttribute()
    {
        if (!$this->disassemble_time_minutes) {
            return 'Not specified';
        }
        
        $hours = floor($this->disassemble_time_minutes / 60);
        $minutes = $this->disassemble_time_minutes % 60;
        
        if ($hours > 0 && $minutes > 0) {
            return "{$hours}h {$minutes}m";
        } elseif ($hours > 0) {
            return "{$hours}h";
        } else {
            return "{$minutes}m";
        }
    }

    // Get estimated time in readable format for both
    public function getBothTimeFormattedAttribute()
    {
        if (!$this->both_time_minutes) {
            return 'Not specified';
        }
        
        $hours = floor($this->both_time_minutes / 60);
        $minutes = $this->both_time_minutes % 60;
        
        if ($hours > 0 && $minutes > 0) {
            return "{$hours}h {$minutes}m";
        } elseif ($hours > 0) {
            return "{$hours}h";
        } else {
            return "{$minutes}m";
        }
    }
}
