<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Booking extends Model
{
    use HasFactory;

    protected $fillable = [
        'uuid',
        'user_id',
        'user_type',
        'name',
        'email',
        'phone',
        'pick_up_property_id',
        'drop_off_property_id',
        'status',
        'move_date',
        'time_preference',
        'custom_time',
        'reason_to_move',
        'booking_item_ids',
        'difficult_items_notes',
        'truck_id',
        'truck_booking_hours',
        'is_completed',
    ];

    protected $casts = [
        'move_date' => 'date',
        'booking_item_ids' => 'array',
        'truck_booking_hours' => 'decimal:2',
        'is_completed' => 'boolean',
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function pickUpProperty()
    {
        return $this->belongsTo(Property::class, 'pick_up_property_id');
    }

    public function dropOffProperty()
    {
        return $this->belongsTo(Property::class, 'drop_off_property_id');
    }

    public function truck()
    {
        return $this->belongsTo(Truck::class);
    }

    public function bookingItems()
    {
        return $this->hasMany(BookingItem::class, 'uuid', 'uuid');
    }

    public function quote()
    {
        return $this->hasOne(Quote::class, 'booking_id', 'id');
    }

    // Scopes
    public function scopeCompleted($query)
    {
        return $query->where('is_completed', true);
    }

    public function scopePending($query)
    {
        return $query->where('is_completed', false);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByMoveDate($query, $date)
    {
        return $query->where('move_date', $date);
    }

    public function scopeByTimePreference($query, $preference)
    {
        return $query->where('time_preference', $preference);
    }

    public function scopeWithTruck($query)
    {
        return $query->whereNotNull('truck_id');
    }


    // Accessors
    public function getTimePreferenceTextAttribute()
    {
        if ($this->time_preference === 'Custom' && $this->custom_time) {
            return $this->custom_time;
        }
        return ucfirst($this->time_preference);
    }

    public function getStatusTextAttribute()
    {
        return ucfirst($this->status);
    }

    public function getTotalItemsAttribute()
    {
        return $this->bookingItems()->sum('quantity');
    }

    // Methods
    public function calculateTotalPrice()
    {
        if (!$this->truck) {
            return 0;
        }

        $totalVolume = $this->bookingItems()->with('entity')->get()->sum(function ($item) {
            return $item->quantity * ($item->getVolume());
        });

        $totalDistance = $this->calculateDistance();

        return $this->truck->calculatePrice($totalVolume, $totalDistance, $this->truck_booking_hours ?? 0);
    }

    public function calculateDistance()
    {
        // This would typically integrate with a mapping service
        // For now, return a default or calculate based on locations
        return 0; // Placeholder
    }

    public function markAsCompleted()
    {
        $this->update(['is_completed' => true, 'status' => 'completed']);
    }

    public function assignTruck($truckId)
    {
        $this->update(['truck_id' => $truckId]);
    }

    // Guest user helpers
    public function isGuestBooking()
    {
        return $this->user_type === 'guest' || $this->user_id === null;
    }

    public function isRegisteredUserBooking()
    {
        return $this->user_type === 'registered' && $this->user_id !== null;
    }

    public function getCustomerNameAttribute()
    {
        if ($this->isGuestBooking()) {
            return $this->name;
        }
        return $this->user->name ?? $this->name;
    }

    public function getCustomerEmailAttribute()
    {
        if ($this->isGuestBooking()) {
            return $this->email;
        }
        return $this->user->email ?? $this->email;
    }

    public function getCustomerPhoneAttribute()
    {
        if ($this->isGuestBooking()) {
            return $this->phone;
        }
        return $this->user->phone ?? $this->phone;
    }

    public function getTruckPricingAttribute()
    {
        if (!$this->truck) {
            return null;
        }

        $truck = $this->truck;
        
        // Basic truck info
        $pricingData = [
            'id' => $truck->id,
            'truck_name' => $truck->truck_name,
            'truck_type' => $truck->truck_type,
            'cubic_capacity' => $truck->cubic_capacity,
            'pricing_type' => $truck->pricing_type,
            'number_of_movers' => $truck->number_of_movers,
            'price_per_mover' => $truck->price_per_mover,
        ];

        // If custom pricing, get from truck table
        if ($truck->pricing_type === 'custom') {
            $pricingData['pricing'] = [
                'fixed_price' => $truck->custom_fixed_price,
                'per_cubic_meter_price' => $truck->custom_per_cubic_meter_price,
                'per_kilometer_price' => $truck->custom_per_kilometer_price,
                'minimum_distance' => $truck->custom_minimum_distance,
                'maximum_distance' => $truck->custom_maximum_distance,
            ];
        } else {
            // Global pricing from general settings
            $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
            $pricingData['pricing'] = [
                'fixed_price' => $generalSettings['_truck_fixed_price'] ?? 0,
                'per_cubic_meter_price' => $generalSettings['_truck_per_cubic_meter_price'] ?? 0,
                'per_kilometer_price' => $generalSettings['_truck_per_kilometer_price'] ?? 0,
                'minimum_distance' => $generalSettings['_truck_minimum_distance'] ?? 0,
                'maximum_distance' => $generalSettings['_truck_maximum_distance'] ?? 0,
            ];
        }

        return $pricingData;
    }
}