<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class BookingItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'uuid',
        'entity_type',
        'entity_id',
        'is_assemble_disassemble',
        'quantity',
        'assemble_disassemble_type',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'is_assemble_disassemble' => 'boolean',
        'assemble_disassemble_type' => 'string',
    ];

    // Relationships
    public function entity()
    {
        return $this->morphTo();
    }

    // Scopes
    public function scopeByEntity($query, $entityType, $entityId)
    {
        return $query->where('entity_type', $entityType)
                     ->where('entity_id', $entityId);
    }

    public function scopeWithQuantity($query, $minQuantity = 1)
    {
        return $query->where('quantity', '>=', $minQuantity);
    }

    public function scopeAssembleDisassemble($query, $value = true)
    {
        return $query->where('is_assemble_disassemble', $value);
    }

    // Accessors
    public function getTotalVolumeAttribute()
    {
        if ($this->is_assemble_disassemble) {
            return 0; // Assemble/disassemble items don't have volume
        }
        return $this->quantity * ($this->entity->volumn ?? 0);
    }

    public function getCategoryNameAttribute()
    {
        return $this->entity->name ?? $this->entity->title ?? 'Unknown Category';
    }

    // Methods
    public function getVolume()
    {
        if ($this->is_assemble_disassemble) {
            return 0; // Assemble/disassemble items don't have volume
        }
        return $this->entity->volumn ?? 0;
    }

    public function getTotalVolume()
    {
        return $this->quantity * $this->getVolume();
    }

    public function getSelectedPricingAttribute()
    {
        if (!$this->is_assemble_disassemble || !$this->entity) {
            return null;
        }

        $type = $this->assemble_disassemble_type;
        $entity = $this->entity;

        switch ($type) {
            case 'assemble':
                return [
                    'price' => $entity->assemble_price,
                    'time_minutes' => $entity->assemble_time_minutes,
                    'type' => 'assemble'
                ];
            case 'disassemble':
                return [
                    'price' => $entity->disassemble_price,
                    'time_minutes' => $entity->disassemble_time_minutes,
                    'type' => 'disassemble'
                ];
            case 'both':
                return [
                    'price' => $entity->both_price,
                    'time_minutes' => $entity->both_time_minutes,
                    'type' => 'both'
                ];
            default:
                return null;
        }
    }
}