<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class FAQQuestion extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'faq_questions';

    protected $fillable = [
        'faq_id',
        'question',
        'answer',
        'sort_order',
        'status',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'sort_order' => 'integer',
        'status' => 'integer',
        'created_by' => 'integer',
        'updated_by' => 'integer',
    ];

    /**
     * Get the FAQ that owns this question
     */
    public function faq()
    {
        return $this->belongsTo(FAQ::class, 'faq_id');
    }

    /**
     * Get status text
     */
    public function getStatusText()
    {
        return $this->status ? 'Active' : 'Inactive';
    }

    /**
     * Get status badge
     */
    public function getStatusBadge()
    {
        return $this->status 
            ? '<span class="badge bg-success-subtle text-success">Active</span>'
            : '<span class="badge bg-secondary-subtle text-secondary">Inactive</span>';
    }

    /**
     * Creator relationship
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Updater relationship
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Scope for active questions
     */
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    /**
     * Scope for ordered questions
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order', 'asc')->orderBy('created_at', 'asc');
    }
}