<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class JobApplication extends Model
{
    use HasFactory;

    protected $fillable = [
        'job_id',
        'full_name',
        'email',
        'phone',
        'cover_letter',
        'cv_file',
        'status',
        'notes',
        'user_id',
    ];

    protected $casts = [
        'job_id' => 'integer',
    ];

    // Application status enum values
    const STATUSES = [
        'pending' => 'Pending',
        'reviewed' => 'Reviewed',
        'shortlisted' => 'Shortlisted',
        'interviewed' => 'Interviewed',
        'accepted' => 'Accepted',
        'rejected' => 'Rejected'
    ];

    public function job()
    {
        return $this->belongsTo(Job::class);
    }

    // Scope for active applications (not rejected)
    public function scopeActive($query)
    {
        return $query->where('status', '!=', 'rejected');
    }

    // Scope for valid applications (not expired or invalid)
    public function scopeValid($query)
    {
        return $query->whereHas('job', function($q) {
            $q->where('is_active', true)
              ->where('valid_from', '<=', now())
              ->where('valid_until', '>=', now());
        });
    }

    // Scope for ordering
    public function scopeOrdered($query)
    {
        return $query->orderBy('created_at', 'desc');
    }

    // Get status label
    public function getStatusLabelAttribute()
    {
        return self::STATUSES[$this->status] ?? $this->status;
    }

    // Get status badge class
    public function getStatusBadgeClassAttribute()
    {
        switch ($this->status) {
            case 'pending':
                return 'bg-warning';
            case 'reviewed':
                return 'bg-info';
            case 'shortlisted':
                return 'bg-primary';
            case 'interviewed':
                return 'bg-secondary';
            case 'accepted':
                return 'bg-success';
            case 'rejected':
                return 'bg-danger';
            default:
                return 'bg-secondary';
        }
    }

    // Get CV file URL
    public function getCvUrlAttribute()
    {
        if ($this->cv_file) {
            return asset('storage/job_applications/' . $this->cv_file);
        }
        return null;
    }
}
