<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Truck extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'truck_name',
        'truck_image',
        'truck_type',
        'registration_number',
        'vin_number',
        'make',
        'model',
        'year',
        'engine_type',
        'fuel_type',
        'gross_vehicle_mass',
        'tare_mass',
        'payload_capacity',
        'dimensions_length',
        'dimensions_width',
        'dimensions_height',
        'cubic_capacity',
        'axle_configuration',
        'driving_license_required',
        'insurance_expiry',
        'registration_expiry',
        'safety_inspection_due',
        'driver_name',
        'driver_license_number',
        'driver_phone',
        'driver_email',
        'base_location',
        'service_area',
        'availability_status',
        'pricing_type',
        'custom_fixed_price',
        'custom_per_cubic_meter_price',
        'custom_per_kilometer_price',
        'custom_minimum_distance',
        'custom_maximum_distance',
        'custom_per_hour_price',
        'number_of_movers',
        'price_per_mover',
        'notes',
        'is_active',
        'sort_order'
    ];

    protected $casts = [
        'insurance_expiry' => 'date',
        'registration_expiry' => 'date',
        'safety_inspection_due' => 'date',
        'gross_vehicle_mass' => 'decimal:2',
        'tare_mass' => 'decimal:2',
        'payload_capacity' => 'decimal:2',
        'dimensions_length' => 'decimal:2',
        'dimensions_width' => 'decimal:2',
        'dimensions_height' => 'decimal:2',
        'cubic_capacity' => 'decimal:2',
        'custom_fixed_price' => 'decimal:2',
        'custom_per_cubic_meter_price' => 'decimal:2',
        'custom_per_kilometer_price' => 'decimal:2',
        'custom_minimum_distance' => 'decimal:2',
        'custom_maximum_distance' => 'decimal:2',
        'custom_per_hour_price' => 'decimal:2',
        'number_of_movers' => 'integer',
        'price_per_mover' => 'integer',
        'is_active' => 'boolean',
        'sort_order' => 'integer'
    ];

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order', 'asc');
    }

    public function scopeAvailable($query)
    {
        return $query->where('availability_status', 'available');
    }

    // Accessors
    public function getFullTruckNameAttribute()
    {
        return $this->truck_name . ' (' . $this->registration_number . ')';
    }

    public function getPricingTypeTextAttribute()
    {
        return $this->pricing_type === 'custom' ? 'Custom Pricing' : 'Global Pricing';
    }

    public function getAvailabilityStatusTextAttribute()
    {
        return ucfirst($this->availability_status);
    }

    public function getTruckTypeTextAttribute()
    {
        return ucfirst(str_replace('_', ' ', $this->truck_type));
    }

    // Methods
    public function isUsingGlobalPricing()
    {
        return $this->pricing_type === 'global';
    }

    public function isUsingCustomPricing()
    {
        return $this->pricing_type === 'custom';
    }

    public function calculatePrice($volume = 0, $distance = 0, $hours = 0)
    {
        if ($this->isUsingGlobalPricing()) {
            // Use global pricing from general settings
            $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
            
            $fixedPrice = $generalSettings['_truck_fixed_price'] ?? 0;
            $perCubicMeterPrice = $generalSettings['_truck_per_cubic_meter_price'] ?? 0;
            $perKilometerPrice = $generalSettings['_truck_per_kilometer_price'] ?? 0;
            $minDistance = $generalSettings['_truck_minimum_distance'] ?? 0;
            $maxDistance = $generalSettings['_truck_maximum_distance'] ?? 0;
            $perHourPrice = $generalSettings['_truck_per_hour_price'] ?? 0;
        } else {
            // Use custom pricing
            $fixedPrice = $this->custom_fixed_price ?? 0;
            $perCubicMeterPrice = $this->custom_per_cubic_meter_price ?? 0;
            $perKilometerPrice = $this->custom_per_kilometer_price ?? 0;
            $minDistance = $this->custom_minimum_distance ?? 0;
            $maxDistance = $this->custom_maximum_distance ?? 0;
            $perHourPrice = $this->custom_per_hour_price ?? 0;
        }

        // Apply distance limits
        if ($minDistance > 0 && $distance < $minDistance) {
            $distance = $minDistance;
        }
        if ($maxDistance > 0 && $distance > $maxDistance) {
            $distance = $maxDistance;
        }

        // Calculate total price including per-hour pricing
        $totalPrice = $fixedPrice + ($volume * $perCubicMeterPrice) + ($distance * $perKilometerPrice) + ($hours * $perHourPrice);
        
        return $totalPrice;
    }

    public function calculatePriceBreakdown($volume = 0, $distance = 0, $hours = 0)
    {
        if ($this->isUsingGlobalPricing()) {
            // Use global pricing from general settings
            $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
            
            $fixedPrice = $generalSettings['_truck_fixed_price'] ?? 0;
            $perCubicMeterPrice = $generalSettings['_truck_per_cubic_meter_price'] ?? 0;
            $perKilometerPrice = $generalSettings['_truck_per_kilometer_price'] ?? 0;
            $minDistance = $generalSettings['_truck_minimum_distance'] ?? 0;
            $maxDistance = $generalSettings['_truck_maximum_distance'] ?? 0;
            $perHourPrice = $generalSettings['_truck_per_hour_price'] ?? 0;
           
        } else {
            // Use custom pricing
            $fixedPrice = $this->custom_fixed_price ?? 0;
            $perCubicMeterPrice = $this->custom_per_cubic_meter_price ?? 0;
            $perKilometerPrice = $this->custom_per_kilometer_price ?? 0;
            $minDistance = $this->custom_minimum_distance ?? 0;
            $maxDistance = $this->custom_maximum_distance ?? 0;
            $perHourPrice = $this->custom_per_hour_price ?? 0;
        }

        // Apply distance limits
        $originalDistance = $distance;
        if ($minDistance > 0 && $distance < $minDistance) {
            $distance = $minDistance;
        }
        if ($maxDistance > 0 && $distance > $maxDistance) {
            $distance = $maxDistance;
        }
        $numberOfMovers = $this->number_of_movers ?? 0;
        $pricePerMover = $this->price_per_mover ?? 0;

        // Calculate individual components
        $fixedComponent = $fixedPrice;
        $volumeComponent = $volume * $perCubicMeterPrice;
        $distanceComponent = $distance * $perKilometerPrice;
        $hourComponent = $hours * $perHourPrice;
        $moverComponent = $numberOfMovers * $pricePerMover;
        
        $totalPrice = $fixedComponent + $volumeComponent + $distanceComponent + $hourComponent + $moverComponent;

        return [
            'fixed_price' => $fixedComponent,
            'volume_price' => $volumeComponent,
            'distance_price' => $distanceComponent,
            'hour_price' => $hourComponent,
            'mover_price' => $moverComponent,
            'total_price' => $totalPrice,
            'breakdown' => [
                'Fixed Price' => '$' . number_format($fixedComponent, 2),
                'Volume (' . $volume . ' m³ × $' . number_format($perCubicMeterPrice, 2) . ')' => '$' . number_format($volumeComponent, 2),
                'Distance (' . $distance . ' km × $' . number_format($perKilometerPrice, 2) . ')' => '$' . number_format($distanceComponent, 2),
                'Hours (' . $hours . ' hrs × $' . number_format($perHourPrice, 2) . ')' => '$' . number_format($hourComponent, 2),
                'Movers (' . $numberOfMovers . ' movers × $' . number_format($pricePerMover, 2) . ')' => '$' . number_format($moverComponent, 2),
            ],
            'distance_adjusted' => $originalDistance !== $distance,
            'original_distance' => $originalDistance,
            'adjusted_distance' => $distance,
        ];
    }

    // Relationships
    public function bookings()
    {
        return $this->hasMany(Booking::class);
    }

    public function activeBookings()
    {
        return $this->hasMany(Booking::class)->where('is_completed', false);
    }

    public function completedBookings()
    {
        return $this->hasMany(Booking::class)->where('is_completed', true);
    }
}
