@include('Admin.Includes.header')
@php
    use Illuminate\Support\Facades\Auth;
    $logged_in_user = Auth::user();
@endphp

<div class="page-content">
    <div class="card shadow-sm mb-4">
        <div class="p-3 d-flex justify-content-between align-items-center flex-wrap">
            <div class="clearfix">
                <div class="pd-t-5 pd-b-5">
                    <h1 class="fs-4 pd-0 mg-0 tx-20 tx-dark">{{ __('Blog Categories') }}</h1>
                </div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb bg-transparent p-0 m-0 align-items-center">
                        <li class="breadcrumb-item">
                            <a href="{{ route('Admin.Dashboard') }}" class="link-primary text-decoration-none">
                                <i class="fa-solid fa-house me-1"></i> Dashboard
                            </a>
                        </li>
                        <li class="breadcrumb-item active" aria-current="page">
                            Blog Categories
                        </li>
                    </ol>
                </nav>
            </div>
        </div>
    </div>

    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    @if ($errors->any())
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    <div class="row">
        <div class="col-lg-4">
            <!-- Basic Blog Category Information Card -->
            <div class="card mb-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">Create Blog Category</h6>
                </div>
                <div class="card-body">
                    <form autocomplete="off" method="POST" action="{{ route('admin.blog-categories.store') }}"
                        enctype="multipart/form-data" id="blogCategoryForm">
                        @csrf
                        <input type="hidden" name="blog_category_id" id="blog_category_id">
                        <div id="update_success_message"></div>
                        <div id="saveform_errList" role="alert"></div>

                        <div class="row">
                            <!-- Title -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="blog_category_title" class="form-label">Category Title<span
                                            class="text-danger">*</span></label>
                                    <input type="text" id="blog_category_title" name="title" required
                                        class="form-control addCategoryTitle" placeholder="Enter Blog Category">
                                    <div class="invalid-feedback">Please enter a category title.</div>
                                </div>
                            </div>

                            <!-- Permalink -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="blog_category_permalink" class="form-label">Permalink<span
                                            class="text-danger">*</span></label>
                                    <input type="text" id="blog_category_permalink" name="permalink"
                                        class="form-control categoryPermalink" placeholder="Auto-generated from title"
                                        required>
                                    <small class="text-muted">This will be auto-generated from the title</small>
                                    <div class="text-danger permalink_error"></div>
                                    <input type="hidden" class="permalink_status" name="permalink_status" value="">
                                </div>
                            </div>

                            <!-- Description -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="blog_category_description" class="form-label">Description</label>
                                    <textarea id="blog_category_description" name="description" class="form-control" rows="3"
                                        placeholder="Enter category description"></textarea>
                                </div>
                            </div>

                            <!-- Status -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="blog_category_status" class="form-label">Status</label>
                                    <select id="blog_category_status" name="is_active" class="form-select">
                                        <option value="1" selected>Active</option>
                                        <option value="0">Inactive</option>
                                    </select>
                                </div>
                            </div>

                            <!-- Submit Button -->
                            <div class="col-12">
                                <div class="d-grid">
                                    <button type="submit" class="btn btn-success" id="saveBtn">
                                        <i class="fa-solid fa-save me-1"></i>Save Blog Category
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-8">
            <!-- Blog Categories List Card -->
            <div class="card">
                <div class="card-header d-flex justify-content-between">
                    <h5 class="card-title mb-0">Blog Category Table</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead class="table-light">
                                <tr>
                                    <th>Title</th>
                                    <th>Description</th>
                                    <th>Status</th>
                                    <th width="120px">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach ($categories as $category)
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="avatar-xs me-2">
                                                    <div class="avatar-title rounded-circle bg-primary text-white">
                                                        {{ strtoupper(substr($category->title, 0, 1)) }}
                                                    </div>
                                                </div>
                                                <div>
                                                    <h6 class="mb-0">{{ $category->title }}</h6>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <span class="text-muted">{{ Str::limit($category->description, 50) }}</span>
                                        </td>
                                        <td>
                                            @if ($category->is_active)
                                                <span class="badge bg-success">Active</span>
                                            @else
                                                <span class="badge bg-danger">Inactive</span>
                                            @endif
                                        </td>
                                        <td>
                                            <div class="dropdown">
                                                <button class="btn btn-sm text-dark rounded shadow-sm" type="button"
                                                    data-bs-toggle="dropdown" aria-expanded="false"
                                                    style="background-color: #c2d6e7;">
                                                    &#8942;
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li>
                                                        <button class="dropdown-item"
                                                            onclick="editBlogCategory({{ $category->id }})">
                                                            <i class="bi bi-pencil-square me-2"></i> Edit
                                                        </button>
                                                    </li>
                                                    <li>
                                                        <button class="dropdown-item text-danger"
                                                            onclick="confirmDeleteBlogCategory({{ $category->id }}, '{{ $category->title }}')">
                                                            <i class="bi bi-trash me-2"></i> Delete
                                                        </button>
                                                    </li>
                                                </ul>
                                            </div>
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Delete Confirmation Modal --}}
<div class="modal fade zoomIn" id="deleteConfirmModal" tabindex="-1" aria-labelledby="deleteConfirmModalLabel"
    aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>

            <div class="modal-body p-md-5">
                <div class="text-center">
                    <div class="text-danger">
                        <i class="bi bi-trash display-4"></i>
                    </div>
                    <div class="mt-4 fs-15">
                        <p class="text-muted mx-3 fs-16 mb-0">
                            Are you sure you want to delete
                            <strong id="deleteCategoryName"></strong>?<br>
                            This action cannot be undone.
                        </p>
                        <p class="text-muted mt-2 mb-0">
                            <small>Suggestion: mark it as <em>Inactive</em> instead.</small>
                        </p>
                    </div>
                </div>

                <div class="d-flex gap-2 justify-content-center mt-4 mb-2">
                    <button type="button" class="btn w-sm btn-light btn-hover"
                        data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn w-sm btn-danger btn-hover" id="confirmDeleteBtn">Yes, Delete
                        It!</button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    function resetForm() {
        document.getElementById('blogCategoryForm').reset();
        document.getElementById('blog_category_id').value = '';
        document.getElementById('blog_category_permalink').value = '';
        document.getElementById('saveBtn').innerHTML = '<i class="fa-solid fa-save me-1"></i>Save Blog Category';
        document.getElementById('saveform_errList').innerHTML = '';
        document.getElementById('update_success_message').innerHTML = '';
        document.querySelector('.permalink_error').innerHTML = '';
    }
    
    // Generate permalink from title
    function generatePermalinkFromTitle(title) {
        return title.toLowerCase()
            .replace(/[^a-z0-9\s-]/g, '') // Remove special characters except spaces and hyphens
            .replace(/\s+/g, '-') // Replace spaces with hyphens
            .replace(/-+/g, '-') // Replace multiple hyphens with single hyphen
            .replace(/^-|-$/g, ''); // Remove leading/trailing hyphens
    }
    
    // Auto-generate permalink when title changes
    document.addEventListener('DOMContentLoaded', function() {
        const titleInput = document.getElementById('blog_category_title');
        const permalinkInput = document.getElementById('blog_category_permalink');
        
        if (titleInput && permalinkInput) {
            let isManuallyEditingPermalink = false;
            
            // Auto-generate permalink from title
            titleInput.addEventListener('input', function() {
                if (!isManuallyEditingPermalink) {
                    const title = this.value;
                    if (title) {
                        permalinkInput.value = generatePermalinkFromTitle(title);
                        validatePermalink();
                    } else {
                        permalinkInput.value = '';
                    }
                }
            });
            
            // Track manual permalink editing
            permalinkInput.addEventListener('input', function() {
                isManuallyEditingPermalink = true;
                validatePermalink();
            });
            
            // Reset flag when permalink loses focus
            permalinkInput.addEventListener('blur', function() {
                setTimeout(() => {
                    isManuallyEditingPermalink = false;
                }, 100);
            });
        }
    });
    
    // Validate permalink uniqueness
    function validatePermalink() {
        const permalink = document.getElementById('blog_category_permalink').value;
        const categoryId = document.getElementById('blog_category_id').value;
        const errorDiv = document.querySelector('.permalink_error');
        const statusInput = document.querySelector('.permalink_status');
        
        if (!permalink) {
            errorDiv.innerHTML = '<small class="text-danger">Permalink is required.</small>';
            if (statusInput) statusInput.value = '';
            return;
        }
        
        // Get CSRF token
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '{{ csrf_token() }}';
        
        // Check permalink availability via AJAX
        fetch('/admin/blog-categories/check-permalink', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify({
                permalink: permalink,
                category_id: categoryId || null
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.available) {
                errorDiv.innerHTML = '<small class="text-success">✓ Permalink is available.</small>';
                if (statusInput) statusInput.value = '1';
            } else {
                errorDiv.innerHTML = '<small class="text-danger">✗ ' + data.message + '</small>';
                if (statusInput) statusInput.value = '';
            }
        })
        .catch(error => {
            console.error('Error:', error);
            errorDiv.innerHTML = '<small class="text-danger">✗ Error checking permalink.</small>';
            if (statusInput) statusInput.value = '';
        });
    }

    function showMessage(message, type) {
        const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
        const icon = type === 'success' ? 'fa-check-circle' : 'fa-exclamation-triangle';

        const messageHtml = `
            <div class="alert ${alertClass} alert-dismissible fade show" role="alert">
                <i class="fa-solid ${icon} me-2"></i>${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `;

        document.getElementById('update_success_message').innerHTML = messageHtml;

        // Auto-hide after 3 seconds
        setTimeout(() => {
            const alert = document.querySelector('#update_success_message .alert');
            if (alert) {
                alert.classList.remove('show');
                setTimeout(() => {
                    document.getElementById('update_success_message').innerHTML = '';
                }, 150);
            }
        }, 3000);
    }

    function editBlogCategory(id) {
        // Get category data via AJAX
        fetch(`/admin/blog-categories/${id}`)
            .then(response => response.json())
            .then(data => {
                document.getElementById('blog_category_id').value = data.id;
                document.getElementById('blog_category_title').value = data.title;
                document.getElementById('blog_category_permalink').value = data.permalink || '';
                document.getElementById('blog_category_description').value = data.description || '';
                document.getElementById('blog_category_status').value = data.is_active ? '1' : '0';
                document.getElementById('saveBtn').innerHTML =
                    '<i class="fa-solid fa-save me-1"></i>Update Blog Category';

                // Scroll to form
                document.getElementById('blogCategoryForm').scrollIntoView({
                    behavior: 'smooth'
                });
            })
            .catch(error => {
                console.error('Error:', error);
                showMessage('Error loading category data', 'error');
            });
    }

    function confirmDeleteBlogCategory(id, name) {
        document.getElementById('deleteCategoryName').textContent = name;
        document.getElementById('confirmDeleteBtn').onclick = function() {
            deleteBlogCategory(id);
        };
        new bootstrap.Modal(document.getElementById('deleteConfirmModal')).show();
    }

    function deleteBlogCategory(id) {
        // Create a form to submit DELETE request
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = `/admin/blog-categories/${id}`;

        const methodInput = document.createElement('input');
        methodInput.type = 'hidden';
        methodInput.name = '_method';
        methodInput.value = 'DELETE';

        const tokenInput = document.createElement('input');
        tokenInput.type = 'hidden';
        tokenInput.name = '_token';
        tokenInput.value = '{{ csrf_token() }}';

        form.appendChild(methodInput);
        form.appendChild(tokenInput);
        document.body.appendChild(form);
        form.submit();
    }

    // Handle form submission - let it submit normally like Category index
    document.getElementById('blogCategoryForm').addEventListener('submit', function(e) {
        const categoryId = document.getElementById('blog_category_id').value;

        // Update form action for edit
        if (categoryId) {
            this.action = `/admin/blog-categories/${categoryId}`;
            // Add _method field for PUT request
            let methodField = this.querySelector('input[name="_method"]');
            if (!methodField) {
                methodField = document.createElement('input');
                methodField.type = 'hidden';
                methodField.name = '_method';
                this.appendChild(methodField);
            }
            methodField.value = 'PUT';
        } else {
            this.action = '{{ route('admin.blog-categories.store') }}';
            // Remove _method field for POST request
            const methodField = this.querySelector('input[name="_method"]');
            if (methodField) {
                methodField.remove();
            }
        }

        // Let the form submit normally - no preventDefault()
    });
</script>
