@include('Admin.Includes.header')
@php
    // Get Sydney zip codes from config
    $sydneyZipCodes = config('australian_zipcodes.sydney', []);
    // Get quote ID from route
    $quoteId = request()->route('id');
@endphp

<!-- Google Maps Places API -->
<script>
    // Load Google Maps API asynchronously
    function loadGoogleMapsAPI() {
        const script = document.createElement('script');
        script.src = 'https://maps.googleapis.com/maps/api/js?key=AIzaSyB7aLtWqhztn7r4Fcce7WqQT-Qdlpl7MnQ&libraries=places&callback=initGoogleMaps';
        script.async = true;
        script.defer = true;
        document.head.appendChild(script);
    }

    // Callback function when Google Maps API is loaded
    window.initGoogleMaps = function() {
        // This will be called after the API loads
        if (typeof initializeAutocomplete === 'function') {
            initializeAutocomplete();
        }
    };

    // Load the API when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', loadGoogleMapsAPI);
    } else {
        loadGoogleMapsAPI();
    }
</script>

<div class="page-content">
    <!-- Header Card -->
    <div class="card shadow-sm mb-2">
        <div class="p-3 d-flex justify-content-between align-items-center flex-wrap">
            <div class="clearfix">
                <div class="pd-t-5 pd-b-5">
                    <h1 class="fs-4 pd-0 mg-0 tx-20 tx-dark">
                        <i class="fa-solid fa-pen me-2"></i>
                        Edit Quote/Order
                    </h1>
                </div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb bg-transparent p-0 m-0 align-items-center">
                        <li class="breadcrumb-item">
                            <a href="{{ url('/admin') }}" class="link-primary text-decoration-none">
                                <i class="fa-solid fa-home me-1"></i> Dashboard
                            </a>
                        </li>
                        <li class="breadcrumb-item">
                            <a href="{{ route('admin.quotes.index') }}" class="link-primary text-decoration-none">
                                Quotes/Orders
                            </a>
                        </li>
                        <li class="breadcrumb-item">
                            <a href="{{ route('admin.quotes.show', $quoteId) }}" class="link-primary text-decoration-none">Details</a>
                        </li>
                        <li class="breadcrumb-item active" aria-current="page">
                            Edit
                        </li>
                    </ol>
                </nav>
            </div>
            <div class="d-flex align-items-center gap-2 mt-md-0 mt-2">
                <a href="{{ route('admin.quotes.show', $quoteId) }}" class="btn btn-primary">
                    <i class="fa-solid fa-arrow-left me-1"></i>Back to Details
                </a>
                <a href="{{ route('admin.quotes.index') }}" class="btn btn-primary">
                    <i class="fa-solid fa-list me-1"></i>Back to List
                </a>
            </div>
        </div>
    </div>

    <div class="row g-2">
        <!-- Left Column (8 columns) -->
        <div class="col-md-8 col-12">
            <!-- Edit Quote & Related Data -->
            <div class="card shadow-sm mb-2">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="mb-0">
                        <i class="fa-solid fa-pen me-2"></i>Edit Quote & Related Data
                    </h6>
                    <button type="button" class="btn btn-primary btn-sm" id="saveChangesBtn">
                        <i class="fa-solid fa-floppy-disk me-1"></i>Save Changes
                    </button>
                </div>
                <div class="card-body p-3" id="quoteFormContainer">
                    <!-- Loading indicator -->
                    <div id="loadingIndicator" class="text-center py-5">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <p class="mt-3 text-muted">Loading quote data...</p>
                    </div>
                    
                    <!-- Form content (hidden initially) -->
                    <div id="quoteFormContent" style="display: none;">
                    <!-- Quote Details -->
                    <div class="mb-3">
                        <h6 class="mb-3">
                            <i class="fa-solid fa-file-invoice me-2"></i>Quote Details
                        </h6>
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label for="quote_number" class="form-label">Quote Number</label>
                                <input type="text" class="form-control" id="quote_number" value="" readonly>
                            </div>
                            <div class="col-md-6">
                                <label for="order_number" class="form-label">Order Number</label>
                                <input type="text" class="form-control" id="order_number" value="" readonly>
                            </div>
                            <div class="col-md-6">
                                <label for="quote_type" class="form-label">Quote Type</label>
                                <select class="form-select" id="quote_type">
                                    <option value="quote">Quote</option>
                                    <option value="estimate">Estimate</option>
                                    <option value="order" selected>Order</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="status" class="form-label">Status</label>
                                <select class="form-select" id="status" style="width:100%;position:unset !important;margin:0px;">
                                    <option value="pending" selected>Pending</option>
                                    <option value="approved">Approved</option>
                                    <option value="rejected">Rejected</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="moving_date" class="form-label">Moving Date</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="moving_date" value="" data-datepicker>
                                    <span class="input-group-text" id="calendar-icon" style="cursor: pointer;">
                                        <i class="fa-solid fa-calendar"></i>
                                    </span>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <label for="job_type" class="form-label">Job Type</label>
                                <select class="form-select" id="job_type" disabled>
                                    <option value="local_job" selected>Local Job</option>
                                    <option value="interstate_job">Interstate Job</option>
                                </select>
                                <small class="text-muted">Auto-selected based on locations.</small>
                            </div>
                            <div class="col-12">
                                <label for="pickup_location" class="form-label">Pickup Location</label>
                                <input type="text" class="form-control" id="pickup_location" value="">
                            </div>
                            <div class="col-12">
                                <label for="dropoff_location" class="form-label">Dropoff Location</label>
                                <input type="text" class="form-control" id="dropoff_location" value="">
                            </div>
                            <div class="col-md-4">
                                <label for="extra_movers" class="form-label">Extra Movers</label>
                                <input type="number" class="form-control" id="extra_movers" value="0" min="0" max="2">
                                <small class="text-muted">Count only (included in Movers Cost @ $50/hour each). Maximum 2.</small>
                                <div id="extra_movers_alert" class="alert alert-warning alert-dismissible fade show mt-2" style="display: none; padding: 0.5rem 1rem; font-size: 0.875rem;">
                                    <strong>Warning:</strong> <span id="extra_movers_alert_message"></span>
                                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close" style="padding: 0.25rem 0.5rem; font-size: 0.75rem;"></button>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <label for="base_to_pickup" class="form-label">Base to Pickup (min)</label>
                                <input type="number" class="form-control" id="base_to_pickup" value="0" min="0">
                            </div>
                            <div class="col-md-4">
                                <label for="dropoff_to_base" class="form-label">Dropoff to Base (min)</label>
                                <input type="number" class="form-control" id="dropoff_to_base" value="0" min="0">
                            </div>
                            <div class="col-md-4">
                                <label for="pickup_to_dropoff" class="form-label">Pickup to Dropoff (min)</label>
                                <input type="number" class="form-control" id="pickup_to_dropoff" value="0" min="0">
                            </div>
                            <div class="col-md-6">
                                <label for="total_cost" class="form-label">Total Cost ($)</label>
                                <input type="text" class="form-control" id="total_cost" value="0.00" readonly>
                            </div>
                            <div class="col-md-6">
                                <label for="remaining_amount" class="form-label">Remaining Amount ($)</label>
                                <input type="number" class="form-control" id="remaining_amount" value="0" min="0" step="0.01" readonly>
                                <small class="text-muted">Auto-calculated: Total Cost - Deposit Amount.</small>
                            </div>
                        </div>
                    </div>
                    </div> <!-- End quoteFormContent -->
                </div>
            </div>

            <!-- Booking Items -->
            <div class="card shadow-sm mb-2">
                <div class="card-header">
                    <h6 class="mb-0">
                        <i class="fa-solid fa-briefcase me-2"></i>Booking Items
                    </h6>
                </div>
                <div class="card-body p-3">
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th>Item</th>
                                    <th>Quantity</th>
                                    <th>Assemble?</th>
                                    <th>Assemble Type</th>
                                    <th>Type</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Booking items will be populated from API -->
                            </tbody>
                        </table>
                    </div>
                    <div class="mt-3">
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addItemModal">
                            <i class="fa-solid fa-plus me-1"></i>Add Item
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Right Column (4 columns) -->
        <div class="col-md-4 col-12">
            <!-- Cost Breakdown (Read-only) -->
            <div class="card shadow-sm mb-2">
                <div class="card-header">
                    <h6 class="mb-0">
                        <i class="fa-solid fa-list me-2"></i>Cost Breakdown (Read-only)
                    </h6>
                </div>
                <div class="card-body p-3">
                    <div class="mb-3">
                        <label for="movers_cost" class="form-label">Movers Cost ($)</label>
                        <input type="text" class="form-control" id="movers_cost" value="0" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="callout_fee" class="form-label">Callout Fee ($)</label>
                        <input type="text" class="form-control" id="callout_fee" value="0" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="storage_items_cost" class="form-label">Storage Items Cost ($)</label>
                        <input type="text" class="form-control" id="storage_items_cost" value="0" readonly>
                    </div>
                    <div class="mb-3">
                        <label for="assemble_disassemble_cost" class="form-label">Assemble Disassemble Cost ($)</label>
                        <input type="text" class="form-control" id="assemble_disassemble_cost" value="0" readonly>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Item Modal -->
<div class="modal fade" id="addItemModal" tabindex="-1" aria-labelledby="addItemModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addItemModalLabel">
                    <i class="fa-solid fa-plus me-2"></i>Add Booking Item
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="addItemForm">
                    <!-- Category Selection -->
                    <div class="mb-3">
                        <label for="item_category" class="form-label">Select Category</label>
                        <select class="form-select" id="item_category" required>
                            <option value="">-- Select Category --</option>
                            <option value="regular">Regular</option>
                            <option value="assemble_disassemble">Assemble & Disassemble</option>
                            <option value="storage_items">Storage Items</option>
                        </select>
                    </div>

                    <!-- Item Selection (Dynamic) -->
                    <div class="mb-3">
                        <label for="item_select" class="form-label">Select Item</label>
                        <select class="form-select" id="item_select" required disabled>
                            <option value="">-- Select Item --</option>
                        </select>
                    </div>

                    <!-- Assemble/Disassemble Type (Shown only for Assemble & Disassemble items) -->
                    <div class="mb-3" id="assemble_type_group" style="display: none;">
                        <label class="form-label">Assemble/Disassemble Type</label>
                        <div class="d-flex gap-3">
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="assemble_type" id="assemble_both" value="both">
                                <label class="form-check-label" for="assemble_both">
                                    Both
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="assemble_type" id="assemble_assemble" value="assemble">
                                <label class="form-check-label" for="assemble_assemble">
                                    Assemble
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="assemble_type" id="assemble_disassemble" value="disassemble">
                                <label class="form-check-label" for="assemble_disassemble">
                                    Disassemble
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Quantity -->
                    <div class="mb-3">
                        <label for="item_quantity" class="form-label">Quantity</label>
                        <input type="number" class="form-control" id="item_quantity" value="1" min="1" required>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveItemBtn">Add Item</button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Item Confirmation Modal -->
<div class="modal fade zoomIn" id="deleteItemModal" tabindex="-1" aria-labelledby="deleteItemModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body p-md-5">
                <div class="text-center">
                    <div class="text-danger">
                        <i class="bi bi-trash display-4"></i>
                    </div>
                    <div class="mt-4 fs-15">
                        <p class="text-muted mx-3 fs-16 mb-0">
                            Are you sure you want to delete
                            <strong id="deleteItemName"></strong>?<br>
                            This action cannot be undone.
                        </p>
                        <p class="text-muted mt-2 mb-0">
                            <small>This item will be removed from the booking.</small>
                        </p>
                    </div>
                </div>
                <div class="d-flex gap-2 justify-content-center mt-4 mb-2">
                    <button type="button" class="btn w-sm btn-light btn-hover" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn w-sm btn-danger btn-hover" id="confirmDeleteItemBtn">Yes, Delete It!</button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Function to enable/disable assemble fields based on item type
    function toggleAssembleFields(row) {
        const typeBadge = row.querySelector('.item-type-badge');
        const assembleCheckbox = row.querySelector('.assemble-checkbox');
        const assembleTypeSelect = row.querySelector('.assemble-type-select');
        
        if (!typeBadge || !assembleCheckbox || !assembleTypeSelect) return;
        
        const itemType = typeBadge.getAttribute('data-type') || typeBadge.textContent.trim().toLowerCase();
        
        // Enable if type is "assemble & disassemble" or "assemble_disassemble"
        if (itemType === 'assemble & disassemble' || itemType === 'assemble_disassemble' || itemType.includes('assemble')) {
            assembleCheckbox.disabled = false;
            assembleTypeSelect.disabled = false;
        } else {
            // Disable for Regular, Storage Items, or any other type
            assembleCheckbox.disabled = true;
            assembleTypeSelect.disabled = true;
            assembleCheckbox.checked = false;
            assembleTypeSelect.value = '';
        }
    }

    // Apply enable/disable logic to all existing rows
    function updateAllRowsAssembleFields() {
        const rows = document.querySelectorAll('tbody tr');
        rows.forEach(row => {
            toggleAssembleFields(row);
        });
    }

    // Initialize on page load
    updateAllRowsAssembleFields();

    // Delete item functionality - show modal instead of alert
    let rowToDelete = null;
    
    document.addEventListener('click', function(e) {
        if (e.target.closest('.delete-item-btn')) {
            const button = e.target.closest('.delete-item-btn');
            const row = button.closest('tr');
            if (row) {
                // Get item name from first cell
                const itemName = row.cells[0]?.textContent?.trim() || 'this item';
                
                // Store row to delete
                rowToDelete = row;
                
                // Set item name in modal
                document.getElementById('deleteItemName').textContent = itemName;
                
                // Show modal
                const deleteModal = new bootstrap.Modal(document.getElementById('deleteItemModal'));
                deleteModal.show();
            }
        }
    });
    
    // Handle confirm delete button click
    document.getElementById('confirmDeleteItemBtn').addEventListener('click', function() {
        if (rowToDelete) {
            rowToDelete.remove();
            rowToDelete = null;
            
            // Hide modal
            const deleteModal = bootstrap.Modal.getInstance(document.getElementById('deleteItemModal'));
            if (deleteModal) {
                deleteModal.hide();
            }
        }
    });

    // Data for dropdowns (will be populated from API)
    let regularItems = [];
    let assembleDisassembleItems = [];
    let storageItems = [];

    const categorySelect = document.getElementById('item_category');
    const itemSelect = document.getElementById('item_select');
    const assembleTypeGroup = document.getElementById('assemble_type_group');

    // Handle category change
    categorySelect.addEventListener('change', function() {
        const selectedCategory = this.value;
        itemSelect.innerHTML = '<option value="">-- Select Item --</option>';
        assembleTypeGroup.style.display = 'none';
        
        // Reset radio buttons
        document.querySelectorAll('input[name="assemble_type"]').forEach(radio => {
            radio.checked = false;
        });

        if (selectedCategory === '') {
            itemSelect.disabled = true;
            return;
        }

        itemSelect.disabled = false;
        let items = [];

        if (selectedCategory === 'regular') {
            items = regularItems;
        } else if (selectedCategory === 'assemble_disassemble') {
            items = assembleDisassembleItems;
        } else if (selectedCategory === 'storage_items') {
            items = storageItems;
        }

        // Populate item dropdown
        items.forEach(item => {
            const option = document.createElement('option');
            option.value = item.id;
            if (selectedCategory === 'storage_items') {
                option.textContent = `${item.name} - $${item.price.toFixed(2)}`;
            } else {
                option.textContent = item.name;
            }
            itemSelect.appendChild(option);
        });
    });

    // Handle item selection for Assemble & Disassemble
    itemSelect.addEventListener('change', function() {
        const selectedCategory = categorySelect.value;
        if (selectedCategory === 'assemble_disassemble' && this.value !== '') {
            assembleTypeGroup.style.display = 'block';
        } else {
            assembleTypeGroup.style.display = 'none';
        }
    });

    // Handle save button
    document.getElementById('saveItemBtn').addEventListener('click', function() {
        const category = categorySelect.value;
        const item = itemSelect.value;
        const quantity = document.getElementById('item_quantity').value;
        
        if (!category || !item || !quantity) {
            alert('Please fill in all required fields.');
            return;
        }

        if (category === 'assemble_disassemble') {
            const assembleType = document.querySelector('input[name="assemble_type"]:checked');
            if (!assembleType) {
                alert('Please select an Assemble/Disassemble type.');
                return;
            }
        }

        // Get item details
        let itemName = '';
        let itemType = '';
        let badgeClass = '';
        
        if (category === 'regular') {
            const selectedItem = regularItems.find(i => i.id == item);
            itemName = selectedItem ? selectedItem.name : 'Regular Item';
            itemType = 'regular';
            badgeClass = 'bg-primary';
        } else if (category === 'assemble_disassemble') {
            const selectedItem = assembleDisassembleItems.find(i => i.id == item);
            itemName = selectedItem ? selectedItem.name : 'Assemble Item';
            itemType = 'assemble_disassemble';
            badgeClass = 'bg-info';
        } else if (category === 'storage_items') {
            const selectedItem = storageItems.find(i => i.id == item);
            itemName = selectedItem ? selectedItem.name : 'Storage Item';
            itemType = 'storage_items';
            badgeClass = 'bg-secondary';
        }

        // Get assemble type if applicable
        const assembleTypeValue = category === 'assemble_disassemble' 
            ? document.querySelector('input[name="assemble_type"]:checked').value 
            : '';

        // Determine entity_type based on category
        let entityType = '';
        if (category === 'regular') {
            entityType = 'App\\Models\\ChildCategory';
        } else if (category === 'assemble_disassemble') {
            entityType = 'App\\Models\\AssembleDisassembleCategory';
        } else if (category === 'storage_items') {
            entityType = 'App\\Models\\StorageItem';
        }

        // Add row to table
        const tbody = document.querySelector('tbody');
        const newRow = document.createElement('tr');
        
        // Store entity_id and entity_type as data attributes
        newRow.setAttribute('data-entity-id', item);
        newRow.setAttribute('data-entity-type', entityType);
        
        newRow.innerHTML = `
            <td>${itemName}</td>
            <td>
                <input type="number" class="form-control form-control-sm quantity-input" value="${quantity}" min="1" style="width: 80px;">
            </td>
            <td>
                <div class="form-check form-switch">
                    <input class="form-check-input assemble-checkbox" type="checkbox" ${category === 'assemble_disassemble' ? 'checked' : ''}>
                </div>
            </td>
            <td>
                <select class="form-select form-select-sm assemble-type-select">
                    <option value="">Select type</option>
                    <option value="assemble" ${assembleTypeValue === 'assemble' ? 'selected' : ''}>Assemble</option>
                    <option value="disassemble" ${assembleTypeValue === 'disassemble' ? 'selected' : ''}>Disassemble</option>
                    <option value="both" ${assembleTypeValue === 'both' ? 'selected' : ''}>Both</option>
                </select>
            </td>
            <td>
                <span class="badge ${badgeClass} item-type-badge" data-type="${itemType}">${itemType === 'assemble_disassemble' ? 'Assemble & Disassemble' : (itemType === 'storage_items' ? 'Storage Items' : 'Regular')}</span>
            </td>
            <td>
                <button type="button" class="btn btn-sm btn-danger delete-item-btn">
                    <i class="fa-solid fa-times"></i>
                </button>
            </td>
        `;
        
        tbody.appendChild(newRow);
        
        // Apply enable/disable logic to the new row
        toggleAssembleFields(newRow);

        // Close modal and reset form
        const modal = bootstrap.Modal.getInstance(document.getElementById('addItemModal'));
        modal.hide();
        
        // Reset form
        document.getElementById('addItemForm').reset();
        itemSelect.innerHTML = '<option value="">-- Select Item --</option>';
        itemSelect.disabled = true;
        assembleTypeGroup.style.display = 'none';
        categorySelect.value = '';
    });

    // Reset form when modal is closed
    document.getElementById('addItemModal').addEventListener('hidden.bs.modal', function() {
        document.getElementById('addItemForm').reset();
        itemSelect.innerHTML = '<option value="">-- Select Item --</option>';
        itemSelect.disabled = true;
        assembleTypeGroup.style.display = 'none';
        categorySelect.value = '';
    });

    // Google Maps Places Autocomplete
    function initializeAutocomplete() {
        // Initialize Pickup Location Autocomplete
        const pickupInput = document.getElementById('pickup_location');
        if (pickupInput && typeof google !== 'undefined' && google.maps && google.maps.places) {
            const pickupAutocomplete = new google.maps.places.Autocomplete(pickupInput, {
                types: ['address'],
                componentRestrictions: { country: ['au', 'nz'] } // Restrict to Australia and New Zealand
            });

            pickupAutocomplete.addListener('place_changed', function() {
                const place = pickupAutocomplete.getPlace();
                if (place.formatted_address) {
                    pickupInput.value = place.formatted_address;
                    // Update job type and calculate times when pickup location changes
                    updateJobType();
                    calculateTravelTimes();
                }
            });
        }

        // Initialize Dropoff Location Autocomplete
        const dropoffInput = document.getElementById('dropoff_location');
        if (dropoffInput && typeof google !== 'undefined' && google.maps && google.maps.places) {
            const dropoffAutocomplete = new google.maps.places.Autocomplete(dropoffInput, {
                types: ['address'],
                componentRestrictions: { country: ['au', 'nz'] } // Restrict to Australia and New Zealand
            });

            dropoffAutocomplete.addListener('place_changed', function() {
                const place = dropoffAutocomplete.getPlace();
                if (place.formatted_address) {
                    dropoffInput.value = place.formatted_address;
                    // Update job type and calculate times when dropoff location changes
                    updateJobType();
                    calculateTravelTimes();
                }
            });
        }
    }

    // Make initializeAutocomplete available globally for the callback
    window.initializeAutocomplete = initializeAutocomplete;

    // Try to initialize if API is already loaded
    if (typeof google !== 'undefined' && google.maps && google.maps.places) {
        initializeAutocomplete();
    }

    // Variables to store addresses and bases
    let bases = [];
    let selectedBase = null;
    let pickupAddress = '';
    let dropoffAddress = '';
    
    // Sydney zip codes from config
    const sydneyZipCodes = @json($sydneyZipCodes);
    console.log('Sydney Zip Codes:', sydneyZipCodes);
    
    // Quote ID from route
    const quoteId = @json($quoteId);
    
    // Store API data
    let quoteData = null;
    let childCategories = [];
    let assembleCategories = [];
    let storageItemsData = [];

    // Fetch quote data from API
    async function fetchQuoteData() {
        if (!quoteId) {
            console.log('No quote ID found');
            const loadingIndicator = document.getElementById('loadingIndicator');
            if (loadingIndicator) {
                loadingIndicator.innerHTML = '<div class="alert alert-warning">Quote ID not found. Please go back and try again.</div>';
            }
            return;
        }
        
        try {
            const response = await fetch(`/admin/quotes/${quoteId}/edit-data`);
            
            // Check if response is OK (status 200-299)
            if (!response.ok) {
                let errorMessage = 'Error loading quote data.';
                
                if (response.status === 404) {
                    errorMessage = 'Quote not found. The quote you are trying to edit does not exist.';
                } else if (response.status === 403) {
                    errorMessage = 'Access denied. You do not have permission to view this quote.';
                } else if (response.status === 500) {
                    errorMessage = 'Server error. Please try again later.';
                } else {
                    errorMessage = `Error ${response.status}: ${response.statusText}`;
                }
                
                const loadingIndicator = document.getElementById('loadingIndicator');
                if (loadingIndicator) {
                    loadingIndicator.innerHTML = `
                        <div class="alert alert-danger">
                            <h6 class="alert-heading">Error Loading Quote</h6>
                            <p class="mb-2">${errorMessage}</p>
                            <a href="{{ route('admin.quotes.index') }}" class="btn btn-sm btn-primary">Go Back to Quotes List</a>
                        </div>
                    `;
                }
                return;
            }
            
            // Check if response is JSON
            const contentType = response.headers.get('content-type');
            if (!contentType || !contentType.includes('application/json')) {
                const loadingIndicator = document.getElementById('loadingIndicator');
                if (loadingIndicator) {
                    loadingIndicator.innerHTML = `
                        <div class="alert alert-danger">
                            <h6 class="alert-heading">Invalid Response</h6>
                            <p class="mb-2">The server returned an invalid response. Please try again.</p>
                            <a href="{{ route('admin.quotes.index') }}" class="btn btn-sm btn-primary">Go Back to Quotes List</a>
                        </div>
                    `;
                }
                return;
            }
            
            const data = await response.json();
            console.log('Quote API Response:', data);
            
            // Check if quote is not editable
            if (response.status === 403 || data.statusCode === 403) {
                const loadingIndicator = document.getElementById('loadingIndicator');
                if (loadingIndicator) {
                    loadingIndicator.innerHTML = `
                        <div class="alert alert-warning">
                            <h6 class="alert-heading">Quote Not Editable</h6>
                            <p class="mb-2">${data.message || 'This quote cannot be edited. Deposit has been paid or quote type is order.'}</p>
                            <div class="d-flex gap-2">
                                <a href="{{ route('admin.quotes.show', $quoteId) }}" class="btn btn-sm btn-primary">Go Back to Details</a>
                                <a href="{{ route('admin.quotes.index') }}" class="btn btn-sm btn-secondary">Go Back to Quotes List</a>
                            </div>
                        </div>
                    `;
                }
                return;
            }
            
            if (data.statusCode === 200 && data.data) {
                quoteData = data.data;
                
                // Store categories and items
                childCategories = data.data.childCategories || [];
                assembleCategories = data.data.assembleCategories || [];
                storageItemsData = data.data.storageItems || [];
                
                // Populate form fields
                populateFormFields();
                
                // Populate booking items table
                populateBookingItems();
                
                // Update dropdowns with API data
                updateDropdowns();
                
                // Hide loading indicator and show form
                const loadingIndicator = document.getElementById('loadingIndicator');
                const quoteFormContent = document.getElementById('quoteFormContent');
                if (loadingIndicator) loadingIndicator.style.display = 'none';
                if (quoteFormContent) quoteFormContent.style.display = 'block';
                
                // Initialize date picker after form is shown
                setTimeout(() => {
                    initializeDatePicker();
                }, 300);
            } else {
                console.error('Invalid API response structure:', data);
                // Show error message
                const loadingIndicator = document.getElementById('loadingIndicator');
                if (loadingIndicator) {
                    loadingIndicator.innerHTML = `
                        <div class="alert alert-danger">
                            <h6 class="alert-heading">Invalid Data</h6>
                            <p class="mb-2">The quote data format is invalid. Please refresh the page.</p>
                            <button onclick="location.reload()" class="btn btn-sm btn-primary">Refresh Page</button>
                        </div>
                    `;
                }
            }
        } catch (error) {
            console.error('Error fetching quote data:', error);
            // Show error message
            const loadingIndicator = document.getElementById('loadingIndicator');
            if (loadingIndicator) {
                let errorMessage = 'An unexpected error occurred while loading quote data.';
                
                if (error.message && error.message.includes('JSON')) {
                    errorMessage = 'Invalid response from server. The server may be returning an error page instead of data.';
                } else {
                    errorMessage = `Error: ${error.message || 'Unknown error'}`;
                }
                
                loadingIndicator.innerHTML = `
                    <div class="alert alert-danger">
                        <h6 class="alert-heading">Error Loading Quote</h6>
                        <p class="mb-2">${errorMessage}</p>
                        <div class="d-flex gap-2">
                            <button onclick="location.reload()" class="btn btn-sm btn-primary">Refresh Page</button>
                            <a href="{{ route('admin.quotes.index') }}" class="btn btn-sm btn-secondary">Go Back to Quotes List</a>
                        </div>
                    </div>
                `;
            }
        }
    }

    // Populate form fields with quote data
    function populateFormFields() {
        if (!quoteData || !quoteData.quote) return;
        
        const quote = quoteData.quote;
        const booking = quote.booking || {};
        // Handle both snake_case and camelCase property names
        const pickupProperty = booking.pick_up_property || booking.pickUpProperty || {};
        const dropoffProperty = booking.drop_off_property || booking.dropOffProperty || {};
        
        // Quote details
        if (quote.quote_number) {
            document.getElementById('quote_number').value = quote.quote_number;
        }
        if (quote.order_number) {
            document.getElementById('order_number').value = quote.order_number;
        }
        if (quote.quote_type) {
            document.getElementById('quote_type').value = quote.quote_type;
        }
        if (quote.status) {
            document.getElementById('status').value = quote.status;
        }
        
        // Moving date
        if (booking.move_date) {
            const moveDate = new Date(booking.move_date);
            // Format as YYYY-MM-DD for date picker
            const year = moveDate.getFullYear();
            const month = String(moveDate.getMonth() + 1).padStart(2, '0');
            const day = String(moveDate.getDate()).padStart(2, '0');
            const formattedDate = `${year}-${month}-${day}`;
            document.getElementById('moving_date').value = formattedDate;
        }
        
        // Job type (will be auto-updated by updateJobType function)
        if (quote.raw_data && quote.raw_data.job_type) {
            document.getElementById('job_type').value = quote.raw_data.job_type;
        }
        
        // Locations - check multiple sources (booking properties and root level)
        const pickupLocation = pickupProperty.location || 
                               (quoteData.pickUpProperty && quoteData.pickUpProperty.location) || 
                               '';
        const dropoffLocation = dropoffProperty.location || 
                                (quoteData.dropOffProperty && quoteData.dropOffProperty.location) || 
                                '';
        
        if (pickupLocation) {
            document.getElementById('pickup_location').value = pickupLocation;
        }
        if (dropoffLocation) {
            document.getElementById('dropoff_location').value = dropoffLocation;
        }
        
        // Log for debugging
        console.log('Pickup Property:', pickupProperty);
        console.log('Dropoff Property:', dropoffProperty);
        console.log('Root Pickup Property:', quoteData.pickUpProperty);
        console.log('Root Dropoff Property:', quoteData.dropOffProperty);
        
        // Extra movers
        if (quote.raw_data && quote.raw_data.extra_movers) {
            document.getElementById('extra_movers').value = quote.raw_data.extra_movers || 0;
        }
        
        // Time fields - check multiple possible field names
        if (quote.raw_data) {
            const rawData = quote.raw_data;
            
            // Base to Pickup - check multiple field names
            if (rawData.base_to_pickup_minutes !== undefined) {
                document.getElementById('base_to_pickup').value = rawData.base_to_pickup_minutes || 0;
            } else if (rawData.base_to_pickup_time !== undefined) {
                document.getElementById('base_to_pickup').value = Math.round(rawData.base_to_pickup_time) || 0;
            }
            
            // Dropoff to Base - check multiple field names
            if (rawData.dropoff_to_base_minutes !== undefined) {
                document.getElementById('dropoff_to_base').value = rawData.dropoff_to_base_minutes || 0;
            } else if (rawData.dropof_to_base !== undefined) {
                document.getElementById('dropoff_to_base').value = Math.round(rawData.dropof_to_base) || 0;
            }
            
            // Pickup to Dropoff - check multiple field names
            if (rawData.pickup_to_dropoff_minutes !== undefined) {
                document.getElementById('pickup_to_dropoff').value = rawData.pickup_to_dropoff_minutes || 0;
            } else if (rawData.pickup_to_dropof !== undefined) {
                document.getElementById('pickup_to_dropoff').value = Math.round(rawData.pickup_to_dropof) || 0;
            }
        }
        
        // Total cost
        if (quote.total_cost) {
            document.getElementById('total_cost').value = parseFloat(quote.total_cost).toFixed(2);
        }
        
        // Remaining amount - calculate from total cost and deposit
        const depositAmount = parseFloat(quote.deposit_amount || 0);
        const totalCost = parseFloat(quote.total_cost || 0);
        const calculatedRemaining = Math.max(0, totalCost - depositAmount);
        
        if (quote.remaining_amount !== undefined && quote.remaining_amount !== null) {
            // Use existing remaining amount if it exists
            document.getElementById('remaining_amount').value = parseFloat(quote.remaining_amount || 0).toFixed(2);
        } else {
            // Calculate remaining amount
            document.getElementById('remaining_amount').value = calculatedRemaining.toFixed(2);
        }
        
        // Cost breakdown (read-only)
        if (quote.raw_data && quote.raw_data.breakdown) {
            const breakdown = quote.raw_data.breakdown;
            if (breakdown.movers_cost !== undefined) {
                document.getElementById('movers_cost').value = parseFloat(breakdown.movers_cost || 0).toFixed(2);
            }
            if (breakdown.callout_fee !== undefined) {
                document.getElementById('callout_fee').value = parseFloat(breakdown.callout_fee || 0).toFixed(2);
            }
            if (breakdown.storage_items_cost !== undefined) {
                document.getElementById('storage_items_cost').value = parseFloat(breakdown.storage_items_cost || 0).toFixed(2);
            }
            if (breakdown.assemble_disassemble_cost !== undefined) {
                document.getElementById('assemble_disassemble_cost').value = parseFloat(breakdown.assemble_disassemble_cost || 0).toFixed(2);
            }
        }
        
        // Update job type after populating addresses
        setTimeout(() => {
            updateJobType();
            // Also trigger travel time calculation if both addresses are available
            const pickupLoc = document.getElementById('pickup_location').value;
            const dropoffLoc = document.getElementById('dropoff_location').value;
            if (pickupLoc && dropoffLoc) {
                calculateTravelTimes();
            }
        }, 1000);
    }

    // Populate booking items table
    function populateBookingItems() {
        if (!quoteData || !quoteData.quote || !quoteData.quote.booking) return;
        
        // Handle both booking_items and bookingItems (camelCase)
        const bookingItems = quoteData.quote.booking.booking_items || quoteData.quote.booking.bookingItems || [];
        const tbody = document.querySelector('tbody');
        if (!tbody) return;
        
        // Clear existing rows (except header)
        tbody.innerHTML = '';
        
        bookingItems.forEach((item) => {
            // Handle entity - it might be null or an object
            const entity = item.entity || null;
            let itemName = 'Unknown Item';
            if (entity) {
                itemName = entity.title || entity.name || 'Unknown Item';
            } else if (item.entity_type) {
                // If entity is null, try to get name from entity_type
                itemName = item.entity_type.split('\\').pop() || 'Unknown Item';
            }
            
            const quantity = item.quantity || 1;
            const isAssemble = item.is_assemble_disassemble || false;
            const assembleType = item.assemble_disassemble_type || '';
            
            // Determine type and badge based on entity_type
            let itemType = 'regular';
            let badgeClass = 'bg-primary';
            let badgeText = 'Regular';
            
            if (isAssemble) {
                itemType = 'assemble_disassemble';
                badgeClass = 'bg-info';
                badgeText = 'Assemble & Disassemble';
            } else if (item.entity_type && (item.entity_type.includes('StorageItem') || item.entity_type.includes('Storage'))) {
                itemType = 'storage_items';
                badgeClass = 'bg-secondary';
                badgeText = 'Storage Items';
            }
            
            // Get entity_id and entity_type
            const entityId = item.entity_id || null;
            const entityType = item.entity_type || '';
            
            const row = document.createElement('tr');
            // Store entity_id and entity_type as data attributes
            row.setAttribute('data-entity-id', entityId);
            row.setAttribute('data-entity-type', entityType);
            row.setAttribute('data-booking-item-id', item.id || '');
            
            row.innerHTML = `
                <td>${itemName}</td>
                <td>
                    <input type="number" class="form-control form-control-sm quantity-input" value="${quantity}" min="1" style="width: 80px;">
                </td>
                <td>
                    <div class="form-check form-switch">
                        <input class="form-check-input assemble-checkbox" type="checkbox" ${isAssemble ? 'checked' : ''}>
                    </div>
                </td>
                <td>
                    <select class="form-select form-select-sm assemble-type-select">
                        <option value="">Select type</option>
                        <option value="assemble" ${assembleType === 'assemble' ? 'selected' : ''}>Assemble</option>
                        <option value="disassemble" ${assembleType === 'disassemble' ? 'selected' : ''}>Disassemble</option>
                        <option value="both" ${assembleType === 'both' ? 'selected' : ''}>Both</option>
                    </select>
                </td>
                <td>
                    <span class="badge ${badgeClass} item-type-badge" data-type="${itemType}">${badgeText}</span>
                </td>
                <td>
                    <button type="button" class="btn btn-sm btn-danger delete-item-btn">
                        <i class="fa-solid fa-times"></i>
                    </button>
                </td>
            `;
            
            tbody.appendChild(row);
            
            // Apply enable/disable logic to the new row
            toggleAssembleFields(row);
        });
    }

    // Update dropdowns with API data
    function updateDropdowns() {
        // Update regular items (childCategories)
        regularItems = childCategories.map(cat => ({
            id: cat.id,
            name: cat.title
        }));
        
        // Update assemble/disassemble items
        assembleDisassembleItems = assembleCategories.map(cat => ({
            id: cat.id,
            name: cat.name
        }));
        
        // Update storage items (use storageItemsData to avoid conflict)
        const updatedStorageItems = storageItemsData.map(item => ({
            id: item.id,
            name: `${item.title} - $${parseFloat(item.price).toFixed(2)}`,
            price: parseFloat(item.price)
        }));
        
        // Update the global storageItems array used in modal
        storageItems = updatedStorageItems;
        
        console.log('Dropdowns updated with API data');
    }

    // Fetch bases from API
    async function fetchBases() {
        try {
            const response = await fetch('/api/general-settings');
            const data = await response.json();
            console.log('API Response:', data); // Debug log
            
            // API returns: { message: "success", statusCode: 200, data: { _bases: [...] } }
            if (data.data && data.data._bases) {
                bases = Array.isArray(data.data._bases) ? data.data._bases : [];
                console.log('Bases fetched:', bases); // Debug log
                console.log('Number of bases:', bases.length); // Debug log
                
                // If bases array is empty, log warning
                if (bases.length === 0) {
                    console.warn('Bases array is empty. Please add bases in General Settings.');
                }
            } else {
                console.log('No bases found in API response. Response structure:', data); // Debug log
                bases = [];
            }
        } catch (error) {
            console.error('Error fetching bases:', error);
            bases = [];
        }
    }

    // Find closest base to pickup location
    async function findClosestBase(pickupAddress) {
        console.log('Finding closest base for:', pickupAddress); // Debug log
        console.log('Available bases:', bases); // Debug log
        
        if (!bases || bases.length === 0) {
            console.log('No bases available. Bases array:', bases); // Debug log
            return null;
        }
        
        if (!pickupAddress) {
            console.log('No pickup address provided'); // Debug log
            return null;
        }

        if (typeof google === 'undefined' || !google.maps || !google.maps.DistanceMatrixService) {
            console.log('Google Maps API not loaded'); // Debug log
            return null;
        }

        return new Promise((resolve) => {
            const service = new google.maps.DistanceMatrixService();
            // Handle different base structures: object with address property or string
            const origins = bases.map((base, idx) => {
                let address;
                if (typeof base === 'string') {
                    address = base;
                } else if (base && base.address) {
                    address = base.address;
                } else if (base && typeof base === 'object') {
                    // Try to find address in object properties
                    address = base.address || base.name || JSON.stringify(base);
                } else {
                    address = base;
                }
                console.log(`Base ${idx + 1} address:`, address); // Debug log
                return address;
            });
            const destinations = [pickupAddress];
            
            console.log('Origins for Distance Matrix:', origins); // Debug log
            console.log('Destination:', destinations); // Debug log

            service.getDistanceMatrix({
                origins: origins,
                destinations: destinations,
                travelMode: google.maps.TravelMode.DRIVING,
                unitSystem: google.maps.UnitSystem.METRIC
            }, (response, status) => {
                console.log('Distance Matrix Status:', status); // Debug log
                console.log('Distance Matrix Response:', response); // Debug log
                
                if (status === 'OK' && response.rows && response.rows.length > 0) {
                    let minDistance = Infinity;
                    let closestBaseIndex = 0;

                    response.rows.forEach((row, index) => {
                        if (row.elements && row.elements[0]) {
                            console.log(`Base ${index + 1} status:`, row.elements[0].status); // Debug log
                            if (row.elements[0].status === 'OK') {
                                const distance = row.elements[0].distance.value; // in meters
                                const duration = row.elements[0].duration.value; // in seconds
                                console.log(`Base ${index + 1} - Distance: ${distance}m, Duration: ${duration}s`); // Debug log
                                if (distance < minDistance) {
                                    minDistance = distance;
                                    closestBaseIndex = index;
                                }
                            }
                        }
                    });

                    selectedBase = bases[closestBaseIndex];
                    console.log('Selected closest base:', selectedBase); // Debug log
                    console.log('Closest base index:', closestBaseIndex); // Debug log
                    resolve(selectedBase);
                } else {
                    console.log('Distance Matrix API error or no results'); // Debug log
                    resolve(null);
                }
            });
        });
    }

    // Calculate travel times
    async function calculateTravelTimes() {
        const pickupInput = document.getElementById('pickup_location');
        const dropoffInput = document.getElementById('dropoff_location');
        const baseToPickupInput = document.getElementById('base_to_pickup');
        const pickupToDropoffInput = document.getElementById('pickup_to_dropoff');
        const dropoffToBaseInput = document.getElementById('dropoff_to_base');

        pickupAddress = pickupInput ? pickupInput.value.trim() : '';
        dropoffAddress = dropoffInput ? dropoffInput.value.trim() : '';

        if (!pickupAddress || !dropoffAddress) {
            return;
        }

        if (typeof google === 'undefined' || !google.maps || !google.maps.DistanceMatrixService) {
            console.log('Google Maps API not loaded yet');
            return;
        }

        // Fetch bases if not already fetched
        if (bases.length === 0) {
            await fetchBases();
        }

        // Find closest base to pickup
        await findClosestBase(pickupAddress);

        if (!selectedBase) {
            console.log('No base found after search. Bases available:', bases.length);
            console.log('Pickup address:', pickupAddress);
            return;
        }
        
        console.log('Using base for calculations:', selectedBase); // Debug log

        // Extract base address - handle different structures
        let baseAddress;
        if (typeof selectedBase === 'string') {
            baseAddress = selectedBase;
        } else if (selectedBase && selectedBase.address) {
            baseAddress = selectedBase.address;
        } else if (selectedBase && typeof selectedBase === 'object') {
            baseAddress = selectedBase.address || selectedBase.name || JSON.stringify(selectedBase);
        } else {
            baseAddress = selectedBase;
        }
        const service = new google.maps.DistanceMatrixService();

        // Calculate Base to Pickup
        service.getDistanceMatrix({
            origins: [baseAddress],
            destinations: [pickupAddress],
            travelMode: google.maps.TravelMode.DRIVING,
            unitSystem: google.maps.UnitSystem.METRIC
        }, (response, status) => {
            if (status === 'OK' && response.rows[0] && response.rows[0].elements[0] && response.rows[0].elements[0].status === 'OK') {
                const duration = response.rows[0].elements[0].duration.value; // in seconds
                const minutes = Math.round(duration / 60);
                if (baseToPickupInput) {
                    baseToPickupInput.value = minutes;
                }
            }
        });

        // Calculate Pickup to Dropoff
        service.getDistanceMatrix({
            origins: [pickupAddress],
            destinations: [dropoffAddress],
            travelMode: google.maps.TravelMode.DRIVING,
            unitSystem: google.maps.UnitSystem.METRIC
        }, (response, status) => {
            if (status === 'OK' && response.rows[0] && response.rows[0].elements[0] && response.rows[0].elements[0].status === 'OK') {
                const duration = response.rows[0].elements[0].duration.value; // in seconds
                const minutes = Math.round(duration / 60);
                if (pickupToDropoffInput) {
                    pickupToDropoffInput.value = minutes;
                }
            }
        });

        // Calculate Dropoff to Base
        service.getDistanceMatrix({
            origins: [dropoffAddress],
            destinations: [baseAddress],
            travelMode: google.maps.TravelMode.DRIVING,
            unitSystem: google.maps.UnitSystem.METRIC
        }, (response, status) => {
            if (status === 'OK' && response.rows[0] && response.rows[0].elements[0] && response.rows[0].elements[0].status === 'OK') {
                const duration = response.rows[0].elements[0].duration.value; // in seconds
                const minutes = Math.round(duration / 60);
                if (dropoffToBaseInput) {
                    dropoffToBaseInput.value = minutes;
                }
            }
        });
    }

    // Extract zip code from address
    function extractZipCode(address) {
        if (!address) return null;
        
        // Try to extract zip code from address string
        // Australian zip codes are 4 digits
        const zipCodeMatch = address.match(/\b(\d{4})\b/);
        if (zipCodeMatch) {
            return parseInt(zipCodeMatch[1]);
        }
        return null;
    }

    // Check if address is in Sydney based on zip code
    function isSydneyAddress(address) {
        const zipCode = extractZipCode(address);
        if (!zipCode) {
            console.log('Could not extract zip code from address:', address);
            return false;
        }
        
        const isSydney = sydneyZipCodes.includes(zipCode);
        console.log(`Zip code ${zipCode} is ${isSydney ? 'in' : 'not in'} Sydney`);
        return isSydney;
    }

    // Update job type based on pickup and dropoff locations
    function updateJobType() {
        const pickupInput = document.getElementById('pickup_location');
        const dropoffInput = document.getElementById('dropoff_location');
        const jobTypeSelect = document.getElementById('job_type');
        
        if (!pickupInput || !dropoffInput || !jobTypeSelect) {
            return;
        }
        
        const pickupAddr = pickupInput.value.trim();
        const dropoffAddr = dropoffInput.value.trim();
        
        // If either address is empty, don't update
        if (!pickupAddr || !dropoffAddr) {
            console.log('Addresses not complete, skipping job type update');
            return;
        }
        
        const pickupIsSydney = isSydneyAddress(pickupAddr);
        const dropoffIsSydney = isSydneyAddress(dropoffAddr);
        
        console.log('Pickup is Sydney:', pickupIsSydney);
        console.log('Dropoff is Sydney:', dropoffIsSydney);
        
        // If both are in Sydney → Local Job
        // If either is out of Sydney → Interstate Job
        if (pickupIsSydney && dropoffIsSydney) {
            jobTypeSelect.value = 'local_job';
            console.log('Job type set to: Local Job');
        } else {
            jobTypeSelect.value = 'interstate_job';
            console.log('Job type set to: Interstate Job');
        }
    }

    // Make functions available globally
    window.calculateTravelTimes = calculateTravelTimes;
    window.updateJobType = updateJobType;

    // Also calculate on manual input change (with debounce)
    let calculateTimeout;
    let jobTypeTimeout;
    
    document.getElementById('pickup_location')?.addEventListener('input', function() {
        clearTimeout(calculateTimeout);
        clearTimeout(jobTypeTimeout);
        
        jobTypeTimeout = setTimeout(() => {
            updateJobType();
        }, 500);
        
        calculateTimeout = setTimeout(() => {
            if (this.value.trim() && document.getElementById('dropoff_location').value.trim()) {
                calculateTravelTimes();
            }
        }, 1000);
    });

    document.getElementById('dropoff_location')?.addEventListener('input', function() {
        clearTimeout(calculateTimeout);
        clearTimeout(jobTypeTimeout);
        
        jobTypeTimeout = setTimeout(() => {
            updateJobType();
        }, 500);
        
        calculateTimeout = setTimeout(() => {
            if (this.value.trim() && document.getElementById('pickup_location').value.trim()) {
                calculateTravelTimes();
            }
        }, 1000);
    });
    
    // Also calculate travel times when locations are changed via autocomplete
    const pickupInput = document.getElementById('pickup_location');
    const dropoffInput = document.getElementById('dropoff_location');
    
    if (pickupInput) {
        pickupInput.addEventListener('change', function() {
            if (this.value.trim() && dropoffInput && dropoffInput.value.trim()) {
                setTimeout(() => {
                    calculateTravelTimes();
                }, 500);
            }
        });
    }
    
    if (dropoffInput) {
        dropoffInput.addEventListener('change', function() {
            if (this.value.trim() && pickupInput && pickupInput.value.trim()) {
                setTimeout(() => {
                    calculateTravelTimes();
                }, 500);
            }
        });
    }

    // Collect booking items from table
    function collectBookingItems() {
        const items = [];
        const assembleItems = [];
        const storageItems = [];
        
        const tbody = document.querySelector('tbody');
        if (!tbody) return { items, assembleItems, storageItems };
        
        const rows = tbody.querySelectorAll('tr');
        rows.forEach((row) => {
            const itemName = row.cells[0]?.textContent?.trim();
            const quantityInput = row.querySelector('.quantity-input');
            const quantity = parseInt(quantityInput?.value) || 0;
            const assembleCheckbox = row.querySelector('.assemble-checkbox');
            const assembleTypeSelect = row.querySelector('.assemble-type-select');
            const typeBadge = row.querySelector('.item-type-badge');
            
            if (!itemName || quantity <= 0) return;
            
            const itemType = typeBadge?.getAttribute('data-type') || 'regular';
            const isAssemble = assembleCheckbox?.checked || false;
            const assembleType = assembleTypeSelect?.value || '';
            
            // Get entity_id and entity_type from row data attributes
            const entityId = row.getAttribute('data-entity-id');
            const entityType = row.getAttribute('data-entity-type');
            
            if (!entityId) {
                console.warn('Row missing entity_id:', row);
                return;
            }
            
            if (itemType === 'assemble_disassemble' && isAssemble && assembleType) {
                assembleItems.push({
                    assemble_disassemble_category_id: parseInt(entityId),
                    assemble_disassemble_type: assembleType,
                    quantity: quantity
                });
            } else if (itemType === 'storage_items') {
                storageItems.push({
                    storage_item_id: parseInt(entityId),
                    quantity: quantity
                });
            } else {
                items.push({
                    category_id: parseInt(entityId),
                    quantity: quantity
                });
            }
        });
        
        return { items, assembleItems, storageItems };
    }

    // Collect form data and prepare payload
    function prepareUpdatePayload() {
        const bookingItems = collectBookingItems();
        
        // Get quote UUID
        const uuid = quoteData?.quote?.uuid || '';
        
        // Get form values
        const status = document.getElementById('status')?.value || 'pending';
        const jobType = document.getElementById('job_type')?.value || 'local_job';
        const extraMovers = parseInt(document.getElementById('extra_movers')?.value) || 0;
        const baseToPickupTime = parseFloat(document.getElementById('base_to_pickup')?.value) || 0;
        const dropoffToBase = parseFloat(document.getElementById('dropoff_to_base')?.value) || 0;
        const pickupToDropoff = parseFloat(document.getElementById('pickup_to_dropoff')?.value) || 0;
        const moveDate = document.getElementById('moving_date')?.value || '';
        const pickupLocation = document.getElementById('pickup_location')?.value || '';
        const dropoffLocation = document.getElementById('dropoff_location')?.value || '';
        
        // Format move_date - date picker already returns YYYY-MM-DD format
        let formattedMoveDate = moveDate;
        if (moveDate && moveDate.includes('/')) {
            const parts = moveDate.split('/');
            if (parts.length === 3) {
                formattedMoveDate = `${parts[2]}-${parts[0].padStart(2, '0')}-${parts[1].padStart(2, '0')}`;
            }
        }
        
        const payload = {
            uuid: uuid,
            items: bookingItems.items,
            assemble_items: bookingItems.assembleItems,
            storage_items: bookingItems.storageItems,
            status: status,
            job_type: jobType,
            extra_movers: extraMovers,
            base_to_pickup_time: baseToPickupTime,
            dropof_to_base: dropoffToBase,
            pickup_to_dropof: pickupToDropoff,
            move_date: formattedMoveDate,
            pickup_location: pickupLocation,
            dropoff_location: dropoffLocation
            // Don't send remaining_amount - let backend calculate it from discounted total_cost
        };
        
        return payload;
    }

    // Update quote
    async function updateQuote() {
        if (!quoteId) {
            alert('Quote ID not found');
            return;
        }
        
        const saveBtn = document.getElementById('saveChangesBtn');
        const originalText = saveBtn.innerHTML;
        
        try {
            // Disable button and show loading
            saveBtn.disabled = true;
            saveBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Saving...';
            
            const payload = prepareUpdatePayload();
            console.log('Update Payload:', payload);
            
            const response = await fetch(`/admin/quotes/${quoteId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || ''
                },
                body: JSON.stringify(payload)
            });
            
            if (!response.ok) {
                let errorMessage = 'Error updating quote.';
                
                if (response.status === 404) {
                    errorMessage = 'Quote not found.';
                } else if (response.status === 422) {
                    const errorData = await response.json().catch(() => ({}));
                    errorMessage = errorData.message || 'Validation error. Please check your input.';
                } else if (response.status === 500) {
                    errorMessage = 'Server error. Please try again later.';
                } else {
                    errorMessage = `Error ${response.status}: ${response.statusText}`;
                }
                
                throw new Error(errorMessage);
            }
            
            const data = await response.json();
            console.log('Update Response:', data);
            
            // Show success message
            if (data.status === 'success' || data.message) {
                // Show success alert
                const alertDiv = document.createElement('div');
                alertDiv.className = 'alert alert-success alert-dismissible fade show';
                alertDiv.innerHTML = `
                    <strong>Success!</strong> ${data.message || 'Quote updated successfully.'}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                document.querySelector('.page-content')?.insertBefore(alertDiv, document.querySelector('.page-content').firstChild);
                
                // Optionally redirect or reload
                setTimeout(() => {
                    window.location.href = `{{ route('admin.quotes.show', $quoteId) }}`;
                }, 1500);
            } else {
                throw new Error('Unexpected response format');
            }
            
        } catch (error) {
            console.error('Error updating quote:', error);
            
            // Show error alert
            const alertDiv = document.createElement('div');
            alertDiv.className = 'alert alert-danger alert-dismissible fade show';
            alertDiv.innerHTML = `
                <strong>Error!</strong> ${error.message || 'Failed to update quote. Please try again.'}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.querySelector('.page-content')?.insertBefore(alertDiv, document.querySelector('.page-content').firstChild);
            
        } finally {
            // Re-enable button
            saveBtn.disabled = false;
            saveBtn.innerHTML = originalText;
        }
    }

    // Add event listener to Save Changes button
    document.getElementById('saveChangesBtn')?.addEventListener('click', function() {
        updateQuote();
    });

    // Show toast notification
    function showToast(message, type = 'warning') {
        // Remove any existing toast
        const existingToast = document.getElementById('validation-toast');
        if (existingToast) {
            existingToast.remove();
        }
        
        // Create toast element
        const toast = document.createElement('div');
        toast.id = 'validation-toast';
        toast.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
        toast.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);';
        toast.innerHTML = `
            <strong>${type === 'warning' ? 'Warning' : 'Info'}:</strong> ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        `;
        
        document.body.appendChild(toast);
        
        // Auto-remove after 3 seconds
        setTimeout(() => {
            if (toast && toast.parentNode) {
                toast.remove();
            }
        }, 3000);
    }

    // Store flatpickr instance globally
    let flatpickrInstance = null;

    // Initialize Bootstrap Date Picker for moving date
    function initializeDatePicker() {
        const movingDateInput = document.getElementById('moving_date');
        const calendarIcon = document.getElementById('calendar-icon');
        
        if (movingDateInput) {
            // Wait for flatpickr to be available
            if (typeof flatpickr !== 'undefined') {
                // Destroy existing instance if any
                if (movingDateInput._flatpickr) {
                    movingDateInput._flatpickr.destroy();
                }
                
                flatpickrInstance = flatpickr(movingDateInput, {
                    dateFormat: 'Y-m-d',
                    minDate: 'today',
                    allowInput: true,
                    clickOpens: true,
                    wrap: false,
                    static: false,
                    enableTime: false,
                    time_24hr: false
                });
                
                // Make calendar icon clickable to open date picker
                if (calendarIcon) {
                    calendarIcon.addEventListener('click', function() {
                        if (flatpickrInstance) {
                            flatpickrInstance.open();
                        }
                    });
                }
            } else {
                // Retry after a short delay if flatpickr is not loaded yet
                setTimeout(() => {
                    if (typeof flatpickr !== 'undefined') {
                        if (movingDateInput._flatpickr) {
                            movingDateInput._flatpickr.destroy();
                        }
                        flatpickrInstance = flatpickr(movingDateInput, {
                            dateFormat: 'Y-m-d',
                            minDate: 'today',
                            allowInput: true,
                            clickOpens: true,
                            wrap: false,
                            static: false,
                            enableTime: false,
                            time_24hr: false
                        });
                        
                        // Make calendar icon clickable to open date picker
                        if (calendarIcon) {
                            calendarIcon.addEventListener('click', function() {
                                if (flatpickrInstance) {
                                    flatpickrInstance.open();
                                }
                            });
                        }
                    }
                }, 500);
            }
        }
    }

    // Validate extra_movers input
    const extraMoversInput = document.getElementById('extra_movers');
    const extraMoversAlert = document.getElementById('extra_movers_alert');
    const extraMoversAlertMessage = document.getElementById('extra_movers_alert_message');
    
    function showExtraMoversAlert(message) {
        if (extraMoversAlert && extraMoversAlertMessage) {
            extraMoversAlertMessage.textContent = message;
            extraMoversAlert.style.display = 'block';
            // Auto-hide after 5 seconds
            setTimeout(() => {
                if (extraMoversAlert) {
                    extraMoversAlert.style.display = 'none';
                }
            }, 5000);
        }
    }
    
    function hideExtraMoversAlert() {
        if (extraMoversAlert) {
            extraMoversAlert.style.display = 'none';
        }
    }
    
    if (extraMoversInput) {
        extraMoversInput.addEventListener('input', function() {
            const value = parseInt(this.value) || 0;
            if (value > 2) {
                this.value = 2;
                showExtraMoversAlert('Extra movers cannot exceed 2. Value has been set to 2.');
            } else if (value < 0) {
                this.value = 0;
                hideExtraMoversAlert();
            } else {
                hideExtraMoversAlert();
            }
        });
        
        extraMoversInput.addEventListener('change', function() {
            const value = parseInt(this.value) || 0;
            if (value > 2) {
                this.value = 2;
                showExtraMoversAlert('Extra movers cannot exceed 2. Value has been set to 2.');
            } else if (value < 0) {
                this.value = 0;
                hideExtraMoversAlert();
            } else {
                hideExtraMoversAlert();
            }
        });
        
        // Also validate on blur
        extraMoversInput.addEventListener('blur', function() {
            const value = parseInt(this.value) || 0;
            if (value > 2) {
                this.value = 2;
                showExtraMoversAlert('Extra movers cannot exceed 2. Value has been set to 2.');
            } else if (value < 0) {
                this.value = 0;
                hideExtraMoversAlert();
            } else {
                hideExtraMoversAlert();
            }
        });
    }

    // Fetch data on page load
    fetchBases();
    fetchQuoteData();
    
    // Initialize date picker after form is loaded and flatpickr is available
    function waitForFlatpickr() {
        if (typeof flatpickr !== 'undefined') {
            initializeDatePicker();
        } else {
            setTimeout(waitForFlatpickr, 100);
        }
    }
    
    // Start waiting for flatpickr after a short delay to ensure scripts are loaded
    setTimeout(() => {
        waitForFlatpickr();
    }, 500);
    
    // Also try to initialize after form content is shown
    const observer = new MutationObserver(function(mutations) {
        const formContent = document.getElementById('quoteFormContent');
        if (formContent && formContent.style.display !== 'none') {
            setTimeout(() => {
                initializeDatePicker();
            }, 500);
            observer.disconnect();
        }
    });
    
    const formContainer = document.getElementById('quoteFormContainer');
    if (formContainer) {
        observer.observe(formContainer, { attributes: true, attributeFilter: ['style'] });
    }
    
    // Check job type on page load if addresses are already filled
    setTimeout(() => {
        updateJobType();
    }, 2000);
});
</script>

@include('Admin.Includes.footer')

