<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use Illuminate\Http\Request;

class BlogController extends Controller
{
    public function index(Request $request)
    {
        $blogs = Blog::where('active', 1)
            ->orderBy('created_at', 'desc')
            ->get();
        return $this->success($blogs , 'Blogs retrieved successfully');
    }

    public function getBlogbyPermalink(Blog $blog,$permalink)
    {
        $mainBlog = $blog->getBlogwithPermalink($permalink);
        if (!$mainBlog) {
            return $this->error('Blog not found', 404);
        }
        $relatedBlogs = $this->getRelatedBlogs($mainBlog);
        $response = [
            'blog' => $mainBlog,
            'related_blogs' => $relatedBlogs ?? [],
        ];
        return $this->success($response);
    }


    private function getRelatedBlogs(Blog $mainBlog, $limit = 10)
    {
        $relatedBlogIds = collect();
        
        // Get blogs related by blog categories using blog_categories_id
        $mainBlogCategoryIds = $mainBlog->blog_category_ids;
        if (!empty($mainBlogCategoryIds) && is_array($mainBlogCategoryIds)) {
            // Find blogs that share any category ID
            $categoryRelatedBlogs = Blog::where('id', '!=', $mainBlog->id)
                ->where('active', 1)
                ->get()
                ->filter(function ($blog) use ($mainBlogCategoryIds) {
                    $blogCategoryIds = $blog->blog_category_ids;
                    if (empty($blogCategoryIds) || !is_array($blogCategoryIds)) {
                        return false;
                    }
                    return !empty(array_intersect($mainBlogCategoryIds, $blogCategoryIds));
                })
                ->pluck('id');
            $relatedBlogIds = $relatedBlogIds->merge($categoryRelatedBlogs);
        }
        
        // Remove duplicates and get unique blog IDs
        $uniqueRelatedBlogIds = $relatedBlogIds->unique()->values();
        
        // If we have related blog IDs, fetch them
        if ($uniqueRelatedBlogIds->count() > 0) {
            return Blog::whereIn('id', $uniqueRelatedBlogIds->toArray())
                ->where('active', 1)
                ->orderBy('created_at', 'desc')
                ->limit($limit)
                ->get();
        }
        
        // If no related blogs found, return recent active blogs
        return Blog::where('id', '!=', $mainBlog->id)
            ->where('active', 1)
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }
}
