<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Property;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\ValidationTrait;
use Carbon\Carbon;

class PropertyController extends Controller
{
    use JsonResponseTrait, ValidationTrait;

    public function store(Request $request)
    {
        // Use provided UUID or generate new one
        $uuid = $request->uuid ?? Str::uuid()->toString();
        $result = [];

        // Parse and validate move_date - store as date only (no timezone conversion)
        $moveDate = null;
        if ($request->has('move_date') && !empty($request->move_date)) {
            // Validate and format date string directly without timezone conversion
            try {
                // Validate the date format is Y-m-d
                $parsedDate = Carbon::createFromFormat('Y-m-d', $request->move_date);
                // Return as date string only (no time, no timezone)
                $moveDate = $parsedDate->format('Y-m-d');
            } catch (\Exception $e) {
                // Fallback: try to parse and extract date part only
                $parsedDate = Carbon::parse($request->move_date);
                // Return as date string only (no time, no timezone)
                $moveDate = $parsedDate->format('Y-m-d');
            }
        }
        if ($request->has('pickup_property') && !empty($request->pickup_property)) {
            $pickupProperty = Property::where('uuid', $uuid)
                ->where('type', 'pick_up')
                ->first();
            $fillableFields = ['location', 'type', 'move_date'];
            $pickupData = array_fill_keys($fillableFields, null);
            $requestData = $request->pickup_property;
            foreach ($fillableFields as $field) {
                if (array_key_exists($field, $requestData)) {
                    $pickupData[$field] = $requestData[$field];
                }
            }
            // Only set move_date if it was provided and parsed
            if ($moveDate) {
                $pickupData['move_date'] = $moveDate; // Use parsed date string (Y-m-d format)
            }
            $pickupData['type'] = 'pick_up';
            $pickupData['uuid'] = $uuid;
            if ($pickupProperty) {
                $pickupProperty->update($pickupData);
                $pickupProperty->refresh();
            } else {
                $pickupProperty = Property::create($pickupData);
            }
            // Format move_date for response to ensure it's a string
            if ($pickupProperty->move_date) {
                $pickupProperty->setAttribute('move_date', $pickupProperty->move_date instanceof \Carbon\Carbon 
                    ? $pickupProperty->move_date->format('Y-m-d') 
                    : $pickupProperty->move_date);
            }
            $result['pickup_property'] = $pickupProperty;
        }
        if ($request->has('dropoff_property') && !empty($request->dropoff_property)) {
            $dropoffProperty = Property::where('uuid', $uuid)
                ->where('type', 'drop_off')
                ->first();

            $fillableFields = ['location', 'type', 'move_date'];
            $dropoffData = array_fill_keys($fillableFields, null);
            $requestData = $request->dropoff_property;

            foreach ($fillableFields as $field) {
                if (array_key_exists($field, $requestData)) {
                    $dropoffData[$field] = $requestData[$field] ?? null;
                }
            }

            // Only set move_date if it was provided and parsed
            if ($moveDate) {
                $dropoffData['move_date'] = $moveDate; // Use parsed date string (Y-m-d format)
            }
            $dropoffData['type'] = 'drop_off';
            $dropoffData['uuid'] = $uuid;

            if ($dropoffProperty) {
                $dropoffProperty->update($dropoffData);
                $dropoffProperty->refresh();
            } else {
                $dropoffProperty = Property::create($dropoffData);
            }
            // Format move_date for response to ensure it's a string
            if ($dropoffProperty->move_date) {
                $dropoffProperty->setAttribute('move_date', $dropoffProperty->move_date instanceof \Carbon\Carbon 
                    ? $dropoffProperty->move_date->format('Y-m-d') 
                    : $dropoffProperty->move_date);
            }
            $result['dropoff_property'] = $dropoffProperty;
        }

        $result['uuid'] = $uuid;

        return $this->success($result, 'Properties saved successfully');
    }


    /**
     * Get properties by UUID
     */
    public function getByUuid(string $uuid)
    {
        $properties = Property::where('uuid', $uuid)->get();
        if ($properties->isEmpty()) {
            return $this->error('Property not found', 404);
        }
        $result = [
            'uuid' => $uuid,
            'move_date' => $properties->where('type', 'pick_up')->first()->move_date,
            'pickup' => $properties->where('type', 'pick_up')->select('location',)->first(),
            'dropoff' => $properties->where('type', 'drop_off')->select('location')->first(),
        ];
        return $this->success($result, 'Property fetched successfully');
    }
}
