<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\Quote;
use App\Models\PaymentProof;
use Illuminate\Http\Request;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Http\Controllers\API\Traits\QuoteCalculationTrait as TraitsQuoteCalculationTrait;
use App\Http\Controllers\API\Traits\StripeTrait;
use App\Http\Controllers\API\Traits\EmailTrait;
use App\Models\Property;
use App\Models\PropertyDetail;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class QuoteController extends Controller
{
    use JsonResponseTrait, TraitsQuoteCalculationTrait, StripeTrait, EmailTrait;

   

    /**
     * Get quote by UUID
     */
    public function getByUuid(string $uuid)
    {
        $quote = Quote::where('uuid', $uuid)
            ->with(['user', 'truck'])
            ->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        $pickUpProperty = Property::where('uuid', $uuid)->where('type', 'pick_up')->first();
        $dropOffProperty = Property::where('uuid', $uuid)->where('type', 'drop_off')->first();
        $quote['pickup_property'] = $pickUpProperty;
        $quote['dropoff_property'] = $dropOffProperty;
        // Get property details with only id and property_category
        $propertyDetailsPickUp = null;
        $propertyDetailsDropOff = null;
        if ($pickUpProperty) {
            $propertyDetailsPickUp = PropertyDetail::where('property_id', $pickUpProperty->id)
                ->where('property_type', 'pick_up')
                ->select('id', 'property_category')
                ->first();
        }

        if ($dropOffProperty) {
            $propertyDetailsDropOff = PropertyDetail::where('property_id', $dropOffProperty->id)
                ->where('property_type', 'drop_off')
                ->select('id', 'property_category')
                ->first();
        }
        $quote['property_details_pickup'] = $propertyDetailsPickUp;
        $quote['property_details_dropoff'] = $propertyDetailsDropOff;
        return $this->success($quote, 'Quote fetched successfully');
    }

    /**
     * Update quote and process Stripe payment (30% deposit or full amount)
     * Simplified one-step flow: If payment_method_id is provided, payment is charged immediately
     */
    public function update(Request $request)
    {
        $quote = Quote::where('uuid', $request->uuid)->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        $updateData = $request->only(['status']);
        if ($request->has('payment_type')) {
            $paymentMethod = $request->input('payment_method', 'stripe'); // 'stripe' or 'bank_transfer'
            if (!in_array($paymentMethod, ['stripe', 'bank_transfer'])) {
                return $this->error('Invalid payment method. Must be "stripe" or "bank_transfer"', 422);
            }
            $paymentType = $request->payment_type; // 'deposit' or 'full'
            if (!in_array($paymentType, ['deposit', 'full'])) {
                return $this->error('Invalid payment type. Must be "deposit" or "full"', 422);
            }
            if ($paymentMethod == 'bank_transfer') {

                if (!$request->hasFile('quote_payment_proof')) {
                    return $this->error('Payment proof image is required for bank transfer', 422);
                }
                $file = $request->file('quote_payment_proof');
                if (!$file->isValid() || !in_array(strtolower($file->getClientOriginalExtension()), ['jpg', 'jpeg', 'webp', 'png', 'pdf'])) {
                    return $this->error('Invalid file type. Please upload an image (jpg, jpeg, webp, png) or PDF', 422);
                }
                $uploadPath = public_path('quotes/payment_proofs');
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }
                $filename = 'payment_proof_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $filePath = $uploadPath . '/' . $filename;
                $file->move($uploadPath, $filename);
                $relativePath = 'quotes/payment_proofs/' . $filename;
                
                // Save payment proof to payment_proofs table
                $paymentProof = PaymentProof::create([
                    'quote_id' => $quote->id,
                    'payment_type' => $paymentType === 'deposit' ? 'deposit' : 'remaining',
                    'proof_image' => $relativePath,
                    'amount' => 0, // Will be updated below
                    'status' => 'pending',
                ]);
                
                $totalAmount = (float) $quote->total_cost;
                $jobType = null;
                if ($quote->raw_data && isset($quote->raw_data['job_type'])) {
                    $jobType = $quote->raw_data['job_type'];
                } elseif ($quote->job_type) {
                    $jobType = $quote->job_type;
                }
                $paymentAmount = 0;
                $remainingAmount = 0;
                if ($paymentType === 'deposit') {
                    $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
                    if ($jobType === 'local_job') {
                        $depositAmount = (float) ($generalSettings['_local_job_deposit_amount'] ?? 100);
                        $paymentAmount = round($depositAmount, 2);
                        $remainingAmount = round($totalAmount - $paymentAmount, 2);
                    } elseif ($jobType === 'interstate_job') {
                        // Interstate job: percentage deposit
                        $depositPercentage = (float) ($generalSettings['_interstate_job_deposit_percentage'] ?? 20);
                        $paymentAmount = round($totalAmount * ($depositPercentage / 100), 2);
                        $remainingAmount = round($totalAmount - $paymentAmount, 2);
                    } else {
                        // Fallback: 30% deposit if job type is not set
                        $paymentAmount = round($totalAmount * 0.30, 2);
                        $remainingAmount = round($totalAmount - $paymentAmount, 2);
                    }
                } else {
                    // Full payment
                    $paymentAmount = $totalAmount;
                    $remainingAmount = 0;
                }

                // Update payment proof with the correct amount
                if (isset($paymentProof)) {
                    $paymentProof->update(['amount' => $paymentAmount]);
                }

                // Update quote with bank transfer payment information
                $updateData['payment_method'] = 'bank_transfer'; // Save directly to payment_method column
                $updateData['payment_type'] = $paymentType;
                $updateData['deposit_amount'] = $paymentAmount;
                $updateData['remaining_amount'] = $remainingAmount;
                $updateData['deposit_payment_status'] = 'pending'; // Pending until admin approves

                // For full payment, also set remaining payment status
                if ($paymentType === 'full') {
                    $updateData['remaining_payment_status'] = 'pending';
                }

                // Convert to order if not already
                if ($quote->quote_type !== 'order') {
                    $updateData['quote_type'] = 'order';
                    if (!$quote->order_number) {
                        $updateData['order_number'] = Quote::generateOrderNumber();
                    }
                }

                // Update quote
                $quote->update($updateData);
                $quote->refresh();
                $quote->load(['user']);


                if ($paymentType === 'deposit') {
                    // Send deposit order confirmation email (payment is pending but order is placed)
                    $emailResults = $this->sendOrderConfirmationEmail($quote, 'deposit', 'pending', false);
                    if (!$emailResults['customer']) {
                        Log::warning('Failed to send order confirmation email to customer for bank transfer deposit - quote ID: ' . $quote->id);
                    }
                    if (!$emailResults['admin']) {
                        Log::warning('Failed to send order confirmation email to admin for bank transfer deposit - quote ID: ' . $quote->id);
                    }
                } else {
                    // Full payment - send full order confirmation email
                    $emailResults = $this->sendOrderConfirmationEmail($quote, 'full', 'pending', false);
                    if (!$emailResults['customer']) {
                        Log::warning('Failed to send order confirmation email to customer for bank transfer full payment - quote ID: ' . $quote->id);
                    }
                    if (!$emailResults['admin']) {
                        Log::warning('Failed to send order confirmation email to admin for bank transfer full payment - quote ID: ' . $quote->id);
                    }
                }
                // Prepare response data
                $responseData = [
                    'quote' => $quote,
                    'payment_method' => 'bank_transfer',
                    'payment_status' => 'pending',
                    'payment_amount' => $paymentAmount,
                    'remaining_amount' => $remainingAmount,
                    'total_amount' => $totalAmount,
                    'payment_proof_path' => $relativePath,
                ];

                $responseData['pickup_property'] = Property::where('uuid', $quote->uuid)->where('type', 'pick_up')->first();
                $responseData['dropoff_property'] = Property::where('uuid', $quote->uuid)->where('type', 'drop_off')->first();
                return $this->success($responseData, 'Bank transfer payment proof uploaded successfully. Payment is pending admin approval.');
            }
            
            // Handle Stripe payment (existing code)
            try {
                $paymentMethodId = $request->input('payment_method_id'); // From frontend Stripe Card Element
                // Check if payment already exists
                if ($paymentType === 'deposit' && $quote->stripe_payment_intent_deposit_id) {
                    return $this->error('Deposit payment intent already exists', 422);
                }
                // Create and optionally confirm payment intent (one-step if payment_method_id provided)
                $paymentResult = $this->createPaymentIntentForQuote($quote, $quote->total_cost, $paymentType, $paymentMethodId);
                $paymentIntent = $paymentResult['payment_intent'];
                $customer = $paymentResult['customer'];

                // Update quote with Stripe information
                $updateData['stripe_customer_id'] = $customer->id;

                // Determine payment status based on payment intent status
                $paymentStatus = 'pending';
                if ($paymentIntent->status === 'succeeded') {
                    $paymentStatus = 'succeeded';
                } elseif ($paymentIntent->status === 'requires_action') {
                    $paymentStatus = 'processing'; // 3D Secure required
                } elseif ($paymentIntent->status === 'canceled' || $paymentIntent->status === 'payment_failed') {
                    $paymentStatus = 'failed';
                }

                if ($paymentType === 'deposit') {
                    $updateData['stripe_payment_intent_deposit_id'] = $paymentIntent->id;
                    $updateData['deposit_amount'] = $paymentResult['payment_amount'];
                    $updateData['remaining_amount'] = $paymentResult['remaining_amount'];
                    $updateData['deposit_payment_status'] = $paymentStatus;
                } else {
                    // Full payment - store in deposit fields for consistency
                    $updateData['stripe_payment_intent_deposit_id'] = $paymentIntent->id;
                    $updateData['deposit_amount'] = $paymentResult['payment_amount'];
                    $updateData['remaining_amount'] = 0;
                    $updateData['deposit_payment_status'] = $paymentStatus;
                    // When payment is full, set remaining_payment_status to succeeded
                    if ($paymentStatus === 'succeeded') {
                        $updateData['remaining_payment_status'] = 'succeeded';
                    }
                }

                // Store payment_method directly to column
                $updateData['payment_method'] = 'stripe';

                // Convert to order if not already
                if ($quote->quote_type !== 'order') {
                    $updateData['quote_type'] = 'order';
                    if (!$quote->order_number) {
                        $updateData['order_number'] = Quote::generateOrderNumber();
                    }
                }

                // Update quote
                $quote->update($updateData);
                $quote->refresh();
                $quote->load(['user']);

                // Prepare response data for frontend
                $responseData = [
                    'quote' => $quote,
                    'payment_status' => $paymentStatus,
                    'amount_charged' => $paymentResult['payment_amount'],
                    'remaining_amount' => $paymentResult['remaining_amount'],
                    'total_amount' => $paymentResult['total_amount'],
                ];

                $responseData['pickup_property'] = Property::where('uuid', $quote->uuid)->where('type', 'pick_up')->first();
                $responseData['dropoff_property'] = Property::where('uuid', $quote->uuid)->where('type', 'drop_off')->first();

                // If deposit succeeded, also create and return remaining payment link
                if ($paymentType === 'deposit' && $paymentStatus === 'succeeded') {
                    $paymentLinkResult = $this->createRemainingPaymentLink($quote);
                    $responseData['remaining_payment_link'] = $paymentLinkResult['payment_link_url'];
                    // Update quote with payment link
                    $quote->update(['remaining_payment_url' => $paymentLinkResult['payment_link_url']]);
                }

                $message = $paymentStatus === 'succeeded'
                    ? 'Payment charged successfully!'
                    : ($paymentStatus === 'requires_action'
                        ? 'Payment requires authentication. Use client_secret to complete 3D Secure.'
                        : 'Payment intent created. Use client_secret to confirm payment on frontend.');
                return $this->success($responseData, $message);
            } catch (\Stripe\Exception\ApiErrorException $e) {
                Log::error('Stripe API Error in quote update: ' . $e->getMessage());
                $errorMessage = $this->getStripeErrorMessage($e);
                return $this->error($errorMessage, 422);
            } catch (\Exception $e) {
                Log::error('Error processing payment: ' . $e->getMessage());
                return $this->error('Error processing payment: ' . $e->getMessage(), 500);
            }
        }
        // Regular update without payment
        $quote->update($updateData);
        $quote->refresh();
        $quote->load(['user']);

        return $this->success($quote, 'Quote updated successfully');
    }

    /**
     * Create remaining payment link (70%)
     * This is called after work completion to send payment link to customer via email
     */
    public function createRemainingPaymentLinkEndpoint(Request $request)
    {
        $quote = Quote::where('uuid', $request->uuid)->first();
        if (!$quote) {
            return $this->error('Quote not found', 404);
        }
        // Check if deposit was paid
        if ($quote->deposit_payment_status !== 'succeeded') {
            return $this->error('Deposit payment must be completed first', 422);
        }
        // Check if remaining payment already exists and is completed
        if ($quote->remaining_payment_status === 'succeeded') {
            return $this->error('Remaining payment already completed', 422);
        }
        // Check if payment link already exists
        if ($quote->remaining_payment_url) {
            return $this->success([
                'quote' => $quote->fresh(),
                'payment_link' => $quote->remaining_payment_url,
                'amount' => $quote->remaining_amount,
            ], 'Payment link already exists');
        }
        try {
            // Create remaining payment link using StripeTrait
            $paymentLinkResult = $this->createRemainingPaymentLink($quote);
            $paymentLink = $paymentLinkResult['payment_link'];
            $paymentLinkUrl = $paymentLinkResult['payment_link_url'];
            $customer = $paymentLinkResult['customer'];
            // Ensure customer ID is saved
            if (!$quote->stripe_customer_id) {
                $quote->update(['stripe_customer_id' => $customer->id]);
            }
            // Update quote with remaining payment link
            $quote->update([
                'remaining_payment_url' => $paymentLinkUrl,
                'remaining_payment_status' => 'pending',
            ]);
            return $this->success([
                'quote' => $quote->fresh(),
                'payment_link' => $paymentLinkUrl,
                'amount' => $paymentLinkResult['remaining_amount'],
            ], 'Payment link created successfully. Send this link to customer via email.');
        } catch (\Stripe\Exception\ApiErrorException $e) {
            Log::error('Stripe API Error creating payment link: ' . $e->getMessage());
            $errorMessage = $this->getStripeErrorMessage($e);
            return $this->error($errorMessage, 422);
        } catch (\Exception $e) {
            Log::error('Error creating payment link: ' . $e->getMessage());
            return $this->error('Error creating payment link: ' . $e->getMessage(), 500);
        }
    }


    /**
     * Get payment intent status (for checking payment status)
     */
    public function getPaymentIntentStatus(Request $request)
    {
        $request->validate([
            'payment_intent_id' => 'required|string',
        ]);

        try {
            $this->initializeStripe();
            $paymentIntent = \Stripe\PaymentIntent::retrieve($request->payment_intent_id);

            return $this->success([
                'payment_intent_id' => $paymentIntent->id,
                'status' => $paymentIntent->status,
                'amount' => $paymentIntent->amount / 100,
                'currency' => $paymentIntent->currency,
            ], 'Payment intent status retrieved');
        } catch (\Exception $e) {
            Log::error('Error retrieving payment intent status: ' . $e->getMessage());
            return $this->error('Error retrieving payment status: ' . $e->getMessage(), 500);
        }
    }

    public function getUserQuotes(Request $request)
    {
        $user = $request->user();
        $booking = Booking::where('user_id', $user->id)->get();
        $quotes = Quote::whereIn('booking_id', $booking->pluck('id'))
            ->orderByDesc('created_at')
            ->get();

        // Get all UUIDs from quotes
        $uuids = $quotes->pluck('uuid')->filter()->unique();

        // Fetch all properties for these quotes (both pick_up and drop_off)
        $properties = Property::whereIn('uuid', $uuids)
            ->get()
            ->groupBy('uuid');

        // Format quotes with their related properties
        $formatted = $quotes->map(function (Quote $quote) use ($properties) {
            $data = $quote->toArray();

            // Get properties for this quote's UUID
            $quoteProperties = $properties->get($quote->uuid, collect());

            // Separate pick_up and drop_off properties
            $pickUpProperty = $quoteProperties->where('type', 'pick_up')->first();
            $dropOffProperty = $quoteProperties->where('type', 'drop_off')->first();

            // Add properties to the quote data
            $data['pickup_property'] = $pickUpProperty ? $pickUpProperty->toArray() : null;
            $data['dropoff_property'] = $dropOffProperty ? $dropOffProperty->toArray() : null;

            return $data;
        });

        return $this->success($formatted, 'User quotes fetched successfully');
    }

    public function getUserQuotesUuid(string $uuid)
    {
        $quotes = Quote::with([
            'user',
            'truck',
            'paymentProofs',
            'booking.bookingItems.entity',
            'booking.pickUpProperty.propertyDetails',
            'booking.dropOffProperty.propertyDetails',
            'booking.truck',

        ])
            ->where('uuid', $uuid)
            ->orderByDesc('created_at')
            ->get();

        // Get all UUIDs from quotes
        $uuids = $quotes->pluck('uuid')->filter()->unique();

        // Fetch all properties for these quotes (both pick_up and drop_off) matched by UUID
        $properties = Property::whereIn('uuid', $uuids)
            ->with('propertyDetails')
            ->get()
            ->groupBy('uuid');

        // Format quotes with all related data
        $formatted = $quotes->map(function (Quote $quote) use ($properties) {
            $data = $quote->toArray();

            // Get properties for this quote's UUID (matched by UUID, not through booking)
            $quoteProperties = $properties->get($quote->uuid, collect());

            // Separate pick_up and drop_off properties matched by UUID
            $pickUpProperty = $quoteProperties->where('type', 'pick_up')->first();
            $dropOffProperty = $quoteProperties->where('type', 'drop_off')->first();

            // Add properties matched by UUID to the quote data
            $data['pickup_property'] = $pickUpProperty ? $pickUpProperty->toArray() : null;
            $data['dropoff_property'] = $dropOffProperty ? $dropOffProperty->toArray() : null;

            // Get property details for UUID-matched properties
            $propertyDetailsPickUp = null;
            $propertyDetailsDropOff = null;
            if ($pickUpProperty && $pickUpProperty->propertyDetails) {
                $propertyDetailsPickUp = $pickUpProperty->propertyDetails->toArray();
            }
            if ($dropOffProperty && $dropOffProperty->propertyDetails) {
                $propertyDetailsDropOff = $dropOffProperty->propertyDetails->toArray();
            }
            $data['property_details_pickup'] = $propertyDetailsPickUp;
            $data['property_details_dropoff'] = $propertyDetailsDropOff;

            return $data;
        });
        return $this->success($formatted, 'User quotes fetched successfully');
    }

   

    /**
     * Upload payment proof for a quote by UUID
     * User submits payment proof image with amount and payment type
     */
    public function uploadPaymentProof(Request $request)
    {

            // Validate request
            $request->validate([
                'payment_proof_image' => 'required|image|mimes:jpg,jpeg,png,webp|max:10240', // Max 10MB
                'amount' => 'required|numeric|min:0',
                'payment_type' => 'required|in:deposit,remaining',
            ]);

            // Find quote by UUID with user relationship
            $quote = Quote::with('user')->where('uuid', $request->uuid)->first();
            
            if (!$quote) {
                return $this->error('Quote not found', 404);
            }

            // Ensure quote amounts are consistent: total_cost = deposit_amount + remaining_amount
            // Recalculate if they don't match (fixes any data inconsistencies)
            $totalCost = (float) ($quote->total_cost ?? 0);
            $depositAmount = (float) ($quote->deposit_amount ?? 0);
            $remainingAmount = (float) ($quote->remaining_amount ?? 0);
            $expectedRemaining = round($totalCost - $depositAmount, 2);
            
            // If amounts don't match, fix the remaining_amount
            if ($totalCost > 0 && abs($remainingAmount - $expectedRemaining) > 0.01) {
                // Recalculate remaining_amount based on total_cost and deposit_amount
                $correctedRemaining = max(0, $expectedRemaining);
                $quote->remaining_amount = $correctedRemaining;
                $quote->save();
                
                Log::info('Corrected remaining_amount in uploadPaymentProof', [
                    'quote_id' => $quote->id,
                    'old_remaining' => $remainingAmount,
                    'corrected_remaining' => $correctedRemaining,
                    'total_cost' => $totalCost,
                    'deposit_amount' => $depositAmount,
                ]);
            }

            // Handle file upload
            if (!$request->hasFile('payment_proof_image')) {
                return $this->error('Payment proof image is required', 422);
            }

            $file = $request->file('payment_proof_image');
            
            // Validate file
            if (!$file->isValid() || !in_array(strtolower($file->getClientOriginalExtension()), ['jpg', 'jpeg', 'webp', 'png'])) {
                return $this->error('Invalid file type. Please upload an image (jpg, jpeg, webp, png)', 422);
            }

            // Create upload directory if it doesn't exist
            $uploadPath = public_path('quotes/payment_proofs');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }

            // Generate unique filename
            $filename = 'payment_proof_' . time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $relativePath = 'quotes/payment_proofs/' . $filename;

            // Get amount and payment type from request
            $amount = round((float) $request->input('amount'), 2);
            $paymentType = $request->input('payment_type'); // 'deposit' or 'remaining'

            // Validate amount is positive
            if ($amount <= 0) {
                return $this->error('Amount must be greater than 0', 422);
            }

            // Validate amount against quote's current amounts
            if ($paymentType === 'remaining') {
                // Check if quote has remaining amount
                $currentRemainingAmount = $quote->remaining_amount ?? 0;
                
                if ($currentRemainingAmount <= 0) {
                    return $this->error('This quote has no remaining amount to pay.', 422);
                }

                // Check if remaining payment is already completed
                if ($quote->remaining_payment_status === 'succeeded') {
                    return $this->error('Remaining payment has already been completed for this quote.', 422);
                }

                // Calculate total pending amount for remaining payments
                $pendingRemainingProofs = PaymentProof::where('quote_id', $quote->id)
                    ->where('payment_type', 'remaining')
                    ->where('status', 'pending')
                    ->sum('amount');
                
                $totalPendingAmount = $pendingRemainingProofs + $amount;
                
                // Validate that total pending amount doesn't exceed current remaining amount
                if ($totalPendingAmount > $currentRemainingAmount) {
                    $availableAmount = $currentRemainingAmount - $pendingRemainingProofs;
                    return $this->error(
                        'The payment amount ($' . number_format($amount, 2) . ') exceeds the available remaining amount. ' .
                        'Current remaining amount: $' . number_format($currentRemainingAmount, 2) . '. ' .
                        'Pending payment proofs: $' . number_format($pendingRemainingProofs, 2) . '. ' .
                        'Available amount: $' . number_format($availableAmount, 2) . '.',
                        422
                    );
                }
            } elseif ($paymentType === 'deposit') {
                // Check if quote has deposit amount
                $currentDepositAmount = $quote->deposit_amount ?? 0;
                $totalCost = $quote->total_cost ?? 0;
                
                // If deposit_amount is not set but total_cost exists, calculate expected deposit
                if ($currentDepositAmount <= 0 && $totalCost > 0) {
                    // Check if payment method is bank transfer (deposits are relevant)
                    $paymentMethod = $quote->payment_method ?? null;
                    if ($paymentMethod === 'bank_transfer') {
                        // For bank transfer, deposit amount might not be set yet, so we check against total cost
                        $currentDepositAmount = $totalCost;
                    }
                }

                // Check if deposit payment is already completed
                if ($quote->deposit_payment_status === 'succeeded') {
                    return $this->error('Deposit payment has already been completed for this quote.', 422);
                }

                // Calculate total pending amount for deposit payments
                $pendingDepositProofs = PaymentProof::where('quote_id', $quote->id)
                    ->where('payment_type', 'deposit')
                    ->where('status', 'pending')
                    ->sum('amount');
                
                $totalPendingAmount = $pendingDepositProofs + $amount;
                
                // Validate that total pending amount doesn't exceed deposit amount
                if ($currentDepositAmount > 0 && $totalPendingAmount > $currentDepositAmount) {
                    $availableAmount = $currentDepositAmount - $pendingDepositProofs;
                    return $this->error(
                        'The payment amount ($' . number_format($amount, 2) . ') exceeds the available deposit amount. ' .
                        'Current deposit amount: $' . number_format($currentDepositAmount, 2) . '. ' .
                        'Pending payment proofs: $' . number_format($pendingDepositProofs, 2) . '. ' .
                        'Available amount: $' . number_format($availableAmount, 2) . '.',
                        422
                    );
                }
            }

            // Update quote amounts if payment type is deposit
            $quoteUpdateData = [];
            $rawData = $quote->raw_data ?? [];
            
            if ($paymentType === 'deposit') {
                // Get total cost
                $totalCost = (float) ($quote->total_cost ?? 0);
                
                // Validate that deposit amount doesn't exceed total cost
                if ($amount > $totalCost) {
                    return $this->error(
                        'Deposit amount ($' . number_format($amount, 2) . ') cannot exceed the total cost ($' . number_format($totalCost, 2) . ').',
                        422
                    );
                }
                
                // Calculate remaining amount
                $remainingAmount = round($totalCost - $amount, 2);
                
                // Update quote with deposit amount
                $quoteUpdateData['deposit_amount'] = $amount;
                $quoteUpdateData['remaining_amount'] = $remainingAmount;
                $quoteUpdateData['deposit_payment_status'] = 'pending';
                
                // Update payment method if not set
                if (!$quote->payment_method) {
                    $quoteUpdateData['payment_method'] = 'bank_transfer';
                }
                
                // Update raw_data to set original_remaining_amount same as new remaining_amount
                $rawData['original_remaining_amount'] = $remainingAmount;
                $quoteUpdateData['raw_data'] = $rawData;
                
                // Convert to order if not already
                if ($quote->quote_type !== 'order') {
                    $quoteUpdateData['quote_type'] = 'order';
                    if (!$quote->order_number) {
                        $quoteUpdateData['order_number'] = Quote::generateOrderNumber();
                    }
                }
                
                // Update the quote
                $quote->update($quoteUpdateData);
                $quote->refresh();
                
                Log::info('Deposit payment proof uploaded - Quote amounts updated', [
                    'quote_id' => $quote->id,
                    'deposit_amount' => $amount,
                    'remaining_amount' => $remainingAmount,
                    'total_cost' => $totalCost
                ]);
            }

            // Check for existing pending payment proof with same payment_type and amount
            // If found, update it instead of creating a new entry
            $existingProof = PaymentProof::where('quote_id', $quote->id)
                ->where('payment_type', $paymentType)
                ->where('amount', $amount)
                ->where('status', 'pending')
                ->orderBy('created_at', 'desc')
                ->first();

            if ($existingProof) {
                // Update existing payment proof instead of creating new one
                $existingProof->update([
                    'proof_image' => $relativePath, // Update with new proof image
                    'updated_at' => now(),
                ]);
                $paymentProof = $existingProof->refresh();
                
                Log::info('Updated existing pending payment proof instead of creating duplicate', [
                    'quote_id' => $quote->id,
                    'payment_proof_id' => $paymentProof->id,
                    'payment_type' => $paymentType,
                    'amount' => $amount,
                    'status' => $paymentProof->status
                ]);
            } else {
                // Create new payment proof record (no duplicate found)
                $paymentProof = PaymentProof::create([
                    'quote_id' => $quote->id,
                    'payment_type' => $paymentType,
                    'proof_image' => $relativePath,
                    'amount' => $amount,
                    'status' => 'pending', // Will be reviewed by admin
                ]);
                
                Log::info('Created new payment proof', [
                    'quote_id' => $quote->id,
                    'payment_proof_id' => $paymentProof->id,
                    'payment_type' => $paymentType,
                    'amount' => $amount
                ]);
            }

            // Send email notification to admin
            try {
                $settings = $this->getGeneralSettingsData();
                $adminEmail = $settings['business_email'] ?? null;
                
                if (!empty($adminEmail)) {
                    $paymentTypeLabel = ucfirst($paymentType); // 'Deposit' or 'Remaining'
                    $subject = 'New ' . $paymentTypeLabel . ' Payment Proof Submitted | ' . $settings['business_name'];
                    
                    $adminParams = [
                        'subject' => $subject,
                        'to' => $adminEmail,
                        'msg' => view('Admin.Emails.payment-proof-notification-admin', [
                            'quote' => $quote,
                            'paymentProof' => $paymentProof,
                            'paymentType' => $paymentType,
                            'paymentTypeLabel' => $paymentTypeLabel,
                            'amount' => $amount,
                            'settings' => $settings,
                            'user' => $quote->user,
                        ])->render(),
                    ];

                    $emailSent = $this->SendInstantEmail($adminParams);
                    if ($emailSent) {
                        Log::info('Payment proof notification email sent to admin', [
                            'quote_id' => $quote->id,
                            'payment_proof_id' => $paymentProof->id,
                            'payment_type' => $paymentType,
                            'admin_email' => $adminEmail
                        ]);
                    } else {
                        Log::warning('Failed to send payment proof notification email to admin', [
                            'quote_id' => $quote->id,
                            'payment_proof_id' => $paymentProof->id,
                            'admin_email' => $adminEmail
                        ]);
                    }
                } else {
                    Log::warning('Admin email not configured in general settings (business_email). Skipping payment proof notification.');
                }
            } catch (\Exception $e) {
                Log::error('Error sending payment proof notification email to admin: ' . $e->getMessage(), [
                    'quote_id' => $quote->id,
                    'payment_proof_id' => $paymentProof->id,
                    'error' => $e->getTraceAsString()
                ]);
                // Don't fail the request if email fails
            }

            return $this->success($paymentProof, 'Payment proof uploaded successfully. It will be reviewed by admin.');
    }
}
