<?php

namespace App\Http\Controllers\API\Traits;

use App\Models\StorageItem;

trait QuoteCalculationTrait
{
    /**
     * Calculate complete quote for a booking
     */
    public function calculateQuote($booking, $distanceKm = null, $additionalTimeMinutes = 0, $request = null)
    {
        if (!$booking) {
            return null;
        }
        $truck = $booking->truck;
        if (!$truck) {
            return null;
        }
        $truckPricing = $booking->truck_pricing;
        if (!$truckPricing) {
            return null;
        }

        // Extract pricing data
        $pricingData = $this->extractPricingData($truckPricing);
        
        // Determine calculation type and get cubic meters
        $calculationType = $this->determineCalculationType($booking);
        $bookingData = $this->extractBookingData($booking, $request, $calculationType);
        
        // Get job type from multiple sources (priority: request > quote raw_data > quote job_type field)
        $jobType = null;
        
        // Priority 1: Check request
        if ($request && $request->has('job_type') && !empty($request->job_type)) {
            $jobType = $request->job_type;
        } else {
            // Priority 2: Check existing quote's raw_data
            $quote = \App\Models\Quote::where('uuid', $booking->uuid)->first();
            if ($quote) {
                // Check raw_data first
                if (isset($quote->raw_data['job_type']) && !empty($quote->raw_data['job_type'])) {
                    $jobType = $quote->raw_data['job_type'];
                }
                // Fallback to job_type field if raw_data doesn't have it
                elseif ($quote->job_type && !empty($quote->job_type)) {
                    $jobType = $quote->job_type;
                }
            }
        }
        
        $calculationResult = null;
        if ($jobType === 'local_job') {
            // Local job calculation (works for both items and trucks)
            // For trucks: uses required_truck_cubic_meter, ignores booking items
            // For items: calculates from booking items
            $calculationResult = $this->calculateLocalJobQuote(
                $bookingData,
                $pricingData,
                $request
            );
        } elseif ($jobType === 'interstate_job') {
            // Interstate job calculation (works for both items and trucks)
            // For trucks: uses required_truck_cubic_meter, ignores booking items
            // For items: calculates from booking items
            $calculationResult = $this->calculateInterstateJobQuote(
                $bookingData,
                $pricingData,
                $request
            );
        } 

        // Build and return response
        return $this->buildQuoteResponse(
            $booking,
            $truck,
            $calculationResult,
            $bookingData,
            $pricingData,
            $calculationType,
            $jobType
        );
    }

    /**
     * Calculate quote for local job
     * Works for both items-based bookings and truck-only bookings
     * For trucks: uses required_truck_cubic_meter from request
     * Calculation: (cubic_meters * 14 minutes) converted to hours * mover_per_hour_price
     * Callout fee: (base_to_pickup_time + dropof_to_base) * mover_per_hour_price
     */
    private function calculateLocalJobQuote($bookingData, $pricingData, $request)
    {
        if($request && isset($request->required_truck_cubic_meter) && $request->required_truck_cubic_meter > 0) {
            $bookingData['totalCubicMeters'] = $request->required_truck_cubic_meter;
        }
        $totalCubicMeters = $bookingData['totalCubicMeters'];
        
        // Get local job settings from general settings
        $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
        $numberOfMovers = (int)($generalSettings['_local_job_number_of_movers'] ?? 2);
        $moverPerHourPrice = (float)($generalSettings['_local_job_mover_per_hour_price'] ?? 85);
        $minimumOrder = (float)($generalSettings['_local_job_minimum_order'] ?? 0);
        
        // Ensure minimum 1 mover
        $numberOfMovers = max(1, $numberOfMovers);
        
        // Calculate time from cubic meters (14 minutes per cubic meter: 7 loading + 7 unloading)
        $timeData = $this->convertMinutesToHoursAndMinutes($totalCubicMeters * 14);
        
        // Handle extra movers if provided (1, 2, or 3 extra movers at $50 per hour each)
        $extraMovers = 0;
        if ($request && $request->has('extra_movers')) {
            $extraMovers = (int)$request->input('extra_movers');
            if ($extraMovers < 0 || $extraMovers > 3) {
                $extraMovers = 0;
            }
        }
        
        // Calculate base mover cost: time in hours * (number of movers * price per mover)
        $baseMoverPerHourPriceTotal = $numberOfMovers * $moverPerHourPrice;
        
        // The effective rate is the base rate (minimum is already set by number of movers * price)
        $effectiveMoverPerHourRate = $baseMoverPerHourPriceTotal;
        
        // Include extra movers in the mover rate ($50 per hour per extra mover)
        $moverPerHourPriceTotal = $effectiveMoverPerHourRate + ($extraMovers * 50);
        
        // Calculate movers cost: time in hours * total mover rate (includes base + extra movers)
        $moversCost = $timeData['decimalHours'] * $moverPerHourPriceTotal;
        
        // Extra movers cost is already included in moversCost, but keep for breakdown display
        $extraMoversCost = $timeData['decimalHours'] * $extraMovers * 50;
        
        // Calculate callout fee (base_to_pickup_time + dropof_to_base)
        $calloutFeeData = $this->calculateCalloutFee(
            $request,
            $moverPerHourPriceTotal,
            false // isInterstate = false
        );
        
        // Calculate total cost (exact amount, no minimum order applied - frontend handles minimum)
        $totalCost = $moversCost + $calloutFeeData['fee'];
        // Add assemble/disassemble and storage items costs (only for items-based bookings, not trucks)
        $totalCost += $bookingData['assembleDisassembleCost'];
        $totalCost += $bookingData['storageItemsCost'];
        
        return [
            'moversCost' => $moversCost,
            'calloutFee' => $calloutFeeData['fee'],
            'totalCost' => $totalCost, // Exact calculated total (no minimum order applied)
            'minimumOrder' => $minimumOrder, // Store for frontend reference only
            'timeData' => $timeData,
            'calloutFeeData' => $calloutFeeData,
            'extraMovers' => $extraMovers,
            'extraMoversCost' => $extraMoversCost,
        ];
    }

    /**
     * Calculate quote for interstate job
     * Works for both items-based bookings and truck-only bookings
     * For trucks: uses required_truck_cubic_meter from request
     * Calculation: (cubic_meters * 14 minutes) converted to hours * mover_per_hour_price
     * Callout fee: (base_to_pickup_time + dropof_to_base + pickup_to_dropof) * mover_per_hour_price
     */
    private function calculateInterstateJobQuote($bookingData, $pricingData, $request)
    {
        if($request && isset($request->required_truck_cubic_meter) && $request->required_truck_cubic_meter > 0) {
            $bookingData['totalCubicMeters'] = $request->required_truck_cubic_meter;
        }
        $totalCubicMeters = $bookingData['totalCubicMeters'];
        
        // Get interstate job settings from general settings
        $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
        $numberOfMovers = (int)($generalSettings['_interstate_job_number_of_movers'] ?? 2);
        $moverPerHourPrice = (float)($generalSettings['_interstate_job_mover_per_hour_price'] ?? 110);
        $minimumOrder = (float)($generalSettings['_interstate_job_minimum_order'] ?? 0);
        
        // Ensure minimum 1 mover
        $numberOfMovers = max(1, $numberOfMovers);
        
        // Calculate time from cubic meters (14 minutes per cubic meter: 7 loading + 7 unloading)
        $timeData = $this->convertMinutesToHoursAndMinutes($totalCubicMeters * 14);
        
        // Handle extra movers if provided (1, 2, or 3 extra movers at $50 per hour each)
        $extraMovers = 0;
        if ($request && $request->has('extra_movers')) {
            $extraMovers = (int)$request->input('extra_movers');
            if ($extraMovers < 0 || $extraMovers > 3) {
                $extraMovers = 0;
            }
        }
        
        // Calculate base mover cost: time in hours * (number of movers * price per mover)
        $baseMoverPerHourPriceTotal = $numberOfMovers * $moverPerHourPrice;
        
        // The effective rate is the base rate (minimum is already set by number of movers * price)
        $effectiveMoverPerHourRate = $baseMoverPerHourPriceTotal;
        
        // Include extra movers in the mover rate ($50 per hour per extra mover)
        $moverPerHourPriceTotal = $effectiveMoverPerHourRate + ($extraMovers * 50);
        
        // Calculate movers cost: time in hours * total mover rate (includes base + extra movers)
        $moversCost = $timeData['decimalHours'] * $moverPerHourPriceTotal;
        
        // Extra movers cost is already included in moversCost, but keep for breakdown display
        $extraMoversCost = $timeData['decimalHours'] * $extraMovers * 50;
        
        // Calculate callout fee (includes pickup_to_dropof for interstate)
        $calloutFeeData = $this->calculateCalloutFee(
            $request,
            $moverPerHourPriceTotal,
            true // isInterstate = true
        );
        
        // Calculate total cost (exact amount, no minimum order applied - frontend handles minimum)
        $totalCost = $moversCost + $calloutFeeData['fee'];
        // Add assemble/disassemble and storage items costs (only for items-based bookings, not trucks)
        $totalCost += $bookingData['assembleDisassembleCost'];
        $totalCost += $bookingData['storageItemsCost'];
        
        return [
            'moversCost' => $moversCost,
            'calloutFee' => $calloutFeeData['fee'],
            'totalCost' => $totalCost, // Exact calculated total (no minimum order applied)
            'minimumOrder' => $minimumOrder, // Store for frontend reference only
            'timeData' => $timeData,
            'calloutFeeData' => $calloutFeeData,
            'extraMovers' => $extraMovers,
            'extraMoversCost' => $extraMoversCost,
        ];
    }

 
    /**
     * Extract pricing data from truck pricing
     */
    private function extractPricingData($truckPricing)
    {
        $numberOfMovers = $truckPricing['number_of_movers'] ?? 0;
        $moverPerHourPrice = $truckPricing['price_per_mover'] ?? 0;
        
        return [
            'perMeterCubePrice' => $truckPricing['pricing']['per_cubic_meter_price'] ?? 0,
            'numberOfMovers' => $numberOfMovers,
            'moverPerHourPrice' => $moverPerHourPrice,
            'moverPerHourPriceTotal' => $numberOfMovers * $moverPerHourPrice,
            'truckCubicCapacity' => $truckPricing['cubic_capacity'] ?? 0,
        ];
    }

    /**
     * Extract booking data (cubic meters, costs, etc.)
     * For items-based bookings: calculates from booking items
     * For truck-only bookings: gets cubic meters from required_truck_cubic_meter in request
     * Priority: If request has required_truck_cubic_meter, it's ALWAYS truck-only (ignores booking items)
     * Truck-only bookings work with both local_job and interstate_job
     */
    private function extractBookingData($booking, $request, $calculationType)
    {
        $totalCubicMeters = 0;
        $assembleDisassembleCost = 0;
        $storageItemsCost = 0;
        $totalPackageTimeMinutes = 0;
        $isTruckOnly = false;
        
        // PRIORITY: Check if request has required_truck_cubic_meter
        // If present, it's ALWAYS a truck-only booking, regardless of booking items in database
        if ($request && 
            isset($request->required_truck_cubic_meter) && 
            $request->required_truck_cubic_meter > 0) {
            
            // Truck-only booking: use required_truck_cubic_meter from request
            // IGNORE any booking items in the database - this is truck-only
            // This will be used in the same calculation as items (multiply by 14 for time)
            $totalCubicMeters = (float)$request->required_truck_cubic_meter;
            $isTruckOnly = true;
            
            // Trucks don't have assemble/disassemble or storage items
            // Explicitly set to 0 to ensure no costs from booking items
            $assembleDisassembleCost = 0;
            $storageItemsCost = 0;
            $totalPackageTimeMinutes = 0;
        } else {
            // Items-based booking: calculate from booking items
            // Only use this path if required_truck_cubic_meter is NOT provided
            $totalCubicMeters = $this->calculateTotalCubicMeters($booking);
            $totalPackageTimeMinutes = $this->calculateTotalPackageTimeMinutes($booking);
            $assembleDisassembleCost = $this->calculateAssembleDisassembleCost($booking);
            $storageItemsCost = $this->calculateStorageItemsCost($booking);
        }
        
        return [
            'totalCubicMeters' => $totalCubicMeters,
            'assembleDisassembleCost' => $assembleDisassembleCost,
            'storageItemsCost' => $storageItemsCost,
            'totalPackageTimeMinutes' => $totalPackageTimeMinutes,
            'isTruckOnly' => $isTruckOnly,
        ];
    }

    /**
     * Convert minutes to hours and remaining minutes
     */
    private function convertMinutesToHoursAndMinutes($minutes)
    {
        $hours = floor($minutes / 60);
        $remainingMinutes = round($minutes - ($hours * 60));
        $decimalHours = $hours + ($remainingMinutes / 60);
        
        return [
            'minutes' => $minutes,
            'hours' => $hours,
            'remainingMinutes' => $remainingMinutes,
            'decimalHours' => $decimalHours,
        ];
    }

    /**
     * Calculate callout fee
     */
    private function calculateCalloutFee($request, $moverPerHourPriceTotal, $isInterstate = false)
    {
        $calloutFee = 0;
        $calloutFeeTime = 0;
        $calloutFeeHours = 0;
        $calloutFeeRemainingMinutes = 0;
        $calloutFeeInHours = 0;
        
        if (!$request) {
            return [
                'fee' => $calloutFee,
                'time' => $calloutFeeTime,
                'hours' => $calloutFeeHours,
                'remainingMinutes' => $calloutFeeRemainingMinutes,
                'decimalHours' => $calloutFeeInHours,
            ];
        }
        
        // Calculate total callout time (works for both items and trucks)
        if ($isInterstate) {
            // Interstate: base_to_pickup_time + dropof_to_base + pickup_to_dropof
            // All times are in minutes
            if (isset($request->base_to_pickup_time) && 
                isset($request->dropof_to_base) && 
                isset($request->pickup_to_dropof)) {
                $calloutFeeTime = $request->base_to_pickup_time + 
                                 $request->dropof_to_base + 
                                 $request->pickup_to_dropof;
            }
        } else {
            // Local: base_to_pickup_time + dropof_to_base
            // All times are in minutes
            if (isset($request->base_to_pickup_time) && isset($request->dropof_to_base)) {
                $calloutFeeTime = $request->base_to_pickup_time + $request->dropof_to_base;
            }
        }
        
        if ($calloutFeeTime > 0) {
            // Save original time before applying multiplier
            $calloutFeeTimeOriginal = $calloutFeeTime;
            
            // Get buffer multiplier percentage from general settings
            $generalSettings = \App\General\GeneralSettingsClass::getAllSettings();
            $bufferPercentage = (float)($generalSettings['_buffer_multiplier_time_percentage'] ?? 10); // Default to 10% if not set
            
            // Convert percentage to multiplier (e.g., 20% = 1.20, 15% = 1.15, 10% = 1.10)
            $multiplier = 1 + ($bufferPercentage / 100);
            
            // Apply buffer multiplier to the total callout time (for both local and interstate)
            $calloutFeeTime = $calloutFeeTime * $multiplier;
            
            // Convert to hours and minutes
            $timeData = $this->convertMinutesToHoursAndMinutes($calloutFeeTime);
            $calloutFeeHours = $timeData['hours'];
            $calloutFeeRemainingMinutes = $timeData['remainingMinutes'];
            $calloutFeeInHours = $timeData['decimalHours'];
            
            // Calculate fee
            $calloutFee = $calloutFeeInHours * $moverPerHourPriceTotal;
            
            return [
                'fee' => $calloutFee,
                'time' => $calloutFeeTime,
                'hours' => $calloutFeeHours,
                'remainingMinutes' => $calloutFeeRemainingMinutes,
                'decimalHours' => $calloutFeeInHours,
                'timeOriginal' => $calloutFeeTimeOriginal,
                'multiplier' => $multiplier,
            ];
        }
        
        return [
            'fee' => $calloutFee,
            'time' => $calloutFeeTime,
            'hours' => $calloutFeeHours,
            'remainingMinutes' => $calloutFeeRemainingMinutes,
            'decimalHours' => $calloutFeeInHours,
        ];
    }

    /**
     * Build quote response data
     */
    private function buildQuoteResponse($booking, $truck, $calculationResult, $bookingData, $pricingData, $calculationType, $jobType)
    {
        // Handle case when calculationResult is null (no job_type provided or invalid)
        if (!$calculationResult) {
            // Return a default response with zero costs
            return [
                'booking_id' => $booking->id,
                'truck_id' => $truck->id,
                'distance_km' => 0,
                'total_time_minutes' => 0,
                'total_cubic_meters' => $bookingData['totalCubicMeters'],
                'total_package_time_minutes' => $bookingData['totalPackageTimeMinutes'],
                'truck_cubic_capacity' => $pricingData['truckCubicCapacity'],
                'per_meter_cube_price' => $pricingData['perMeterCubePrice'],
                'number_of_movers' => max(2, $pricingData['numberOfMovers']),
                'mover_per_hour_price' => $pricingData['moverPerHourPrice'],
                'extra_movers' => 0,
                'extra_movers_cost' => 0,
                'volume_cost' => 0,
                'assemble_disassemble_cost' => round($bookingData['assembleDisassembleCost'], 2),
                'storage_items_cost' => round($bookingData['storageItemsCost'], 2),
                'movers_cost' => 0,
                'total_cost' => round($bookingData['assembleDisassembleCost'] + $bookingData['storageItemsCost'], 2),
                'calculation_type' => $calculationType,
                'job_type' => $jobType,
                'raw_data' => [
                    'base_time_minutes' => 0,
                    'additional_time_minutes' => 0,
                    'package_time_minutes' => $bookingData['totalPackageTimeMinutes'],
                    'breakdown' => [],
                    'job_type' => $jobType,
                ],
            ];
        }
        
        $moversCost = $calculationResult['moversCost'];
        $calloutFee = $calculationResult['calloutFee'];
        $totalCost = $calculationResult['totalCost'];
        $timeData = $calculationResult['timeData'];
        $calloutFeeData = $calculationResult['calloutFeeData'];
        $extraMovers = $calculationResult['extraMovers'] ?? 0;
        $extraMoversCost = $calculationResult['extraMoversCost'] ?? 0;
        
        // Get minimum order info from calculation result (for frontend reference only)
        $minimumOrder = $calculationResult['minimumOrder'] ?? 0;
        
        // Build breakdown
        $breakdown = [
            'movers_cost' => round($moversCost, 2),
        ];
        
        // Add extra movers cost to breakdown if applicable
        if ($extraMovers > 0) {
            $breakdown['extra_movers'] = $extraMovers;
            $breakdown['extra_movers_cost'] = round($extraMoversCost, 2);
        }
        
        if (($jobType === 'local_job' || $jobType === 'interstate_job') && $calloutFee > 0) {
            $breakdown['callout_fee'] = round($calloutFee, 2);
        }
        
        if ($bookingData['storageItemsCost'] > 0) {
            $breakdown['storage_items_cost'] = round($bookingData['storageItemsCost'], 2);
        }
        
        // Only add assemble/disassemble cost for items-based bookings, not truck-only
        if (!$bookingData['isTruckOnly'] && $bookingData['assembleDisassembleCost'] > 0) {
            $breakdown['assemble_disassemble_cost'] = round($bookingData['assembleDisassembleCost'], 2);
        }
        
        // Add minimum order to breakdown for frontend reference (not applied to total)
        if ($minimumOrder > 0) {
            $breakdown['minimum_order'] = round($minimumOrder, 2);
        }
        
        // Build raw data
        $rawData = [
            'base_time_minutes' => 0,
            'additional_time_minutes' => 0,
            'package_time_minutes' => $bookingData['totalPackageTimeMinutes'],
            'breakdown' => $breakdown,
            'job_type' => $jobType, // Save job_type in raw_data
        ];
        
        if ($timeData) {
            $rawData['time_from_cubic_meters_minutes'] = $timeData['minutes'];
            $rawData['time_from_cubic_meters_hours'] = $timeData['hours'];
            $rawData['time_from_cubic_meters_remaining_minutes'] = $timeData['remainingMinutes'];
            $rawData['time_from_cubic_meters_in_hours'] = round($timeData['decimalHours'], 2);
        }
        
        if ($calloutFeeData) {
            $rawData['callout_fee_time'] = $calloutFeeData['time'];
            $rawData['callout_fee_hours'] = $calloutFeeData['hours'];
            $rawData['callout_fee_remaining_minutes'] = $calloutFeeData['remainingMinutes'];
            $rawData['callout_fee_in_hours'] = round($calloutFeeData['decimalHours'], 2);
            $rawData['callout_fee'] = round($calloutFee, 2);
            // Add original time and multiplier for transparency
            if (isset($calloutFeeData['timeOriginal'])) {
                $rawData['callout_fee_time_original'] = $calloutFeeData['timeOriginal'];
            }
            if (isset($calloutFeeData['multiplier'])) {
                $rawData['callout_fee_multiplier'] = $calloutFeeData['multiplier'];
            }
        }
        
        // Add extra movers information to raw data (always include, even if 0)
        $rawData['extra_movers'] = $extraMovers;
        $rawData['extra_movers_cost'] = round($extraMoversCost, 2);
        
        // Build main response
        $response = [
            'booking_id' => $booking->id,
            'truck_id' => $truck->id,
            'distance_km' => 0,
            'total_time_minutes' => $timeData ? round($timeData['minutes'], 0) : 0,
            'total_cubic_meters' => $bookingData['totalCubicMeters'],
            'total_package_time_minutes' => $bookingData['totalPackageTimeMinutes'],
            'truck_cubic_capacity' => $pricingData['truckCubicCapacity'],
            'per_meter_cube_price' => $pricingData['perMeterCubePrice'],
            'number_of_movers' => max(2, $pricingData['numberOfMovers']), // Minimum 2 movers
            'mover_per_hour_price' => $pricingData['moverPerHourPrice'],
            'extra_movers' => $extraMovers,
            'extra_movers_cost' => round($extraMoversCost, 2),
            'volume_cost' => 0,
            'assemble_disassemble_cost' => round($bookingData['assembleDisassembleCost'], 2),
            'storage_items_cost' => round($bookingData['storageItemsCost'], 2),
            'movers_cost' => round($moversCost, 2),
            'total_cost' => round($totalCost, 2), // Exact calculated total (no minimum order applied)
            'minimum_order' => $minimumOrder > 0 ? round($minimumOrder, 2) : 0, // For frontend reference only
            'calculation_type' => $calculationType,
            'job_type' => $jobType,
            'raw_data' => $rawData,

        ];
        
        // Add job-specific data for local_job and interstate_job
        if ($jobType === 'local_job' || $jobType === 'interstate_job') {
            $response['callout_fee'] = round($calloutFee, 2);
            if ($timeData) {
                $response['time_from_cubic_meters_minutes'] = round($timeData['minutes'], 2);
                $response['time_from_cubic_meters_hours'] = $timeData['hours'];
                $response['time_from_cubic_meters_remaining_minutes'] = $timeData['remainingMinutes'];
                $response['time_from_cubic_meters_in_hours'] = round($timeData['decimalHours'], 2);
            }
            if ($calloutFeeData) {
                $response['callout_fee_hours'] = $calloutFeeData['hours'];
                $response['callout_fee_remaining_minutes'] = $calloutFeeData['remainingMinutes'];
                $response['callout_fee_in_hours'] = round($calloutFeeData['decimalHours'], 2);
            }
        }
        
        return $response;
    }

    /**
     * Calculate total cubic meters from regular items (non-assemble, non-storage items)
     * Excludes: assemble/disassemble items and storage items
     */
    private function calculateTotalCubicMeters($booking)
    {
        $bookingItems = $booking->bookingItems()->with('entity')->get();
        $totalCubicMeters = 0;
        foreach ($bookingItems as $item) {
            // Exclude assemble/disassemble items
            if ($item->is_assemble_disassemble) {
                continue;
            }
            // Exclude storage items (they have their own pricing, not volume-based)
            if ($item->entity_type === StorageItem::class) {
                continue;
            }
            // Only count regular items (ChildCategory items)
            // Multiply quantity by volume per item: if item has volume 2 and quantity is 2, total volume = 4
            $totalCubicMeters += $item->quantity * ($item->getVolume());
        }
        return $totalCubicMeters;
    }

    /**
     * Calculate total package time from assemble/disassemble items
     * Only processes items with entity_type = App\Models\AssembleDisassembleCategory
     */
    private function calculateTotalPackageTimeMinutes($booking)
    {
        $bookingItems = $booking->bookingItems()->with('entity')->get();
        $totalMinutes = 0;
        foreach ($bookingItems as $item) {
            // Check: must be assemble/disassemble AND entity_type must be AssembleDisassembleCategory
            if (
                $item->is_assemble_disassemble
                && $item->entity_type === 'App\Models\AssembleDisassembleCategory'
                && $item->entity
            ) {
                $type = $item->assemble_disassemble_type;
                $entity = $item->entity;
                $timeMinutes = 0;
                // Get time based on the assemble_disassemble_type
                switch ($type) {
                    case 'assemble':
                        $timeMinutes = $entity->assemble_time_minutes ?? 0;
                        break;
                    case 'disassemble':
                        $timeMinutes = $entity->disassemble_time_minutes ?? 0;
                        break;
                    case 'both':
                        $timeMinutes = $entity->both_time_minutes ?? 0;
                        break;
                    default:
                        $timeMinutes = 0;
                        break;
                }
                // Multiply quantity by time: if item takes 10 minutes and quantity is 2, total time = 20 minutes
                $totalMinutes += $item->quantity * $timeMinutes;
            }
        }
        return $totalMinutes;
    }

    /**
     * Calculate total cost from assemble/disassemble items
     * Uses the correct price based on assemble_disassemble_type (assemble, disassemble, or both)
     * Only processes items with entity_type = App\Models\AssembleDisassembleCategory
     */
    private function calculateAssembleDisassembleCost($booking)
    {
        $bookingItems = $booking->bookingItems()->with('entity')->get();
        $totalCost = 0;

        foreach ($bookingItems as $item) {
            // Check: must be assemble/disassemble AND entity_type must be AssembleDisassembleCategory
            if (
                $item->is_assemble_disassemble
                && $item->entity_type === 'App\Models\AssembleDisassembleCategory'
                && $item->entity
            ) {
                $type = $item->assemble_disassemble_type;
                $entity = $item->entity;
                $price = 0;
                // Get price based on the assemble_disassemble_type
                switch ($type) {
                    case 'assemble':
                        $price = $entity->assemble_price ?? 0;
                        break;
                    case 'disassemble':
                        $price = $entity->disassemble_price ?? 0;
                        break;
                    case 'both':
                        $price = $entity->both_price ?? 0;
                        break;
                    default:
                        $price = 0;
                        break;
                }
                // Multiply quantity by price: if item costs $10 and quantity is 2, total cost = $20
                $totalCost += $item->quantity * $price;
            }
        }
        return $totalCost;
    }

    private function calculateStorageItemsCost($booking): float
    {
        $bookingItems = $booking->bookingItems()->with('entity')->get();
        $totalCost = 0.0;
        foreach ($bookingItems as $item) {
            if (
                !$item->is_assemble_disassemble
                && $item->entity_type === StorageItem::class
                && $item->entity
            ) {
                $price = $item->entity->price ?? 0;
                // Multiply quantity by price: if item costs $10 and quantity is 2, total cost = $20
                $totalCost += $item->quantity * (float) $price;
            }
        }
        return $totalCost;
    }

   

    /**
     * Determine the calculation type based on booking_item_ids column
     * If booking_item_ids is null or empty = truck_only
     * Otherwise check booking items for items_only or items_and_packages
     */
    private function determineCalculationType($booking)
    {
        // Always check actual booking items first (most reliable source)
        // This ensures we get the current state even if booking_item_ids is stale
        $bookingItems = $booking->bookingItems()->get();
        
        if ($bookingItems->isEmpty()) {
            // No booking items found - check booking_item_ids as fallback
            $bookingItemIds = $booking->booking_item_ids;
            if (empty($bookingItemIds) || (is_array($bookingItemIds) && count($bookingItemIds) === 0)) {
                return 'truck_only';
            }
            // booking_item_ids has values but no actual items - still truck_only
            return 'truck_only';
        }
        
        // We have booking items - determine type based on item types
        $hasRegularItems = $bookingItems->where('is_assemble_disassemble', false)->isNotEmpty();
        $hasPackageItems = $bookingItems->where('is_assemble_disassemble', true)->isNotEmpty();
        
        // Case 2: Both regular items and assemble/disassemble packages
        if ($hasRegularItems && $hasPackageItems) {
            return 'items_and_packages';
        }
        // Case 1: Only regular items
        if ($hasRegularItems) {
            return 'items_only';
        }
        // Case 2: Only assemble/disassemble packages
        if ($hasPackageItems) {
            return 'items_and_packages';
        }
        // Fallback: truck rental if somehow we get here
        return 'truck_only';
    }
}
