<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Controllers\API\Traits\JsonResponseTrait;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    use JsonResponseTrait;

    /**
     * Create or update user for booking
     * This handles user creation/update logic that was previously in BookingController
     */
    public function createOrUpdate(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'user_id' => 'nullable|integer|exists:users,id', // Optional: if updating existing user
        ]);

        $name = $request->input('name');
        $email = $request->input('email');
        $phone = $request->input('phone');
        $existingUserId = $request->input('user_id');

        // Priority 1: If user_id is provided, update that user
        if ($existingUserId) {
            $user = User::find($existingUserId);
            
            if (!$user) {
                return $this->error('User not found', 404);
            }

            $needsUpdate = false;
            $userUpdateData = [];

            // Check if email is being changed
            if ($email && $user->email !== $email) {
                // Check if new email already exists for another user
                $emailExists = User::where('email', $email)
                    ->where('id', '!=', $user->id)
                    ->exists();

                if ($emailExists) {
                    return $this->error('Email already exists for another user', 422);
                }

                $userUpdateData['email'] = $email;
                $needsUpdate = true;
            }

            // Check if name is being changed
            if ($name && $user->name !== $name) {
                $userUpdateData['name'] = $name;
                $needsUpdate = true;
            }

            // Check if phone is being changed
            if ($phone && $user->phone !== $phone) {
                $userUpdateData['phone'] = $phone;
                $needsUpdate = true;
            }

            // Update user if email/name/phone changed
            if ($needsUpdate && !empty($userUpdateData)) {
                try {
                    $user->update($userUpdateData);
                } catch (\Illuminate\Database\QueryException $e) {
                    // Handle unique constraint violation for email
                    if ($e->getCode() == 23000) {
                        return $this->error('Email already exists. Please use a different email address.', 422);
                    }
                    throw $e;
                }
            }

            return $this->success($user->fresh(), 'User updated successfully');
        }

        // Priority 2: Check if user with this email already exists
        $existingUser = User::where('email', $email)->first();

        if ($existingUser) {
            // Update existing user's name and phone if they differ
            $updateData = [];
            if ($name && $existingUser->name !== $name) {
                $updateData['name'] = $name;
            }
            if ($phone && $existingUser->phone !== $phone) {
                $updateData['phone'] = $phone;
            }
            if (!empty($updateData)) {
                $existingUser->update($updateData);
            }
            return $this->success($existingUser->fresh(), 'User found and updated successfully');
        }

        // Create new user
        try {
            $randomPassword = $this->generateRandomPassword();
            $user = User::create([
                'name' => $name,
                'email' => $email,
                'phone' => $phone,
                'password' => Hash::make($randomPassword),
                'role_id' => 2,
                'profile_image' => '/user.jpg',
                'email_verified_at' => now(),
                'active' => 1,
            ]);

            return $this->success($user, 'User created successfully');
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle unique constraint violation for email
            if ($e->getCode() == 23000) {
                return $this->error('Email already exists. Please use a different email address.', 422);
            }
            throw $e;
        }
    }

    /**
     * Generate random password
     */
    private function generateRandomPassword($length = 12)
    {
        $uppercase = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $lowercase = 'abcdefghijklmnopqrstuvwxyz';
        $numbers = '0123456789';
        $special = '@#$%&*';

        $allChars = $uppercase . $lowercase . $numbers . $special;

        $password = '';
        $password .= $uppercase[rand(0, strlen($uppercase) - 1)];
        $password .= $lowercase[rand(0, strlen($lowercase) - 1)];
        $password .= $numbers[rand(0, strlen($numbers) - 1)];
        $password .= $special[rand(0, strlen($special) - 1)];

        for ($i = 4; $i < $length; $i++) {
            $password .= $allChars[rand(0, strlen($allChars) - 1)];
        }

        return str_shuffle($password);
    }
}

