<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AssembleDisassembleCategory;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class AssembleDisassembleCategoryController extends Controller
{
    public function index(Request $request)
    {
        $categories = AssembleDisassembleCategory::orderBy('sort_order', 'asc')->orderBy('name', 'asc')->get();
        return view('Admin.AssembleDisassembleCategory.index', compact('categories'));
    }

    public function create()
    {
        return view('Admin.AssembleDisassembleCategory.create');
    }

    public function store(Request $request)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'assemble_price' => 'nullable|numeric|min:0',
            'disassemble_price' => 'nullable|numeric|min:0',
            'both_price' => 'nullable|numeric|min:0',
            'assemble_time_minutes' => 'nullable|integer|min:1',
            'disassemble_time_minutes' => 'nullable|integer|min:1',
            'both_time_minutes' => 'nullable|integer|min:1',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ];
        $validated_data = $request->validate($rules);
        if (!isset($validated_data['sort_order']) || $validated_data['sort_order'] == 0) {
            $validated_data['sort_order'] = AssembleDisassembleCategory::getNextSortOrder();
        }
        Storage::makeDirectory('public/assemble_disassemble');
        $saveDir = public_path('assemble_disassemble');
        if (!File::isDirectory($saveDir)) {
            File::makeDirectory($saveDir, 0755, true);
        }
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_assemble_disassemble.' . $image->getClientOriginalExtension();
            $image->move($saveDir, $imageName);
            $validated_data['image'] = 'assemble_disassemble/' . $imageName;
        }
        if ($request->filled('category_id')) {
            $category = AssembleDisassembleCategory::findOrFail($request->category_id);
            if (!$request->hasFile('image')) {
                $validated_data['image'] = $category->image;
            }
            $category->update($validated_data);
            $message = 'Assembly setting updated successfully!';
        } else {
            $category = AssembleDisassembleCategory::create($validated_data);
            $message = 'Assembly setting created successfully!';
        }
        return redirect()->route('admin.assemble-disassemble-categories.index')->with('success', $message);
    }

    public function show(AssembleDisassembleCategory $assembleDisassembleCategory)
    {
        return view('Admin.AssembleDisassembleCategory.show', compact('assembleDisassembleCategory'));
    }

    public function edit(AssembleDisassembleCategory $assembleDisassembleCategory)
    {
        return view('Admin.AssembleDisassembleCategory.edit', compact('assembleDisassembleCategory'));
    }

    public function update(Request $request, AssembleDisassembleCategory $assembleDisassembleCategory)
    {
        $rules = [
            'name' => 'required|string|max:255',
          
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'assemble_price' => 'nullable|numeric|min:0',
            'disassemble_price' => 'nullable|numeric|min:0',
            'both_price' => 'nullable|numeric|min:0',
            'assemble_time_minutes' => 'nullable|integer|min:1',
            'disassemble_time_minutes' => 'nullable|integer|min:1',
            'both_time_minutes' => 'nullable|integer|min:1',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ];
        $validated_data = $request->validate($rules);
        Storage::makeDirectory('public/assemble_disassemble');
        $saveDir = public_path('assemble_disassemble');
        if (!File::isDirectory($saveDir)) {
            File::makeDirectory($saveDir, 0755, true);
        }
        if ($request->hasFile('image')) {
            if (!empty($assembleDisassembleCategory->image) && File::exists(public_path($assembleDisassembleCategory->image))) {
                File::delete(public_path($assembleDisassembleCategory->image));
            }
            $image = $request->file('image');
            $imageName = time() . '_assemble_disassemble.' . $image->getClientOriginalExtension();
            $image->move($saveDir, $imageName);
            $validated_data['image'] = 'assemble_disassemble/' . $imageName;
        }
        $assembleDisassembleCategory->update($validated_data);
        return redirect()->route('admin.assemble-disassemble-categories.index')->with('success', 'Assembly setting updated successfully.');
    }

    public function destroy(AssembleDisassembleCategory $assembleDisassembleCategory)
    {
        if (!empty($assembleDisassembleCategory->image) && File::exists(public_path($assembleDisassembleCategory->image))) {
            File::delete(public_path($assembleDisassembleCategory->image));
        }
        $assembleDisassembleCategory->delete();
        return redirect()->route('admin.assemble-disassemble-categories.index')->with('success', 'Assembly setting deleted successfully.');
    }

    public function getNextSortOrder()
    {
        $nextSortOrder = AssembleDisassembleCategory::getNextSortOrder();
        return response()->json(['next_sort_order' => $nextSortOrder]);
    }

    public function edit_category(Request $request)
    {
        $id = $request->id;
        $category = AssembleDisassembleCategory::find($id);
        return response()->json($category, 200, [], JSON_UNESCAPED_SLASHES);
    }

    public function show_category_detail(Request $request)
    {
        $category = AssembleDisassembleCategory::find($request->id);
        if (!$category) {
            return response()->json(['error' => 'Assembly setting not found'], 404);
        }
        return response()->json([
            'id' => $category->id,
            'name' => $category->name,
            'description' => $category->description,
            'image' => $category->image,
            'assemble_price' => $category->assemble_price ?? 0,
            'disassemble_price' => $category->disassemble_price ?? 0,
            'both_price' => $category->both_price ?? 0,
            'assemble_time_minutes' => $category->assemble_time_minutes ?? 0,
            'disassemble_time_minutes' => $category->disassemble_time_minutes ?? 0,
            'both_time_minutes' => $category->both_time_minutes ?? 0,
            'is_active' => $category->is_active,
            'sort_order' => $category->sort_order,
        ]);
    }

    public function update_category(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'is_active' => 'boolean',
        ]);
        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'message' => $validator->errors()], 422);
        }
        $category = AssembleDisassembleCategory::find($id);
        if (!$category) {
            return response()->json(['status' => 'error', 'message' => 'Assembly setting not found.'], 404);
        }        
        $category->update($validator->validated());
        return response()->json(['status' => 'success', 'message' => 'Assembly setting updated successfully.']);
    }
}
