<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BlogCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class BlogCategoryController extends Controller
{
    public function index()
    {
        $categories = BlogCategory::ordered()->get();
        return view('Admin.BlogCategory.index', compact('categories'));
    }

    public function create()
    {
        return view('Admin.BlogCategory.create');
    }

    public function store(Request $request)
    {
        $isUpdate = $request->filled('blog_category_id');
        $categoryId = $request->input('blog_category_id');
        
        $request->validate([
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                'max:255',
                Rule::unique('blog_categories', 'permalink')->ignore($categoryId)
            ],
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean',
        ]);
        
        $data = $request->except(['blog_category_id']);
        
        // Generate permalink from title if not provided
        if (empty($data['permalink']) && !empty($data['title'])) {
            $data['permalink'] = Str::slug($data['title']);
            
            // Ensure uniqueness
            $basePermalink = $data['permalink'];
            $counter = 1;
            while (BlogCategory::where('permalink', $data['permalink'])->where('id', '!=', $categoryId)->exists()) {
                $data['permalink'] = $basePermalink . '-' . $counter;
                $counter++;
            }
        }
        
        // Set is_active default
        $data['is_active'] = $request->has('is_active') ? true : false;
        
        // Save or update category
        if ($isUpdate) {
            $category = BlogCategory::findOrFail($categoryId);
            $category->update($data);
            $message = 'Blog category updated successfully.';
        } else {
            BlogCategory::create($data);
            $message = 'Blog category created successfully.';
        }
        
        return redirect()->route('admin.blog-categories.index')->with('success', $message);
    }

    public function show(BlogCategory $blogCategory)
    {
        return response()->json($blogCategory);
    }

    public function edit(BlogCategory $blogCategory)
    {
        return view('Admin.BlogCategory.edit', compact('blogCategory'));
    }


    public function destroy(BlogCategory $blogCategory)
    {
        $blogCategory->delete();
        return redirect()->route('admin.blog-categories.index')->with('success', 'Blog category deleted successfully.');
    }
    
    public function checkPermalink(Request $request)
    {
        $permalink = $request->input('permalink');
        $categoryId = $request->input('category_id');
        
        if (empty($permalink)) {
            return response()->json([
                'available' => false,
                'message' => 'Permalink is required.'
            ]);
        }
        
        $query = BlogCategory::where('permalink', $permalink);
        
        // Exclude current category if editing
        if ($categoryId) {
            $query->where('id', '!=', $categoryId);
        }
        
        $exists = $query->exists();
        
        return response()->json([
            'available' => !$exists,
            'message' => $exists ? 'Permalink already exists.' : 'Permalink is available.'
        ]);
    }
}