<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use App\Models\BlogCategory;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;

class BlogController extends Controller
{
    public function index(Request $request)
    {
        $query = Blog::query();
        if ($request->filled('status')) {
            $query->where('active', $request->status);
        }
        if ($request->filled('category')) {
            // Filter blogs by category ID in blog_categories_id JSON column
            $query->whereRaw('JSON_CONTAINS(blog_categories_id, ?)', [json_encode((int)$request->category)]);
        }
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('description', 'LIKE', "%{$searchTerm}%")
                    ->orWhere('content', 'LIKE', "%{$searchTerm}%");
            });
        }
        if ($request->filled('date_range')) {
            $dateRange = $request->date_range;
            $now = Carbon::now();
            switch ($dateRange) {
                case 'today':
                    $query->whereDate('created_at', $now->toDateString());
                    break;
                case 'week':
                    $query->whereBetween('created_at', [
                        $now->startOfWeek()->toDateTimeString(),
                        $now->endOfWeek()->toDateTimeString()
                    ]);
                    break;
                case 'month':
                    $query->whereMonth('created_at', $now->month)
                        ->whereYear('created_at', $now->year);
                    break;
                case 'year':
                    $query->whereYear('created_at', $now->year);
                    break;
            }
        }
        $blogs = $query->orderBy('created_at', 'desc')->get();
        $allCategories = BlogCategory::active()->ordered()->get();
        return view('Admin.Blog.index', compact('blogs', 'allCategories'));
    }

    public function create()
    {
        $blogCategories = BlogCategory::active()->ordered()->get();
        return view('Admin.Blog.create')->with([
            'blogCategories' => $blogCategories,
        ]);
    }

    public function store(Request $request)
    {
        $arrayFields = ['blog_category_ids'];
        foreach ($arrayFields as $field) {
            if ($request->has($field) && is_array($request->input($field))) {
                $filtered = array_filter($request->input($field), function ($value) {
                    return $value !== '' && $value !== null && $value !== 'null';
                });
                $filtered = array_values($filtered);
                $request->merge([$field => empty($filtered) ? null : $filtered]);
            }
        }
        $request->validate([
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('blogs', 'permalink')->whereNull('deleted_at'),
            ],
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'image_tagline' => 'nullable|string|max:255',
            'image_description' => 'nullable|string',
            'video_url' => 'nullable|url|max:500',
            'video_content' => 'nullable|string',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'featured_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'blog_category_ids' => 'nullable|array',
            'blog_category_ids.*' => 'integer|exists:blog_categories,id',
            'active' => 'required|integer|in:0,1,2,3',
            'is_featured' => 'nullable|boolean',
            'fimage_alt_text' => 'nullable|string',
            '_seo_title' => 'nullable|string|max:255',
            '_seo_tags' => 'nullable|string',
            '_seo_description' => 'nullable|string',
            '_canonical_url' => 'nullable|url',
            '_schema_markup' => 'nullable|string',
            '_page_index' => 'nullable|integer',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'image_alt_text' => 'nullable|string',
            'video_tagline' => 'nullable|string|max:255',
        ]);
        $data = $request->except(['cover_image', 'featured_images', 'blog_category_ids', 'blog_categories_id']);
        $data['created_by'] = Auth::id();
        
        // Process Summernote content fields to save base64 images as physical files
        // Note: This is a fallback in case frontend processing didn't catch all base64 images
        if (!empty($data['content'])) {
            $data['content'] = \App\Helpers\BlogImageHelper::processSummernoteContent($data['content']);
        }
        if (!empty($data['video_content'])) {
            $data['video_content'] = \App\Helpers\BlogImageHelper::processSummernoteContent($data['video_content']);
        }
        if (!empty($data['image_description'])) {
            $data['image_description'] = \App\Helpers\BlogImageHelper::processSummernoteContent($data['image_description']);
        }
        
        // Handle blog_categories_id - save as JSON array
        if ($request->has('blog_category_ids') && is_array($request->blog_category_ids)) {
            $validCategoryIds = array_filter($request->blog_category_ids, function ($id) {
                return is_numeric($id) && $id > 0;
            });
            $data['blog_categories_id'] = !empty($validCategoryIds) ? json_encode(array_values($validCategoryIds)) : null;
        } elseif ($request->has('blog_categories_id') && is_array($request->blog_categories_id)) {
            $validCategoryIds = array_filter($request->blog_categories_id, function ($id) {
                return is_numeric($id) && $id > 0;
            });
            $data['blog_categories_id'] = !empty($validCategoryIds) ? json_encode(array_values($validCategoryIds)) : null;
        } else {
            $data['blog_categories_id'] = null;
        }
        
        // Handle is_featured
        $data['is_featured'] = $request->boolean('is_featured');
        $coverDir = public_path('blogs/covers');
        $featuredDir = public_path('blogs/featured');
        $imageDir = public_path('blogs/images');
        if (!File::exists($coverDir)) {
            File::makeDirectory($coverDir, 0755, true);
        }
        if (!File::exists($featuredDir)) {
            File::makeDirectory($featuredDir, 0755, true);
        }
        if (!File::exists($imageDir)) {
            File::makeDirectory($imageDir, 0755, true);
        }
        if ($request->hasFile('cover_image')) {
            $coverImage = $request->file('cover_image');
            $coverImageName = 'blog_cover_' . time() . '_' . uniqid() . '.' . $coverImage->getClientOriginalExtension();
            $coverImage->move($coverDir, $coverImageName);
            $data['cover_image'] = 'blogs/covers/' . $coverImageName;
        }
        $featuredImagePaths = [];
        if ($request->hasFile('featured_images')) {
            foreach ($request->file('featured_images') as $index => $image) {
                $imageName = 'blog_featured_' . time() . '_' . $index . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $image->move($featuredDir, $imageName);
                $featuredImagePaths[] = 'blogs/featured/' . $imageName;
            }
        }
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = 'blog_image_' . time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $image->move($imageDir, $imageName);
            $data['image'] = 'blogs/images/' . $imageName;
        }
        $data['featured_images'] = implode(',', $featuredImagePaths);
        $blog = Blog::create($data);
        
        // blog_categories_id is already saved in the update above
        $message = 'Blog created successfully.';
        if ($request->has('stay_on_page')) {
            return redirect()->route('blogs.create')->with('success', $message);
        }
        return redirect()->route('blogs.index')->with('success', $message);
    }

    public function show(Blog $blog)
    {
        // Categories are accessed via the blogCategories accessor (getBlogCategoriesAttribute)
        // No need to eager load since it's already handled by the accessor
        return view('Admin.Blog.show', compact('blog'));
    }

    public function edit(Blog $blog)
    {
        // Categories are accessed via the blogCategories accessor (getBlogCategoriesAttribute)
        // No need to eager load since it's already handled by the accessor
        $blogCategories = BlogCategory::active()->ordered()->get();
        return view('Admin.Blog.edit', compact('blog', 'blogCategories'));
    }

    public function update(Request $request, Blog $blog)
    {
        $arrayFields = ['blog_category_ids'];
        foreach ($arrayFields as $field) {
            if ($request->has($field) && is_array($request->input($field))) {
                $filtered = array_filter($request->input($field), function ($value) {
                    return $value !== '' && $value !== null && $value !== 'null';
                });
                $filtered = array_values($filtered);
                $request->merge([$field => empty($filtered) ? null : $filtered]);
            }
        }
        $request->validate([
            'title' => 'required|string|max:255',
            'permalink' => [
                'required',
                'string',
                Rule::unique('blogs', 'permalink')->whereNull('deleted_at')->ignore($blog->id),
            ],
            'description' => 'nullable|string',
            'content' => 'nullable|string',
            'image_tagline' => 'nullable|string|max:255',
            'image_description' => 'nullable|string',
            'video_url' => 'nullable|url|max:500',
            'video_content' => 'nullable|string',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'featured_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'blog_category_ids' => 'nullable|array',
            'blog_category_ids.*' => 'integer|exists:blog_categories,id',
            'blog_categories_id' => 'nullable|array',
            'blog_categories_id.*' => 'integer|exists:blog_categories,id',
            'is_featured' => 'nullable|boolean',
            'fimage_alt_text' => 'nullable|string',
            'active' => 'required|integer|in:0,1,2,3',
            '_seo_title' => 'nullable|string|max:255',
            '_seo_tags' => 'nullable|string',
            '_seo_description' => 'nullable|string',
            '_canonical_url' => 'nullable|url',
            '_schema_markup' => 'nullable|string',
            '_page_index' => 'nullable|integer',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'image_alt_text' => 'nullable|string',
            'video_tagline' => 'nullable|string|max:255',
        ]);
        $data = $request->except(['cover_image', 'featured_images', 'blog_category_ids', 'blog_categories_id']);
        $data['updated_by'] = Auth::id();
        
        // Process Summernote content fields to save base64 images as physical files
        // Note: This is a fallback in case frontend processing didn't catch all base64 images
        if (!empty($data['content'])) {
            $data['content'] = \App\Helpers\BlogImageHelper::processSummernoteContent($data['content'], $blog->id);
        }
        if (!empty($data['video_content'])) {
            $data['video_content'] = \App\Helpers\BlogImageHelper::processSummernoteContent($data['video_content'], $blog->id);
        }
        if (!empty($data['image_description'])) {
            $data['image_description'] = \App\Helpers\BlogImageHelper::processSummernoteContent($data['image_description'], $blog->id);
        }
        
        // Handle blog_categories_id - save as JSON array
        if ($request->has('blog_category_ids') && is_array($request->blog_category_ids)) {
            $validCategoryIds = array_filter($request->blog_category_ids, function ($id) {
                return is_numeric($id) && $id > 0;
            });
            $data['blog_categories_id'] = !empty($validCategoryIds) ? json_encode(array_values($validCategoryIds)) : null;
        } elseif ($request->has('blog_categories_id') && is_array($request->blog_categories_id)) {
            $validCategoryIds = array_filter($request->blog_categories_id, function ($id) {
                return is_numeric($id) && $id > 0;
            });
            $data['blog_categories_id'] = !empty($validCategoryIds) ? json_encode(array_values($validCategoryIds)) : null;
        } else {
            // If no categories provided, set to null
            if ($request->has('blog_category_ids') || $request->has('blog_categories_id')) {
                $data['blog_categories_id'] = null;
            }
        }
        
        // Handle is_featured
        $data['is_featured'] = $request->boolean('is_featured');
        $coverDir = public_path('blogs/covers');
        $featuredDir = public_path('blogs/featured');
        $imageDir = public_path('blogs/images');
        if (!File::exists($coverDir)) {
            File::makeDirectory($coverDir, 0755, true);
        }
        if (!File::exists($featuredDir)) {
            File::makeDirectory($featuredDir, 0755, true);
        }
        if (!File::exists($imageDir)) {
            File::makeDirectory($imageDir, 0755, true);
        }
        if ($request->hasFile('cover_image')) {
            if ($blog->cover_image && file_exists(public_path($blog->cover_image))) {
                unlink(public_path($blog->cover_image));
            }
            $coverImage = $request->file('cover_image');
            $coverImageName = 'blog_cover_' . time() . '_' . uniqid() . '.' . $coverImage->getClientOriginalExtension();
            $coverImage->move($coverDir, $coverImageName);
            $data['cover_image'] = 'blogs/covers/' . $coverImageName;
        }
        if ($request->hasFile('featured_images')) {
            $oldFeaturedImages = $blog->featured_images_array ?? [];
            foreach ($oldFeaturedImages as $oldImage) {
                if (file_exists(public_path($oldImage))) {
                    unlink(public_path($oldImage));
                }
            }
            $featuredImagePaths = [];
            foreach ($request->file('featured_images') as $index => $image) {
                $imageName = 'blog_featured_' . time() . '_' . $index . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $image->move($featuredDir, $imageName);
                $featuredImagePaths[] = 'blogs/featured/' . $imageName;
            }
            $data['featured_images'] = implode(',', $featuredImagePaths);
        }
        if ($request->hasFile('image')) {
            if ($blog->image && file_exists(public_path($blog->image))) {
                unlink(public_path($blog->image));
            }
            $image = $request->file('image');
            $imageName = 'blog_image_' . time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $image->move($imageDir, $imageName);
            $data['image'] = 'blogs/images/' . $imageName;
        }
        $blog->update($data);
        // blog_categories_id is already saved in the update above
        return redirect()->route('blogs.index')->with('success', 'Blog updated successfully.');
    }

    public function uploadContentImage(Request $request)
    {
        // Check if it's a URL or file upload
        if ($request->has('image_url') && $request->filled('image_url')) {
            // Download and save external image from URL
            return $this->downloadAndSaveExternalImage($request->input('image_url'));
        }
        
        $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:5120', // 5MB max
        ]);

        try {
            // Create directory if it doesn't exist
            $uploadPath = public_path('blogs/content-images');
            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            // Generate unique filename
            $filename = 'blog_' . time() . '_' . uniqid() . '.jpg';
            $filePath = $uploadPath . '/' . $filename;

            // Get uploaded image
            $image = $request->file('image');
            $tempPath = $image->getPathname();

            // Convert to JPEG using GD library
            $this->convertToJpeg($tempPath, $filePath);

            // Return URL
            $url = asset('blogs/content-images/' . $filename);

            return response()->json([
                'success' => true,
                'url' => $url,
                'path' => 'blogs/content-images/' . $filename
            ]);
        } catch (\Exception $e) {
            Log::error('Blog content image upload failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload image: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download and save external image from URL
     */
    private function downloadAndSaveExternalImage($imageUrl)
    {
        try {
            // Create directory if it doesn't exist
            $uploadPath = public_path('blogs/content-images');
            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }

            // Generate unique filename
            $filename = 'blog_' . time() . '_' . uniqid() . '.jpg';
            $filePath = $uploadPath . '/' . $filename;

            // Download the image
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $imageUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            
            $imageData = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode === 200 && $imageData !== false) {
                // Convert to JPEG
                $image = imagecreatefromstring($imageData);
                if ($image !== false) {
                    $width = imagesx($image);
                    $height = imagesy($image);
                    $jpegImage = imagecreatetruecolor($width, $height);
                    $white = imagecolorallocate($jpegImage, 255, 255, 255);
                    imagefill($jpegImage, 0, 0, $white);
                    imagecopy($jpegImage, $image, 0, 0, 0, 0, $width, $height);
                    imagejpeg($jpegImage, $filePath, 90);
                    imagedestroy($image);
                    imagedestroy($jpegImage);

                    $url = asset('blogs/content-images/' . $filename);
                    return response()->json([
                        'success' => true,
                        'url' => $url,
                        'path' => 'blogs/content-images/' . $filename
                    ]);
                }
            }

            return response()->json([
                'success' => false,
                'message' => 'Failed to download image from URL'
            ], 400);
        } catch (\Exception $e) {
            Log::error('Failed to download external image', [
                'url' => $imageUrl,
                'error' => $e->getMessage()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to download image: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Convert uploaded image to JPEG format
     */
    private function convertToJpeg($sourcePath, $destinationPath)
    {
        try {
            // Get image info
            $imageInfo = getimagesize($sourcePath);
            
            if ($imageInfo === false) {
                throw new \Exception('Invalid image file');
            }
            
            $mimeType = $imageInfo['mime'];
            $width = $imageInfo[0];
            $height = $imageInfo[1];
            
            // Create image resource based on MIME type
            switch ($mimeType) {
                case 'image/jpeg':
                    $image = imagecreatefromjpeg($sourcePath);
                    break;
                case 'image/png':
                    $image = imagecreatefrompng($sourcePath);
                    break;
                case 'image/gif':
                    $image = imagecreatefromgif($sourcePath);
                    break;
                case 'image/webp':
                    $image = imagecreatefromwebp($sourcePath);
                    break;
                default:
                    throw new \Exception('Unsupported image format: ' . $mimeType);
            }
            
            if ($image === false) {
                throw new \Exception('Failed to create image resource');
            }
            
            // Create a white background for JPEG
            $jpegImage = imagecreatetruecolor($width, $height);
            $white = imagecolorallocate($jpegImage, 255, 255, 255);
            imagefill($jpegImage, 0, 0, $white);
            
            // Copy the original image onto the white background
            imagecopy($jpegImage, $image, 0, 0, 0, 0, $width, $height);
            
            // Save as JPEG with high quality (90%)
            $success = imagejpeg($jpegImage, $destinationPath, 90);
            
            // Clean up memory
            imagedestroy($image);
            imagedestroy($jpegImage);
            
            if (!$success) {
                throw new \Exception('Failed to save JPEG image');
            }
            
        } catch (\Exception $e) {
            // Fallback: copy the original file if conversion fails
            copy($sourcePath, $destinationPath);
            throw $e;
        }
    }

    public function destroy(Blog $blog)
    {
        if ($blog->cover_image && file_exists(public_path($blog->cover_image))) {
            unlink(public_path($blog->cover_image));
        }
        $featuredImages = $blog->featured_images_array;
        foreach ($featuredImages as $image) {
            if (file_exists(public_path($image))) {
                unlink(public_path($image));
            }
        }
        if ($blog->image && file_exists(public_path($blog->image))) {
            unlink(public_path($blog->image));
        }
        $blog->delete();
        return redirect()->route('blogs.index')->with('success', 'Blog deleted successfully.');
    }

    public function updateStatus(Request $request, Blog $blog)
    {
        $request->validate([
            'active' => 'required|integer|in:0,1,2,3'
        ]);
        $blog->update([
            'active' => $request->active,
            'updated_by' => Auth::id()
        ]);
        return response()->json([
            'success' => true,
            'message' => 'Blog status updated successfully.'
        ]);
    }

    public function trash()
    {
        // Categories are accessed via the blogCategories accessor (getBlogCategoriesAttribute)
        // No need to eager load since it's already handled by the accessor
        $blogs = Blog::where('active', 3)
            ->orderBy('updated_at', 'desc')
            ->get();

        return view('Admin.Blog.trash', compact('blogs'));
    }

    public function restore($id)
    {
        $blog = Blog::findOrFail($id);
        $blog->update([
            'active' => 0, // Restore as draft
            'updated_by' => Auth::id()
        ]);
        return redirect()->route('blogs.trash')->with('success', 'Blog restored successfully.');
    }

    public function checkPermalink(Request $request)
    {
        $permalink = $request->input('permalink');
        if (empty($permalink)) {
            return response()->json([
                'available' => false,
                'message' => 'Permalink is required.'
            ]);
        }
        $query = Blog::where('permalink', $permalink);
        $exists = $query->whereNull('deleted_at')->exists();
        return response()->json([
            'available' => !$exists,
            'message' => $exists ? 'Permalink already exists.' : 'Permalink is available.'
        ]);
    }
}