<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use Carbon\Carbon;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver as GdDriver;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class CategoryController extends Controller
{
    public function index(Request $request)
    {
        $query = Category::with('childCategories');

        // Keyword filter
        if ($request->filled('keyword')) {
            $query->where('title', 'like', '%' . $request->keyword . '%');
        }

        // Status filter
        if ($request->filled('status')) {
            switch ($request->status) {
                case 'active':
                    $query->where('is_active', true);
                    break;
                case 'inactive':
                    $query->where('is_active', false);
                    break;
                case 'today':
                    $query->whereDate('created_at', Carbon::today());
                    break;
                case 'yesterday':
                    $query->whereDate('created_at', Carbon::yesterday());
                    break;
                case '7days':
                    $query->where('created_at', '>=', Carbon::now()->subDays(7));
                    break;
                case '30days':
                    $query->where('created_at', '>=', Carbon::now()->subDays(30));
                    break;
                case 'this_month':
                    $query->whereMonth('created_at', Carbon::now()->month);
                    break;
                case 'last_month':
                    $query->whereMonth('created_at', Carbon::now()->subMonth()->month);
                    break;
            }
        }

        $categories = $query->ordered()->get();
        return view('Admin.Category.index', compact('categories'));
    }

    public function get_category()
    {
        $categories = Category::active()->ordered()->get();
        return response()->json(['data' => $categories, 'status' => 'success']);
    }

    public function store(Request $request)
    {
        $isUpdate = $request->filled('category_id');
        $categoryId = $request->input('category_id');
        
        $rules = [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ];
        
        $validated_data = $request->validate($rules);
        
        // Auto-assign sort order if not provided or is 0
        if (!isset($validated_data['sort_order']) || $validated_data['sort_order'] == 0) {
            $validated_data['sort_order'] = Category::getNextSortOrder();
        }
        
        // Ensure directories exist
        Storage::makeDirectory('public/categories');
        $saveDir = public_path('categories');
        if (!File::isDirectory($saveDir)) {
            File::makeDirectory($saveDir, 0755, true);
        }
        
        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = time() . '_category.' . $image->getClientOriginalExtension();
            $image->move($saveDir, $imageName);
            $validated_data['image'] = 'categories/' . $imageName;
        }
        
        // Save or update category
        if ($isUpdate) {
            $category = Category::findOrFail($categoryId);
            $category->update($validated_data);
            $message = 'Category updated successfully.';
        } else {
            $category = Category::create($validated_data);
            $message = 'Category created successfully.';
        }
        
        return redirect()->route('category.index')->with('success', $message);
    }

    public function getNextSortOrder()
    {
        $nextSortOrder = Category::getNextSortOrder();
        return response()->json(['next_sort_order' => $nextSortOrder]);
    }

  

    public function delete(Request $request)
    {
        $id = $request->id;
        $category = Category::find($id);
        if ($category) {
            // Delete image if it exists
            if (!empty($category->image) && File::exists(public_path($category->image))) {
                File::delete(public_path($category->image));
            }
            // Delete the category
            $category->delete();
            return redirect()->route('category.index')->with('success', 'Category deleted successfully.');
        }

        return redirect()->route('category.index')->with('error', 'Something went wrong.');
    }

    public function edit_category(Request $request)
    {
        $id = $request->id;
        $cat = Category::with('categoryMeta')->find($id);

        return response()->json($cat, 200, [], JSON_UNESCAPED_SLASHES);
    }

    public function show_category_detail(Request $request)
    {
        $category = Category::with('childCategories')->find($request->id);

        if (!$category) {
            return response()->json(['error' => 'Category not found'], 404);
        }

        return response()->json([
            'id' => $category->id,
            'title' => $category->title,
            'icon' => $category->icon,
            'description' => $category->description,
            'image' => $category->image,
            'is_active' => $category->is_active,
            'sort_order' => $category->sort_order,
            'child_categories' => $category->childCategories
        ]);
    }

    public function update_category(Request $request, $id)
    {
        // Validate the incoming request data
        $validator = Validator::make($request->all(), [
           'is_active' => 'boolean',
        ]);
        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'message' => $validator->errors()], 422);
        }
        $category = Category::find($id);
        if (!$category) {
            return response()->json(['status' => 'error', 'message' => 'Category not found.'], 404);
        }
        $category->update($validator->validated());
        return response()->json(['status' => 'success', 'message' => 'Category updated successfully.']);
    }

}
