<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StorageItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class StorageItemController extends Controller
{
    public function index(Request $request)
    {
        $storageItems = StorageItem::ordered()->get();

        return view('Admin.StorageItem.index', compact('storageItems'));
    }

    public function store(Request $request)
    {
        $isUpdate = $request->filled('storage_item_id');

        $permalinkInput = $request->input('permalink');
        $generatedPermalink = Str::slug($request->input('title', ''));

        if (!$permalinkInput && empty($generatedPermalink)) {
            $generatedPermalink = Str::uuid()->toString();
        }

        $request->merge([
            'permalink' => $permalinkInput ? Str::slug($permalinkInput) : $generatedPermalink,
        ]);

        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'permalink' => [
                'required',
                'string',
                'max:255',
                Rule::unique('storage_items', 'permalink')->ignore($request->input('storage_item_id')),
            ],
            'price' => ['nullable', 'numeric', 'min:0'],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        if (!isset($validated['sort_order']) || (int) $validated['sort_order'] <= 0) {
            $validated['sort_order'] = StorageItem::getNextSortOrder();
        }

        $validated['is_active'] = $validated['is_active'] ?? true;

        DB::transaction(function () use ($validated, $isUpdate, $request) {
            if ($isUpdate) {
                $storageItem = StorageItem::findOrFail($request->input('storage_item_id'));
                $storageItem->update($validated);
            } else {
                StorageItem::create($validated);
            }
        });

        $message = $isUpdate ? 'Storage item updated successfully.' : 'Storage item created successfully.';

        return redirect()->route('admin.storage-items.index')->with('success', $message);
    }

    public function getNextSortOrder()
    {
        return response()->json([
            'next_sort_order' => StorageItem::getNextSortOrder(),
        ]);
    }

    public function delete(Request $request)
    {
        $storageItem = StorageItem::find($request->input('storage_item'));

        if (!$storageItem) {
            return redirect()
                ->route('admin.storage-items.index')
                ->with('error', 'Storage item not found.');
        }

        $storageItem->delete();

        return redirect()
            ->route('admin.storage-items.index')
            ->with('success', 'Storage item deleted successfully.');
    }

    public function edit(Request $request)
    {
        $storageItem = StorageItem::findOrFail($request->input('id'));

        return response()->json($storageItem);
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'is_active' => ['required', 'boolean'],
        ]);

        $storageItem = StorageItem::findOrFail($id);
        $storageItem->update($validated);

        return response()->json([
            'status' => 'success',
            'message' => 'Storage item status updated successfully.',
        ]);
    }
}

