@include('Admin.Includes.header')
@php
    use Illuminate\Support\Facades\Auth;
    $logged_in_user = Auth::user();
@endphp

<div class="page-content">
    <div class="card shadow-sm mb-4">
        <div class="p-3 d-flex justify-content-between align-items-center flex-wrap">
            <div class="clearfix">
                <div class="pd-t-5 pd-b-5">
                    <h1 class="fs-4 pd-0 mg-0 tx-20 tx-dark">{{ __('Categories') }}</h1>
                </div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb bg-transparent p-0 m-0 align-items-center">
                        <li class="breadcrumb-item">
                            <a href="{{ route('Admin.Dashboard') }}" class="link-primary text-decoration-none">
                                <i class="fa-solid fa-house me-1"></i> Dashboard
                            </a>
                        </li>
                        <li class="breadcrumb-item active" aria-current="page">
                            Categories
                        </li>
                    </ol>
                </nav>
            </div>
          
        </div>
    </div>

    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    @if ($errors->any())
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    <div class="row">
        <div class="col-xxl-5">
            <!-- Basic Category Information Card -->
            <div class="card mb-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">Create Category</h6>
                </div>
                <div class="card-body">
                    <form autocomplete="off" method="POST" action="{{ route('category.store') }}"
                        enctype="multipart/form-data" id="categoryForm">
                        @csrf
                        <input type="hidden" name="category_id" id="category_id">
                        <div id="update_success_message"></div>
                        <div id="saveform_errList" role="alert"></div>

                        <div class="row">
                            <!-- Title -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="category_title" class="form-label">Category Title<span
                                            class="text-danger">*</span></label>
                                    <input type="text" id="category_title" name="title" required
                                        class="form-control" placeholder="Enter Category">
                                    <div class="invalid-feedback">Please enter a category title.</div>
                                </div>
                            </div>





                            <!-- Description -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="category_description" class="form-label">Description</label>
                                    <textarea id="category_description" name="description" class="form-control" rows="3"
                                        placeholder="Enter category description"></textarea>
                                </div>
                            </div>

                            <!-- Image -->
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="category_image" class="form-label">Image</label>
                                    <input type="file" id="category_image" name="image" class="form-control"
                                        accept="image/*" onchange="previewImage(this)">
                                    <div id="image_preview" class="mt-2" style="display: none;">
                                        <img id="preview_img" src="" alt="Preview"
                                            style="max-width: 200px; max-height: 150px; border-radius: 5px;">
                                    </div>
                                </div>
                            </div>

                            <!-- Status -->
                            <div class="col-6">
                                <div class="mb-3">
                                    <label for="category_status" class="form-label">Status</label>
                                    <select id="category_status" name="is_active" class="form-select">
                                        <option value="1">Active</option>
                                        <option value="0">Inactive</option>
                                    </select>
                                </div>
                            </div>

                            <!-- Sort Order -->
                            <div class="col-6">
                                <div class="mb-3">
                                    <label for="category_sort_order" class="form-label">Sort Order</label>
                                    <input type="number" id="category_sort_order" name="sort_order"
                                        class="form-control" value="0" min="0">
                                </div>
                            </div>
                        </div>

                        <div class="d-flex justify-content-end">
                            <button type="button" class="btn btn-light me-2" onclick="resetForm()">Reset</button>
                            <button type="submit" class="btn btn-success" id="submitBtn">Save Category</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-xxl-7">
            <!-- Categories List Card -->
            <div class="card">
                <div class="card-header d-flex justify-content-between">
                    <h5 class="card-title mb-0">Category Table</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="data_Table" class="table table-borderless">
                            <thead>
                                <tr>
                                    <th>Title</th>
                                    <th>Status</th>
                                    <th>Sort Order</th>
                                    <th width="180px">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($categories as $category)
                                    <tr class="align-middle">
                                        <td>
                                            <div class="d-flex align-items-center">
                                                @if ($category->image)
                                                    <img src="{{ asset($category->image) }}"
                                                        alt="{{ $category->title }}" class="rounded me-2"
                                                        style="width: 30px; height: 30px; object-fit: cover;">
                                                @endif
                                                <div>
                                                    <div class="fw-medium">{{ $category->title }}</div>
                                                    @if ($category->description)
                                                        <small
                                                            class="text-muted">{{ Str::limit($category->description, 50) }}</small>
                                                    @endif
                                                </div>
                                            </div>
                                        </td>

                                        <td class="status-cell" data-category-id="{{ $category->id }}">
                                            @if ($category->is_active)
                                                <span
                                                    class="badge bg-success-subtle text-success px-3 py-2">Active</span>
                                            @else
                                                <span
                                                    class="badge bg-danger-subtle text-danger px-3 py-2">Inactive</span>
                                            @endif
                                        </td>
                                        <td>{{ $category->sort_order }}</td>
                                        <td>
                                            <div class="dropdown">
                                                <button class="btn btn-sm text-dark rounded shadow-sm" type="button"
                                                    data-bs-toggle="dropdown" aria-expanded="false"
                                                    style="background-color: #c2d6e7;">
                                                    &#8942;
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li>
                                                        <button class="dropdown-item"
                                                            onclick="editCategory({{ $category->id }})">
                                                            <i class="bi bi-pencil-square me-2"></i>Edit
                                                        </button>
                                                    </li>
                                                    <li>
                                                        <button class="dropdown-item"
                                                            onclick="updateStatus({{ $category->id }}, '{{ $category->is_active ? 0 : 1 }}', this)">
                                                            <i
                                                                class="fa fa-{{ $category->is_active ? 'ban' : 'check' }} me-2"></i>{{ $category->is_active ? 'Inactive' : 'Active' }}
                                                        </button>
                                                    </li>
                                                    <li>
                                                        <button class="dropdown-item text-danger"
                                                            data-bs-toggle="modal"
                                                            data-bs-target="#deleteConfirmModal{{ $category->id }}">
                                                            <i class="bi bi-trash me-2"></i>Delete
                                                        </button>
                                                    </li>
                                                </ul>
                                            </div>
                                        </td>
                                    </tr>
                                @empty
                                <tr class="no-data-row">
                                    <td></td>
                                    <td class="text-end text-muted py-4">No data</td>
                                    <td></td>
                                    <td></td>
                                </tr>
                                @endforelse
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th>Title</th>
                                    <th>Status</th>
                                    <th>Sort Order</th>
                                    <th width="180px">Actions</th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Delete Modals for each category --}}
@foreach ($categories as $category)
    @include('Admin.Components.delete-modal', [
        'id' => $category->id,
        'name' => $category->title,
        'route' => 'category.delete',
        'routeParam' => $category->id,
    ])
@endforeach

@include('Admin.Includes.footer')

<script>
    // Image preview
    function previewImage(input) {
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            reader.onload = function(e) {
                document.getElementById('preview_img').src = e.target.result;
                document.getElementById('image_preview').style.display = 'block';
            }
            reader.readAsDataURL(input.files[0]);
        }
    }

    // Reset form
    function resetForm() {
        document.getElementById('categoryForm').reset();
        document.getElementById('category_id').value = '';
        document.getElementById('image_preview').style.display = 'none';
        document.getElementById('submitBtn').textContent = 'Save Category';
        loadNextCategorySortOrder();
    }

    // Load next sort order for categories
    function loadNextCategorySortOrder() {
        $.ajax({
            url: "{{ route('admin.categories.nextSortOrder') }}",
            type: "GET",
            success: function(response) {
                // Always set the next sort order as default value
                $('#category_sort_order').val(response.next_sort_order);
            },
            error: function() {
                $('#category_sort_order').val('1');
            }
        });
    }

    // Edit category
    function editCategory(id) {
        fetch(`/edit_category?id=${id}`)
            .then(response => response.json())
            .then(data => {
                document.getElementById('category_id').value = data.id;
                document.getElementById('category_title').value = data.title;
                document.getElementById('category_description').value = data.description || '';
                document.getElementById('category_status').value = data.is_active ? '1' : '0';
                document.getElementById('category_sort_order').value = data.sort_order || 0;
                document.getElementById('submitBtn').textContent = 'Update Category';

                // Show current image if exists
                if (data.image) {
                    document.getElementById('preview_img').src = `{{ asset('') }}${data.image}`;
                    document.getElementById('image_preview').style.display = 'block';
                }

                // Scroll to form
                document.getElementById('categoryForm').scrollIntoView({
                    behavior: 'smooth'
                });
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error loading category data');
            });
    }

    // Update status function
    function updateStatus(categoryId, status, element) {
        const statusText = status === '1' ? 'Active' : 'Inactive';

        $.ajaxSetup({
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        });

        $.ajax({
            url: `/update_category/${categoryId}`,
            type: 'POST',
            data: {
                is_active: status,
                _token: '{{ csrf_token() }}'
            },
            beforeSend: function() {
                $(element).prop('disabled', true);
            },
            success: function(response) {
                Swal.fire({
                    icon: 'success',
                    title: 'Status Updated',
                    text: 'Category status updated successfully',
                    timer: 1500,
                    showConfirmButton: false
                });

                const row = $(element).closest('tr');
                const statusCell = row.find('.status-cell');

                if (status === '1') {
                    statusCell.html(
                        '<span class="badge bg-success-subtle text-success px-3 py-2">Active</span>');
                } else {
                    statusCell.html(
                        '<span class="badge bg-danger-subtle text-danger px-3 py-2">Inactive</span>');
                }
                setTimeout(function() {
                    window.location.reload();
                }, 1000); 
            },
            error: function(xhr) {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'Failed to update status'
                });
            },
            complete: function() {
                $(element).prop('disabled', false);
            }
        });
    }

    // Initialize DataTable properly
    $(document).ready(function() {
        // Load next sort order on page load
        loadNextCategorySortOrder();

        // Destroy existing DataTable if it exists
        if ($.fn.DataTable.isDataTable('#data_Table')) {
            $('#data_Table').DataTable().destroy();
        }

        // Initialize DataTable with proper configuration
        $('#data_Table').DataTable({
            responsive: true,
            autoWidth: false,
            columnDefs: [{
                    orderable: false,
                    targets: -1
                } // Disable sorting on "Actions" column
            ],
            language: {
                searchPlaceholder: 'Search...',
                sSearch: ''
            },
            "order": [
                [0, "desc"]
            ]
        });
    });
</script>
