@include('Admin.Includes.header')
@php
    use Illuminate\Support\Facades\Auth;
    $logged_in_user = Auth::user();
@endphp

<div class="page-content">
    <div class="card shadow-sm mb-4">
        <div class="p-3 d-flex justify-content-between align-items-center flex-wrap">
            <div class="clearfix">
                <div class="pd-t-5 pd-b-5">
                    <h1 class="fs-4 pd-0 mg-0 tx-20 tx-dark">{{ __('Quotes/Orders Management') }}</h1>
                </div>

                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb bg-transparent p-0 m-0 align-items-center">
                        <li class="breadcrumb-item">
                            <a href="{{ url('/admin') }}" class="link-primary text-decoration-none">
                                <i class="fa-solid fa-home me-1"></i> Dashboard
                            </a>
                        </li>
                        <li class="breadcrumb-item active" aria-current="page">
                            Quotes/Orders
                        </li>
                    </ol>
                </nav>
            </div>
        </div>
    </div>

    <!-- Advanced Filters Card -->
    <div class="card shadow-sm mb-4">
        <div class="card-header">
            <div class="d-flex justify-content-between align-items-center">
                <h6 class="mb-0">
                    <i class="bi bi-funnel me-2"></i>Advanced Filters
                </h6>
                <button class="btn btn-sm btn-outline-primary" type="button" data-bs-toggle="collapse"
                    data-bs-target="#filterCollapse" aria-expanded="true" aria-controls="filterCollapse">
                    <i class="bi bi-chevron-down"></i>
                </button>
            </div>
        </div>
        <div class="collapse show" id="filterCollapse">
            <div class="card-body">
                <form method="GET" action="{{ route('admin.quotes.index') }}" class="row g-3">
                    <div class="col-lg-3 col-md-4">
                        <label for="keyword_filter" class="form-label">Search</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="bi bi-search"></i></span>
                            <input type="text" class="form-control" id="keyword_filter" name="keyword"
                                value="{{ request('keyword') }}" placeholder="Search quote/order number, customer...">
                        </div>
                    </div>

                    <div class="col-lg-2 col-md-4">
                        <label for="quote_type_filter" class="form-label">Type</label>
                        <select class="form-select" id="quote_type_filter" name="quote_type">
                            <option value="">All Types</option>
                            <option value="quote" {{ request('quote_type') == 'quote' ? 'selected' : '' }}>Quote</option>
                            <option value="order" {{ request('quote_type') == 'order' ? 'selected' : '' }}>Order</option>
                        </select>
                    </div>

                    <div class="col-lg-2 col-md-4">
                        <label for="status_filter" class="form-label">Status</label>
                        <select class="form-select" id="status_filter" name="status">
                            <option value="">All Status</option>
                            <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                            <option value="approved" {{ request('status') == 'approved' ? 'selected' : '' }}>Approved</option>
                            <option value="rejected" {{ request('status') == 'rejected' ? 'selected' : '' }}>Rejected</option>
                        </select>
                    </div>

                    <div class="col-lg-2 col-md-4">
                        <label for="calculation_type_filter" class="form-label">Calculation Type</label>
                        <select class="form-select" id="calculation_type_filter" name="calculation_type">
                            <option value="">All Types</option>
                            <option value="truck_only" {{ request('calculation_type') == 'truck_only' ? 'selected' : '' }}>Truck Only</option>
                            <option value="items_only" {{ request('calculation_type') == 'items_only' ? 'selected' : '' }}>Items Only</option>
                            <option value="items_and_packages" {{ request('calculation_type') == 'items_and_packages' ? 'selected' : '' }}>Items & Packages</option>
                        </select>
                    </div>

                    <div class="col-lg-2 col-md-4">
                        <label for="job_type_filter" class="form-label">Job Type</label>
                        <select class="form-select" id="job_type_filter" name="job_type">
                            <option value="">All Job Types</option>
                            <option value="local_job" {{ request('job_type') == 'local_job' ? 'selected' : '' }}>Local Job</option>
                            <option value="interstate_job" {{ request('job_type') == 'interstate_job' ? 'selected' : '' }}>Interstate Job</option>
                        </select>
                    </div>

                    <div class="col-lg-2 col-md-4">
                        <label for="payment_method_filter" class="form-label">Payment Method</label>
                        <select class="form-select" id="payment_method_filter" name="payment_method">
                            <option value="">All Methods</option>
                            <option value="stripe" {{ request('payment_method') == 'stripe' ? 'selected' : '' }}>Stripe</option>
                            <option value="bank_transfer" {{ request('payment_method') == 'bank_transfer' ? 'selected' : '' }}>Bank Transfer</option>
                        </select>
                    </div>

                    <div class="col-lg-2 col-md-4">
                        <label for="date_from_filter" class="form-label">Date From</label>
                        <input type="date" class="form-control" id="date_from_filter" name="date_from"
                            value="{{ request('date_from') }}">
                    </div>

                    <div class="col-lg-2 col-md-4">
                        <label for="date_to_filter" class="form-label">Date To</label>
                        <input type="date" class="form-control" id="date_to_filter" name="date_to"
                            value="{{ request('date_to') }}">
                    </div>

                    <div class="col-lg-3 col-md-12 d-flex align-items-end">
                        <div class="btn-group w-100" role="group">
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-funnel me-1"></i>Apply Filters
                            </button>
                            <a href="{{ route('admin.quotes.index') }}" class="btn btn-outline-secondary">
                                <i class="bi bi-arrow-clockwise me-1"></i>Reset
                            </a>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <div class="card">
        <div class="card-header d-flex justify-content-between">
            <h5 class="card-title mb-0">Quotes/Orders Table</h5>
            <div class="text-muted">
                <div class="d-flex align-items-center">
                    <label for="per_page_select" class="form-label mb-0 me-2 text-muted small">Per page:</label>
                    <select id="per_page_select" class="form-select form-select-sm" style="width: auto;">
                        <option value="10" {{ request('per_page', 10) == 10 ? 'selected' : '' }}>10</option>
                        <option value="25" {{ request('per_page') == 25 ? 'selected' : '' }}>25</option>
                        <option value="50" {{ request('per_page') == 50 ? 'selected' : '' }}>50</option>
                        <option value="100" {{ request('per_page') == 100 ? 'selected' : '' }}>100</option>
                    </select>
                </div>
                  
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-borderless">
                    <thead>
                        <tr>
                            <th width="30px"></th>
                            <th>Order/Quote #</th>
                            <th>Customer</th>
                            <th>Job Type</th>
                            <th>Total Amount</th>
                            <th>Payment Method</th>
                            <th>Payment</th>
                            <th>Status</th>
                            <th width="80px">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse ($quotes as $quote)
                            <tr class="align-middle quote-row" data-quote-id="{{ $quote->id }}">
                                <td class="text-center">
                                    <button class="btn btn-sm btn-link p-0 toggle-details" data-quote-id="{{ $quote->id }}">
                                        <i class="fa-solid fa-chevron-right"></i>
                                    </button>
                                </td>
                                <td class="clickable-cell" data-order="{{ $quote->created_at->timestamp }}">
                                    <div>
                                        @if($quote->order_number)
                                            <div class="fw-semibold text-success">#{{ $quote->order_number }}</div>
                                        @elseif($quote->quote_number)
                                            <div class="fw-semibold">#{{ $quote->quote_number }}</div>
                                        @endif
                                        <small class="text-muted">{{ $quote->created_at->format('j F Y') }}</small>
                                    </div>
                                </td>
                                <td class="clickable-cell">
                                    @if($quote->user)
                                        <div class="fw-semibold">{{ $quote->user->name }}</div>
                                        <small class="text-muted">{{ $quote->user->email }}</small>
                                    @elseif($quote->booking)
                                        <div class="fw-semibold">{{ $quote->booking->name }}</div>
                                        <small class="text-muted">{{ $quote->booking->email }}</small>
                                    @else
                                        <span class="text-muted">N/A</span>
                                    @endif
                                </td>
                                <td class="clickable-cell">
                                    @php
                                        $jobType = $quote->job_type ?? ($quote->raw_data['job_type'] ?? null);
                                        $jobTypeColors = [
                                            'local_job' => 'primary',
                                            'interstate_job' => 'info',
                                        ];
                                        $jobTypeColor = $jobTypeColors[$jobType] ?? 'secondary';
                                    @endphp
                                    @if($jobType)
                                        <span class="badge bg-{{ $jobTypeColor }}">
                                            {{ ucfirst(str_replace('_', ' ', $jobType)) }}
                                        </span>
                                    @else
                                        <span class="text-muted">N/A</span>
                                    @endif
                                </td>
                                <td class="clickable-cell">
                                    @php
                                        // If total_cost is 0, show original_total_cost from raw_data
                                        $displayTotalCost = $quote->total_cost;
                                        if ($displayTotalCost == 0 || $displayTotalCost == null) {
                                            $displayTotalCost = $quote->raw_data['original_total_cost'] ?? ($quote->raw_data['original_total_cost_before_adjustment'] ?? 0);
                                        }
                                    @endphp
                                    <div class="fw-semibold">${{ number_format($displayTotalCost, 2) }}</div>
                                </td>
                                <td class="clickable-cell">
                                    @php
                                        $paymentMethod = $quote->payment_method ?? ($quote->raw_data['payment_method'] ?? null);
                                    @endphp
                                    @if($paymentMethod)
                                        @if($paymentMethod === 'stripe')
                                            <span class="badge bg-primary">Stripe</span>
                                        @elseif($paymentMethod === 'bank_transfer')
                                            <span class="badge bg-info">Bank Transfer</span>
                                        @else
                                            <span class="badge bg-secondary">{{ ucfirst(str_replace('_', ' ', $paymentMethod)) }}</span>
                                        @endif
                                    @else
                                        <span class="text-muted">N/A</span>
                                    @endif
                                </td>
                                <td class="clickable-cell">
                                    @if($quote->deposit_amount || $quote->remaining_amount)
                                        @if($quote->deposit_amount)
                                            <div class="mb-1">
                                                <small class="text-success">
                                                    Deposit: ${{ number_format($quote->deposit_amount, 2) }}
                                                    @if($quote->deposit_payment_status === 'succeeded')
                                                        <span class="badge bg-success ms-1">Paid</span>
                                                    @elseif($quote->deposit_payment_status === 'pending')
                                                        <span class="badge bg-warning ms-1">Pending</span>
                                                    @elseif($quote->deposit_payment_status === 'failed')
                                                        <span class="badge bg-danger ms-1">Failed</span>
                                                    @elseif($quote->deposit_payment_status === 'rejected')
                                                        <span class="badge bg-danger ms-1">Rejected</span>
                                                    @endif
                                                </small>
                                            </div>
                                        @endif
                                        @if($quote->remaining_payment_status === 'succeeded' || (!$quote->remaining_amount || $quote->remaining_amount <= 0))
                                            <small class="text-success">Fully Paid</small>
                                        @elseif($quote->remaining_amount && $quote->remaining_amount > 0)
                                            <div>
                                                <small class="text-warning">
                                                    Remaining: ${{ number_format($quote->remaining_amount, 2) }}
                                                    @if($quote->remaining_payment_status === 'pending')
                                                        <span class="badge bg-warning ms-1">Pending</span>
                                                    @elseif($quote->remaining_payment_status === 'failed')
                                                        <span class="badge bg-danger ms-1">Failed</span>
                                                    @elseif($quote->remaining_payment_status === 'rejected')
                                                        <span class="badge bg-danger ms-1">Rejected</span>
                                                    @endif
                                                </small>
                                            </div>
                                        @else
                                            <small class="text-success">Fully Paid</small>
                                        @endif
                                    @else
                                        <small class="text-muted">No payment</small>
                                    @endif
                                </td>
                                <td class="clickable-cell">
                                    @php
                                        $statusColors = [
                                            'pending' => ['bg-warning', 'Pending'],
                                            'approved' => ['bg-success', 'Approved'],
                                            'rejected' => ['bg-danger', 'Rejected']
                                        ];
                                        $status = $statusColors[$quote->status] ?? ['bg-secondary', ucfirst($quote->status)];
                                        
                                        $typeColors = [
                                            'quote' => ['bg-primary', 'Quote'],
                                            'order' => ['bg-success', 'Order']
                                        ];
                                        $type = $typeColors[$quote->quote_type] ?? ['bg-secondary', ucfirst($quote->quote_type)];
                                    @endphp
                                    <div class="mb-1">
                                        <span class="badge {{ $status[0] }}">{{ $status[1] }}</span>
                                    </div>
                                    <div>
                                        <span class="badge {{ $type[0] }}">{{ $type[1] }}</span>
                                    </div>
                                </td>
                                <td>
                                    <div class="dropdown" onclick="event.stopPropagation();">
                                        <button class="btn btn-sm text-dark rounded shadow-sm"
                                            type="button" data-bs-toggle="dropdown" aria-expanded="false"
                                            style="background-color: #c2d6e7;">
                                            &#8942;
                                        </button>
                                        <ul class="dropdown-menu">
                                            <li>
                                                <a class="dropdown-item" href="{{ route('admin.quotes.show', $quote->id) }}">
                                                    <i class="bi bi-eye me-2"></i> View Details
                                                </a>
                                            </li>
                                            <li>
                                                <hr class="dropdown-divider">
                                            </li>
                                            <li>
                                                <button class="dropdown-item" onclick="changeQuoteStatus({{ $quote->id }}, 'pending')">
                                                    <i class="bi bi-clock me-2"></i> Set Pending
                                                </button>
                                            </li>
                                            <li>
                                                <button class="dropdown-item" onclick="changeQuoteStatus({{ $quote->id }}, 'approved')">
                                                    <i class="bi bi-check-circle me-2"></i> Approve
                                                </button>
                                            </li>
                                            <li>
                                                <button class="dropdown-item" onclick="changeQuoteStatus({{ $quote->id }}, 'rejected')">
                                                    <i class="bi bi-x-circle me-2"></i> Reject
                                                </button>
                                            </li>
                                            @php
                                                $paymentMethod = $quote->payment_method ?? ($quote->raw_data['payment_method'] ?? null);
                                                $isBankTransfer = $paymentMethod === 'bank_transfer';
                                            @endphp
                                            @if($isBankTransfer && $quote->deposit_payment_status === 'pending' && $quote->deposit_amount)
                                                <li>
                                                    <hr class="dropdown-divider">
                                                </li>
                                                <li>
                                                    <button class="dropdown-item text-success" onclick="event.stopPropagation(); approveBankTransferPayment({{ $quote->id }}, 'deposit');">
                                                        <i class="fa-solid fa-check-circle me-2"></i> Approve Deposit Payment
                                                    </button>
                                                </li>
                                                <li>
                                                    <button class="dropdown-item text-danger" onclick="event.stopPropagation(); rejectBankTransferPayment({{ $quote->id }}, 'deposit');">
                                                        <i class="fa-solid fa-times-circle me-2"></i> Reject Deposit Payment
                                                    </button>
                                                </li>
                                            @endif
                                            @if($isBankTransfer && $quote->deposit_payment_status === 'succeeded' && $quote->remaining_amount && $quote->remaining_amount > 0 && $quote->remaining_payment_status !== 'succeeded')
                                                <li>
                                                    <hr class="dropdown-divider">
                                                </li>
                                                <li>
                                                    <button class="dropdown-item" onclick="event.stopPropagation(); adjustRemainingAmountAndProof({{ $quote->id }}, {{ $quote->remaining_amount ?? 0 }});">
                                                        <i class="fa-solid fa-edit me-2"></i> Adjust Amount & Upload Proof
                                                    </button>
                                                </li>
                                                @if($quote->remaining_payment_status === 'pending')
                                                    <li>
                                                        <button class="dropdown-item text-success" onclick="event.stopPropagation(); approveBankTransferPayment({{ $quote->id }}, 'remaining');">
                                                            <i class="fa-solid fa-check-circle me-2"></i> Approve Remaining Payment
                                                        </button>
                                                    </li>
                                                @endif
                                            @elseif($quote->remaining_amount && $quote->remaining_amount > 0 && $quote->remaining_payment_status !== 'succeeded' && !$isBankTransfer)
                                                <li>
                                                    <hr class="dropdown-divider">
                                                </li>
                                                <li>
                                                    <button class="dropdown-item" onclick="event.stopPropagation(); copyPaymentLink({{ $quote->id }});">
                                                        <i class="fa-solid fa-copy me-2"></i> Copy Payment Link
                                                    </button>
                                                </li>
                                                <li>
                                                    <button class="dropdown-item" onclick="event.stopPropagation(); sendRemainingPaymentEmail({{ $quote->id }});">
                                                        <i class="fa-solid fa-envelope me-2"></i> Send Payment Link
                                                    </button>
                                                </li>
                                                <li>
                                                    <button class="dropdown-item" onclick="event.stopPropagation(); releaseRemainingPayment({{ $quote->id }});">
                                                        <i class="fa-solid fa-check me-2"></i> Release Payment
                                                    </button>
                                                </li>
                                            @endif
                                            <li>
                                                <hr class="dropdown-divider">
                                            </li>
                                            <li>
                                                <button class="dropdown-item text-danger" onclick="deleteQuote({{ $quote->id }})">
                                                    <i class="bi bi-trash me-2"></i> Delete
                                                </button>
                                            </li>
                                        </ul>
                                    </div>
                                </td>
                            </tr>
                            <!-- Details Row -->
                            <tr class="quote-details-row" id="details-{{ $quote->id }}" style="display: none;">
                                <td></td>
                                <td colspan="8" class="bg-light p-0" style="padding: 0 !important;">
                                    <div class="p-3">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <h6 class="mb-2"><i class="fa-solid fa-info-circle me-2"></i>Quote Information</h6>
                                                <table class="table table-sm table-borderless mb-0">
                                                    <tr>
                                                        <td class="fw-bold" style="width: 150px;">Quote Number:</td>
                                                        <td>{{ $quote->quote_number ?? 'N/A' }}</td>
                                                    </tr>
                                                    @if($quote->order_number)
                                                        <tr>
                                                            <td class="fw-bold">Order Number:</td>
                                                            <td><span class="badge bg-success">{{ $quote->order_number }}</span></td>
                                                        </tr>
                                                    @endif
                                                    <tr>
                                                        <td class="fw-bold">Quote Type:</td>
                                                        <td><span class="badge {{ $type[0] }}">{{ $type[1] }}</span></td>
                                                    </tr>
                                                    <tr>
                                                        <td class="fw-bold">Status:</td>
                                                        <td><span class="badge {{ $status[0] }}">{{ $status[1] }}</span></td>
                                                    </tr>
                                                    <tr>
                                                        <td class="fw-bold">Created At:</td>
                                                        <td>{{ $quote->created_at->format('j F Y H:i:s') }}</td>
                                                    </tr>
                                                    <tr>
                                                        <td class="fw-bold">Total Cost:</td>
                                                        <td class="fw-bold text-primary">
                                                            @php
                                                                // If total_cost is 0, show original_total_cost from raw_data
                                                                $displayTotalCost = $quote->total_cost;
                                                                if ($displayTotalCost == 0 || $displayTotalCost == null) {
                                                                    $displayTotalCost = $quote->raw_data['original_total_cost'] ?? ($quote->raw_data['original_total_cost_before_adjustment'] ?? 0);
                                                                }
                                                            @endphp
                                                            ${{ number_format($displayTotalCost, 2) }}
                                                        </td>
                                                    </tr>
                                                    @php
                                                        $jobType = $quote->job_type ?? ($quote->raw_data['job_type'] ?? null);
                                                        $jobTypeColors = [
                                                            'local_job' => 'primary',
                                                            'interstate_job' => 'info',
                                                        ];
                                                        $jobTypeColor = $jobTypeColors[$jobType] ?? 'secondary';
                                                    @endphp
                                                    @if($jobType)
                                                    <tr>
                                                        <td class="fw-bold">Job Type:</td>
                                                        <td>
                                                            <span class="badge bg-{{ $jobTypeColor }}">
                                                                {{ ucfirst(str_replace('_', ' ', $jobType)) }}
                                                            </span>
                                                        </td>
                                                    </tr>
                                                    @endif
                                                    @if($quote->movers_cost > 0)
                                                        <tr>
                                                            <td class="fw-bold">Movers Cost:</td>
                                                            <td>${{ number_format($quote->movers_cost, 2) }}</td>
                                                        </tr>
                                                    @endif
                                                    @php
                                                        $calloutFee = $quote->callout_fee ?? ($quote->raw_data['callout_fee'] ?? 0);
                                                    @endphp
                                                    @if(($jobType === 'local_job' || $jobType === 'interstate_job') && $calloutFee > 0)
                                                    <tr>
                                                        <td class="fw-bold">Callout Fee:</td>
                                                        <td>${{ number_format($calloutFee, 2) }}</td>
                                                    </tr>
                                                    @endif
                                                    @php
                                                        $extraMovers = $quote->extra_movers ?? ($quote->raw_data['extra_movers'] ?? 0);
                                                    @endphp
                                                    @if($extraMovers > 0)
                                                    <tr>
                                                        <td class="fw-bold">Extra Movers:</td>
                                                        <td>{{ $extraMovers }} movers <small class="text-muted">(included in Movers Cost)</small></td>
                                                    </tr>
                                                    @endif
                                                </table>
                                                
                                                @php
                                                    $paymentMethod = $quote->payment_method ?? ($quote->raw_data['payment_method'] ?? null);
                                                @endphp
                                                @if($quote->deposit_amount || $quote->remaining_amount || $paymentMethod)
                                                    <hr class="my-2">
                                                    <h6 class="mb-2"><i class="fa-solid fa-credit-card me-2"></i>Payment Information</h6>
                                                    <table class="table table-sm table-borderless mb-0">
                                                        @if($paymentMethod)
                                                            <tr>
                                                                <td class="fw-bold" style="width: 150px;">Payment Method:</td>
                                                                <td>
                                                                    @if($paymentMethod === 'stripe')
                                                                        <span class="badge bg-primary">Stripe</span>
                                                                    @elseif($paymentMethod === 'bank_transfer')
                                                                        <span class="badge bg-info">Bank Transfer</span>
                                                                    @else
                                                                        <span class="badge bg-secondary">{{ ucfirst(str_replace('_', ' ', $paymentMethod)) }}</span>
                                                                    @endif
                                                                </td>
                                                            </tr>
                                                        @endif
                                                        @if($quote->deposit_amount)
                                                            <tr>
                                                                <td class="fw-bold" style="width: 150px;">Deposit Amount:</td>
                                                                <td>
                                                                    <span class="fw-bold text-success">${{ number_format($quote->deposit_amount, 2) }}</span>
                                                                    @if($quote->deposit_payment_status === 'succeeded')
                                                                        <span class="badge bg-success ms-2">Paid</span>
                                                                    @elseif($quote->deposit_payment_status === 'pending')
                                                                        <span class="badge bg-warning ms-2">Pending</span>
                                                                    @elseif($quote->deposit_payment_status === 'failed')
                                                                        <span class="badge bg-danger ms-2">Failed</span>
                                                                    @elseif($quote->deposit_payment_status === 'rejected')
                                                                        <span class="badge bg-danger ms-2">Rejected</span>
                                                                    @endif
                                                                </td>
                                                            </tr>
                                                        @endif
                                                        @if($quote->remaining_payment_status === 'succeeded' || (!$quote->remaining_amount || $quote->remaining_amount <= 0))
                                                            <tr>
                                                                <td class="fw-bold">Remaining Amount:</td>
                                                                <td><span class="text-success">Fully Paid</span></td>
                                                            </tr>
                                                        @elseif($quote->remaining_amount && $quote->remaining_amount > 0)
                                                            <tr>
                                                                <td class="fw-bold">Remaining Amount:</td>
                                                                <td>
                                                                    <span class="fw-bold text-warning">${{ number_format($quote->remaining_amount, 2) }}</span>
                                                                    @if($quote->remaining_payment_status === 'pending')
                                                                        <span class="badge bg-warning ms-2">Pending</span>
                                                                    @elseif($quote->remaining_payment_status === 'failed')
                                                                        <span class="badge bg-danger ms-2">Failed</span>
                                                                    @elseif($quote->remaining_payment_status === 'rejected')
                                                                        <span class="badge bg-danger ms-2">Rejected</span>
                                                                    @endif
                                                                </td>
                                                            </tr>
                                                        @else
                                                            <tr>
                                                                <td class="fw-bold">Remaining Amount:</td>
                                                                <td><span class="text-success">Fully Paid</span></td>
                                                            </tr>
                                                        @endif
                                                        @if($paymentMethod === 'bank_transfer' && $quote->quote_payment_proof)
                                                            <tr>
                                                                <td class="fw-bold">Deposit Payment Proof:</td>
                                                                <td>
                                                                    <a href="{{ asset($quote->quote_payment_proof) }}" target="_blank" class="btn btn-sm btn-outline-primary">
                                                                        <i class="bi bi-file-earmark-image me-1"></i>View Proof
                                                                    </a>
                                                                </td>
                                                            </tr>
                                                        @endif
                                                        @php
                                                            $remainingProof = $quote->raw_data['remaining_payment_proof'] ?? null;
                                                        @endphp
                                                        @if($paymentMethod === 'bank_transfer' && $remainingProof)
                                                            <tr>
                                                                <td class="fw-bold">Remaining Payment Proof:</td>
                                                                <td>
                                                                    <a href="{{ asset($remainingProof) }}" target="_blank" class="btn btn-sm btn-outline-primary">
                                                                        <i class="bi bi-file-earmark-image me-1"></i>View Proof
                                                                    </a>
                                                                </td>
                                                            </tr>
                                                        @endif
                                                    </table>
                                                @endif
                                            </div>
                                            <div class="col-md-6">
                                                <h6 class="mb-2"><i class="fa-solid fa-user me-2"></i>Customer Information</h6>
                                                <table class="table table-sm table-borderless mb-3">
                                                    @if($quote->user)
                                                        <tr><td class="fw-bold" style="width: 150px;">Name:</td><td>{{ $quote->user->name }}</td></tr>
                                                        <tr><td class="fw-bold">Email:</td><td>{{ $quote->user->email }}</td></tr>
                                                        @if($quote->user->phone)
                                                            <tr><td class="fw-bold">Phone:</td><td>{{ $quote->user->phone }}</td></tr>
                                                        @endif
                                                    @elseif($quote->booking)
                                                        <tr><td class="fw-bold" style="width: 150px;">Name:</td><td>{{ $quote->booking->name }}</td></tr>
                                                        <tr><td class="fw-bold">Email:</td><td>{{ $quote->booking->email }}</td></tr>
                                                        @if($quote->booking->phone)
                                                            <tr><td class="fw-bold">Phone:</td><td>{{ $quote->booking->phone }}</td></tr>
                                                        @endif
                                                    @endif
                                                </table>
                                                
                                                @if($quote->booking)
                                                    <h6 class="mb-2"><i class="fa-solid fa-calendar me-2"></i>Booking Details</h6>
                                                    <table class="table table-sm table-borderless mb-0">
                                                        <tr><td class="fw-bold" style="width: 150px;">Booking ID:</td><td>{{ $quote->booking->id }}</td></tr>
                                                        <tr><td class="fw-bold">Move Date:</td><td>{{ $quote->booking->move_date ? $quote->booking->move_date->format('j F Y') : 'N/A' }}</td></tr>
                                                    </table>
                                                @endif
                                                
                                                <div class="mt-3">
                                                    <a href="{{ route('admin.quotes.show', $quote->id) }}" class="btn btn-sm btn-primary">
                                                        <i class="bi bi-eye me-1"></i> View Full Details
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                        @empty
                        <tr class="no-data-row">
                            <td></td>
                            <td></td>
                            <td></td>
                            <td></td>
                            <td class="text-center text-muted py-4">No quotes/orders found</td>
                            <td></td>
                            <td></td>
                            <td></td>
                            <td></td>
                        </tr>
                        @endforelse
                    </tbody>
                    <tfoot>
                        <tr>
                            <th width="30px"></th>
                            <th>Order/Quote #</th>
                            <th>Customer</th>
                            <th>Job Type</th>
                            <th>Total Amount</th>
                            <th>Payment Method</th>
                            <th>Payment</th>
                            <th>Status</th>
                            <th width="80px">Actions</th>
                        </tr>
                    </tfoot>
                </table>
            </div>
            
            <!-- Pagination -->
            @if($quotes->hasPages() || $quotes->total() > 0)
                <div class="d-flex justify-content-between align-items-center mt-3 flex-wrap gap-2">
                    <div class="text-muted">
                        Showing {{ $quotes->firstItem() ?? 0 }} to {{ $quotes->lastItem() ?? 0 }} of {{ $quotes->total() }} entries
                    </div>
                    <div class="d-flex align-items-center gap-2">
                       
                        @if($quotes->hasPages())
                            <div>
                                {{ $quotes->links('pagination::bootstrap-4') }}
                            </div>
                        @endif
                    </div>
                </div>
            @endif
        </div>
    </div>
</div>

<style>
    .badge {
        font-size: 0.75rem;
        font-weight: 500;
    }
    
    .quote-row:hover {
        background-color: #f8f9fa;
    }
    
    .quote-details-row {
        background-color: #f8f9fa !important;
    }
    
    .toggle-details {
        transition: transform 0.3s ease;
    }
    
    .toggle-details.expanded {
        transform: rotate(90deg);
    }
    
    .clickable-cell {
        cursor: pointer;
    }
    
    tr.quote-row {
        cursor: pointer;
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Initially hide all details rows
        $('.quote-details-row').hide();
        
        // Handle filter collapse toggle icon
        $('#filterCollapse').on('show.bs.collapse', function () {
            $('[data-bs-target="#filterCollapse"]').find('i').removeClass('bi-chevron-up').addClass('bi-chevron-down');
        });
        $('#filterCollapse').on('hide.bs.collapse', function () {
            $('[data-bs-target="#filterCollapse"]').find('i').removeClass('bi-chevron-down').addClass('bi-chevron-up');
        });
        
        // Handle per page change
        $('#per_page_select').on('change', function() {
            const perPage = $(this).val();
            const url = new URL(window.location.href);
            url.searchParams.set('per_page', perPage);
            url.searchParams.set('page', '1'); // Reset to first page
            window.location.href = url.toString();
        });
    });

    function toggleRowDetails(quoteId) {
        const detailsRow = $('#details-' + quoteId);
        const toggleBtn = $('.toggle-details[data-quote-id="' + quoteId + '"]');
        const icon = toggleBtn.find('i');
        
        if (detailsRow.is(':visible')) {
            detailsRow.slideUp(300);
            toggleBtn.removeClass('expanded');
            icon.removeClass('fa-chevron-down').addClass('fa-chevron-right');
        } else {
            // Close all other expanded rows (optional - remove if you want multiple open)
            $('.quote-details-row:visible').slideUp(300);
            $('.toggle-details.expanded').removeClass('expanded').find('i').removeClass('fa-chevron-down').addClass('fa-chevron-right');
            
            detailsRow.slideDown(300);
            toggleBtn.addClass('expanded');
            icon.removeClass('fa-chevron-right').addClass('fa-chevron-down');
        }
    }
    
    // Handle toggle button click - use event delegation
    $(document).on('click', '.toggle-details', function(e) {
        e.stopPropagation();
        const quoteId = $(this).data('quote-id');
        toggleRowDetails(quoteId);
    });
    
    // Handle row click - use event delegation
    $(document).on('click', 'tr.quote-row', function(e) {
        // Don't trigger if clicking on toggle button or dropdown
        if ($(e.target).closest('.toggle-details, .dropdown').length) {
            return;
        }
        const quoteId = $(this).data('quote-id');
        toggleRowDetails(quoteId);
    });

    function changeQuoteStatus(quoteId, status) {
        Swal.fire({
            title: 'Are you sure?',
            text: "You want to change the quote status?",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Yes, change it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: "{{ url('/admin/quotes') }}/" + quoteId + "/update-status",
                    type: "POST",
                    data: { status: status },
                    success: function(response) {
                        if (response.status === 'success') {
                            Swal.fire({
                                icon: 'success',
                                title: 'Status Updated',
                                text: response.message,
                                timer: 1500,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.message || 'Failed to update status'
                            });
                        }
                    },
                    error: function(xhr) {
                        let errorMessage = 'Something went wrong';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage
                        });
                    }
                });
            }
        });
    }

    function deleteQuote(quoteId) {
        Swal.fire({
            title: 'Are you sure?',
            text: "You won't be able to revert this!",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Yes, delete it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: "{{ url('/admin/quotes') }}/" + quoteId,
                    type: "DELETE",
                    success: function(response) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Deleted!',
                            text: 'Quote has been deleted.',
                            timer: 1500,
                            showConfirmButton: false
                        }).then(() => {
                            location.reload();
                        });
                    },
                    error: function(xhr) {
                        let errorMessage = 'Something went wrong';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage
                        });
                    }
                });
            }
        });
    }

    function sendRemainingPaymentEmail(quoteId) {
        Swal.fire({
            title: 'Send Payment Link?',
            text: "Send remaining payment link email to customer?",
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#01A7EF',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, send it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: "{{ url('/admin/quotes') }}/" + quoteId + "/send-remaining-payment-email",
                    type: "POST",
                    success: function(response) {
                        if (response.status === 'success') {
                            Swal.fire({
                                icon: 'success',
                                title: 'Email Sent!',
                                text: response.message,
                                timer: 2000,
                                showConfirmButton: false
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.message || 'Failed to send email'
                            });
                        }
                    },
                    error: function(xhr) {
                        let errorMessage = 'Something went wrong';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage
                        });
                    }
                });
            }
        });
    }

    function releaseRemainingPayment(quoteId) {
        Swal.fire({
            title: 'Release Remaining Payment?',
            text: "This will mark the remaining payment as paid/waived. Are you sure?",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#28a745',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, release it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: "{{ url('/admin/quotes') }}/" + quoteId + "/release-remaining-payment",
                    type: "POST",
                    success: function(response) {
                        if (response.status === 'success') {
                            Swal.fire({
                                icon: 'success',
                                title: 'Payment Released!',
                                text: response.message,
                                timer: 2000,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.message || 'Failed to release payment'
                            });
                        }
                    },
                    error: function(xhr) {
                        let errorMessage = 'Something went wrong';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage
                        });
                    }
                });
            }
        });
    }

    function copyPaymentLink(quoteId) {
        // Show loading state
        Swal.fire({
            title: 'Getting payment link...',
            allowOutsideClick: false,
            didOpen: () => {
                Swal.showLoading();
            }
        });

        $.ajaxSetup({
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        });

        $.ajax({
            url: "{{ url('/admin/quotes') }}/" + quoteId + "/get-payment-link",
            type: "GET",
            success: function(response) {
                if (response.status === 'success' && response.payment_link) {
                    // Copy to clipboard
                    const paymentLink = response.payment_link;
                    
                    // Try modern Clipboard API first (only available in secure contexts)
                    if (navigator.clipboard && navigator.clipboard.writeText) {
                        navigator.clipboard.writeText(paymentLink).then(function() {
                            Swal.fire({
                                icon: 'success',
                                title: 'Link Copied!',
                                html: 'Payment link has been copied to clipboard.<br><br><small style="word-break: break-all; color: #666;">' + paymentLink + '</small>',
                                timer: 3000,
                                showConfirmButton: true,
                                confirmButtonText: 'OK'
                            });
                        }).catch(function(err) {
                            // Fallback to execCommand
                            copyToClipboardFallback(paymentLink);
                        });
                    } else {
                        // Fallback to execCommand for older browsers or non-secure contexts
                        copyToClipboardFallback(paymentLink);
                    }
                } else {
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: response.message || 'Failed to get payment link'
                    });
                }
            },
            error: function(xhr) {
                let errorMessage = 'Something went wrong';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    errorMessage = xhr.responseJSON.message;
                }

                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: errorMessage
                });
            }
        });
    }

    function copyToClipboardFallback(paymentLink) {
        // Create a temporary textarea element
        const tempTextarea = document.createElement('textarea');
        tempTextarea.value = paymentLink;
        tempTextarea.style.position = 'fixed';
        tempTextarea.style.left = '-999999px';
        tempTextarea.style.top = '-999999px';
        document.body.appendChild(tempTextarea);
        tempTextarea.focus();
        tempTextarea.select();
        tempTextarea.setSelectionRange(0, 99999); // For mobile devices
        
        try {
            const successful = document.execCommand('copy');
            document.body.removeChild(tempTextarea);
            
            if (successful) {
                Swal.fire({
                    icon: 'success',
                    title: 'Link Copied!',
                    html: 'Payment link has been copied to clipboard.<br><br><small style="word-break: break-all; color: #666;">' + paymentLink + '</small>',
                    timer: 3000,
                    showConfirmButton: true,
                    confirmButtonText: 'OK'
                });
            } else {
                // Final fallback: show the link in a dialog for manual copying
                showPaymentLinkDialog(paymentLink);
            }
        } catch (err) {
            document.body.removeChild(tempTextarea);
            // Final fallback: show the link in a dialog for manual copying
            showPaymentLinkDialog(paymentLink);
        }
    }

    function showPaymentLinkDialog(paymentLink) {
        Swal.fire({
            icon: 'info',
            title: 'Payment Link',
            html: 'Please copy the link manually:<br><br><input type="text" class="form-control" value="' + paymentLink + '" id="paymentLinkInput" readonly style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; word-break: break-all; margin: 10px 0;"><br><small>Click on the input field and press Ctrl+C (or Cmd+C on Mac) to copy</small>',
            showConfirmButton: true,
            confirmButtonText: 'Close',
            width: '600px',
            didOpen: () => {
                const input = document.getElementById('paymentLinkInput');
                if (input) {
                    input.focus();
                    input.select();
                    input.setSelectionRange(0, 99999);
                }
            }
        });
    }

    function approveBankTransferPayment(quoteId, paymentType) {
        Swal.fire({
            title: 'Are you sure?',
            text: `You want to approve the ${paymentType} payment?`,
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#28a745',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, approve it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: "{{ url('/admin/quotes') }}/" + quoteId + "/approve-bank-transfer-payment",
                    type: "POST",
                    data: {
                        payment_type: paymentType,
                        action: 'approve'
                    },
                    success: function(response) {
                        if (response.status === 'success') {
                            Swal.fire({
                                icon: 'success',
                                title: 'Payment Approved',
                                text: response.message,
                                timer: 2000,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.message || 'Failed to approve payment'
                            });
                        }
                    },
                    error: function(xhr) {
                        let errorMessage = 'Something went wrong';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage
                        });
                    }
                });
            }
        });
    }

    function rejectBankTransferPayment(quoteId, paymentType) {
        Swal.fire({
            title: 'Are you sure?',
            text: `You want to reject the ${paymentType} payment?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, reject it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: "{{ url('/admin/quotes') }}/" + quoteId + "/approve-bank-transfer-payment",
                    type: "POST",
                    data: {
                        payment_type: paymentType,
                        action: 'reject'
                    },
                    success: function(response) {
                        if (response.status === 'success') {
                            Swal.fire({
                                icon: 'success',
                                title: 'Payment Rejected',
                                text: response.message,
                                timer: 2000,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.message || 'Failed to reject payment'
                            });
                        }
                    },
                    error: function(xhr) {
                        let errorMessage = 'Something went wrong';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage
                        });
                    }
                });
            }
        });
    }

    function adjustRemainingAmountAndProof(quoteId, currentAmount) {
        Swal.fire({
            title: 'Adjust Remaining Amount & Upload Proof',
            html: `
                <form id="adjustRemainingForm" enctype="multipart/form-data">
                    <div class="text-start mb-3">
                        <label for="remaining_amount" class="form-label">Amount to Pay/Adjust ($)</label>
                        <input type="number" step="0.01" min="0" class="form-control" id="remaining_amount" value="${currentAmount || ''}" placeholder="Enter new remaining amount">
                        <small class="text-muted">Current amount: $${parseFloat(currentAmount || 0).toFixed(2)}</small>
                    </div>
                    <div class="text-start mb-3">
                        <label for="proof_file" class="form-label">Payment Proof (Image or PDF) <span class="text-danger">*</span></label>
                        <input type="file" class="form-control" required id="proof_file" name="remaining_payment_proof" accept="image/*,.pdf">
                        <small class="text-muted">Accepted formats: JPG, JPEG, PNG, WEBP, PDF (required)</small>
                    </div>
                    <div class="text-start">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="mark_as_paid">
                            <label class="form-check-label" for="mark_as_paid">
                                Mark as paid (It settles all remaining amount) 
                            </label>
                        </div>
                    </div>
                </form>
            `,
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Update',
            cancelButtonText: 'Cancel',
            width: '600px',
            preConfirm: () => {
                const amount = document.getElementById('remaining_amount').value;
                const fileInput = document.getElementById('proof_file');
                
                // Payment proof file is required
                if (!fileInput.files || !fileInput.files[0]) {
                    Swal.showValidationMessage('Payment proof file is required');
                    return false;
                }
                
                // Validate amount if provided
                if (amount && (isNaN(amount) || parseFloat(amount) < 0)) {
                    Swal.showValidationMessage('Please enter a valid amount');
                    return false;
                }
                
                return true;
            },
            didOpen: () => {
                document.getElementById('remaining_amount').focus();
            }
        }).then((result) => {
            if (result.isConfirmed) {
                const amount = document.getElementById('remaining_amount').value;
                const fileInput = document.getElementById('proof_file');
                const markAsPaid = document.getElementById('mark_as_paid').checked;
                
                const formData = new FormData();
                if (amount && amount !== '') {
                    formData.append('remaining_amount', amount);
                }
                if (fileInput.files && fileInput.files[0]) {
                    formData.append('remaining_payment_proof', fileInput.files[0]);
                }
                if (markAsPaid) {
                    formData.append('mark_as_paid', '1');
                }
                formData.append('_token', $('meta[name="csrf-token"]').attr('content'));

                Swal.fire({
                    title: 'Processing...',
                    allowOutsideClick: false,
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });

                $.ajax({
                    url: "{{ url('/admin/quotes') }}/" + quoteId + "/adjust-remaining-amount",
                    type: "POST",
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        if (response.status === 'success') {
                            Swal.fire({
                                icon: 'success',
                                title: 'Updated Successfully',
                                text: response.message,
                                timer: 2000,
                                showConfirmButton: false
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.message || 'Failed to update'
                            });
                        }
                    },
                    error: function(xhr) {
                        let errorMessage = 'Something went wrong';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage
                        });
                    }
                });
            }
        });
    }
</script>
@include('Admin.Includes.footer')

