@include('Admin.Includes.header')
@php
    use Illuminate\Support\Facades\Auth;
    $logged_in_user = Auth::user();
@endphp

<div class="page-content">
    <div class="card shadow-sm mb-4">
        <div class="p-3 d-flex justify-content-between align-items-center flex-wrap">
            <div class="clearfix">
                <div class="pd-t-5 pd-b-5">
                    <h1 class="fs-4 pd-0 mg-0 tx-20 tx-dark">{{ __('Storage Items') }}</h1>
                </div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb bg-transparent p-0 m-0 align-items-center">
                        <li class="breadcrumb-item">
                            <a href="{{ route('Admin.Dashboard') }}" class="link-primary text-decoration-none">
                                <i class="fa-solid fa-house me-1"></i> Dashboard
                            </a>
                        </li>
                        <li class="breadcrumb-item active" aria-current="page">
                            Storage Items
                        </li>
                    </ol>
                </nav>
            </div>
        </div>
    </div>

    @if (session('success'))
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    @if (session('error'))
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            {{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    @if ($errors->any())
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    @endif

    <div class="row">
        <div class="col-xxl-5">
            <div class="card mb-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">Create Storage Item</h6>
                </div>
                <div class="card-body">
                    <form autocomplete="off"
                          method="POST"
                          action="{{ route('admin.storage-items.store') }}"
                          id="storageItemForm">
                        @csrf
                        <input type="hidden" name="storage_item_id" id="storage_item_id">

                        <div class="row">
                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="storage_item_title" class="form-label">Title<span class="text-danger">*</span></label>
                                    <input type="text"
                                           id="storage_item_title"
                                           name="title"
                                           required
                                           class="form-control"
                                           placeholder="Enter storage item title">
                                    <div class="invalid-feedback">Please enter a title.</div>
                                </div>
                            </div>

                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="storage_item_permalink" class="form-label">Permalink<span class="text-danger">*</span></label>
                                    <input type="text"
                                           id="storage_item_permalink"
                                           name="permalink"
                                           required
                                           class="form-control"
                                           placeholder="enter-storage-item">
                                    <div class="invalid-feedback">Please enter a unique permalink.</div>
                                    <small class="text-muted">Permalink is auto-generated from the title. You can adjust if needed.</small>
                                </div>
                            </div>

                            <div class="col-12">
                                <div class="mb-3">
                                    <label for="storage_item_price" class="form-label">Price ($)</label>
                                    <input type="number"
                                           id="storage_item_price"
                                           name="price"
                                           class="form-control"
                                           placeholder="e.g., 149.99"
                                           step="0.01"
                                           min="0">
                                </div>
                            </div>

                            <div class="col-6">
                                <div class="mb-3">
                                    <label for="storage_item_status" class="form-label">Status</label>
                                    <select id="storage_item_status"
                                            name="is_active"
                                            class="form-select">
                                        <option value="1">Active</option>
                                        <option value="0">Inactive</option>
                                    </select>
                                </div>
                            </div>

                            <div class="col-6">
                                <div class="mb-3">
                                    <label for="storage_item_sort_order" class="form-label">Sort Order</label>
                                    <input type="number"
                                           id="storage_item_sort_order"
                                           name="sort_order"
                                           class="form-control"
                                           value="0"
                                           min="0">
                                </div>
                            </div>
                        </div>

                        <div class="d-flex justify-content-end">
                            <button type="button" class="btn btn-light me-2" onclick="resetForm()">Reset</button>
                            <button type="submit" class="btn btn-success" id="submitBtn">Save Storage Item</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-xxl-7">
            <div class="card">
                <div class="card-header d-flex justify-content-between">
                    <h5 class="card-title mb-0">Storage Items Table</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="data_Table" class="table table-borderless">
                            <thead>
                                <tr>
                                    <th>Title</th>
                                    <th>Permalink</th>
                                    <th>Price</th>
                                    <th>Status</th>
                                    <th>Sort Order</th>
                                    <th width="180px">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($storageItems as $item)
                                    <tr class="align-middle">
                                        <td class="fw-medium">{{ $item->title }}</td>
                                        <td><span class="text-muted">{{ $item->permalink }}</span></td>
                                        <td>
                                            @if (!is_null($item->price))
                                                <span class="text-muted">${{ number_format($item->price, 2) }}</span>
                                            @else
                                                <span class="text-muted">-</span>
                                            @endif
                                        </td>
                                        <td class="status-cell" data-storage-item-id="{{ $item->id }}">
                                            @if ($item->is_active)
                                                <span class="badge bg-success-subtle text-success px-3 py-2">Active</span>
                                            @else
                                                <span class="badge bg-danger-subtle text-danger px-3 py-2">Inactive</span>
                                            @endif
                                        </td>
                                        <td>{{ $item->sort_order ?? '-' }}</td>
                                        <td>
                                            <div class="dropdown">
                                                <button class="btn btn-sm text-dark rounded shadow-sm"
                                                        type="button"
                                                        data-bs-toggle="dropdown"
                                                        aria-expanded="false"
                                                        style="background-color: #c2d6e7;">
                                                    &#8942;
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li>
                                                        <button class="dropdown-item"
                                                                onclick="editStorageItem({{ $item->id }})">
                                                            <i class="bi bi-pencil-square me-2"></i>Edit
                                                        </button>
                                                    </li>
                                                    <li>
                                                        <button class="dropdown-item"
                                                                onclick="updateStatus({{ $item->id }}, '{{ $item->is_active ? 0 : 1 }}', this)">
                                                            <i class="fa fa-{{ $item->is_active ? 'ban' : 'check' }} me-2"></i>{{ $item->is_active ? 'Inactive' : 'Active' }}
                                                        </button>
                                                    </li>
                                                    <li>
                                                        <button class="dropdown-item text-danger"
                                                                onclick="confirmDeleteStorageItem({{ $item->id }}, '{{ addslashes($item->title) }}')">
                                                            <i class="bi bi-trash me-2"></i>Delete
                                                        </button>
                                                    </li>
                                                </ul>
                                            </div>
                                        </td>
                                    </tr>
                                @empty
                                    <tr class="no-data-row">
                                        <td></td>
                                        <td></td>
                                        <td class="text-end text-muted py-4">No data</td>
                                        <td></td>
                                        <td></td>
                                        <td></td>
                                    </tr>
                                @endforelse
                            </tbody>
                            <tfoot>
                                <tr>
                                    <th>Title</th>
                                    <th>Permalink</th>
                                    <th>Price</th>
                                    <th>Status</th>
                                    <th>Sort Order</th>
                                    <th width="180px">Actions</th>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="modal fade zoomIn" id="deleteConfirmModal" tabindex="-1" aria-labelledby="deleteConfirmModalLabel"
    aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>

            <div class="modal-body p-md-5">
                <div class="text-center">
                    <div class="text-danger">
                        <i class="bi bi-trash display-4"></i>
                    </div>
                    <div class="mt-4 fs-15">
                        <p class="text-muted mx-3 fs-16 mb-0">
                            Are you sure you want to delete
                            <strong id="deleteItemName"></strong>?<br>
                            This action cannot be undone.
                        </p>
                        <p class="text-muted mt-2 mb-0">
                            <small>Suggestion: mark it as <em>Inactive</em> instead.</small>
                        </p>
                    </div>
                </div>

                <div class="d-flex gap-2 justify-content-center mt-4 mb-2">
                    <button type="button" class="btn w-sm btn-light btn-hover"
                        data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn w-sm btn-danger btn-hover" id="confirmDeleteBtn">Yes, Delete
                        It!</button>
                </div>
            </div>
        </div>
    </div>
</div>

@include('Admin.Includes.footer')

<script>
    let permalinkManuallyEdited = false;

    document.getElementById('storage_item_title').addEventListener('input', function() {
        if (permalinkManuallyEdited) {
            return;
        }

        const slug = this.value
            .toLowerCase()
            .trim()
            .replace(/[^a-z0-9\s-]/g, '')
            .replace(/\s+/g, '-')
            .replace(/-+/g, '-');

        document.getElementById('storage_item_permalink').value = slug;
    });

    document.getElementById('storage_item_permalink').addEventListener('input', function() {
        permalinkManuallyEdited = this.value.length > 0;
    });

    function resetForm() {
        document.getElementById('storageItemForm').reset();
        document.getElementById('storage_item_id').value = '';
        document.getElementById('submitBtn').textContent = 'Save Storage Item';
        permalinkManuallyEdited = false;
        loadNextStorageItemSortOrder();
    }

    function loadNextStorageItemSortOrder() {
        $.ajax({
            url: "{{ route('admin.storage-items.nextSortOrder') }}",
            type: "GET",
            success: function(response) {
                $('#storage_item_sort_order').val(response.next_sort_order);
            },
            error: function() {
                $('#storage_item_sort_order').val('1');
            }
        });
    }

    function editStorageItem(id) {
        fetch(`{{ route('admin.storage-items.edit') }}?id=${id}`)
            .then(response => response.json())
            .then(data => {
                document.getElementById('storage_item_id').value = data.id;
                document.getElementById('storage_item_title').value = data.title;
                document.getElementById('storage_item_permalink').value = data.permalink;
                document.getElementById('storage_item_price').value = data.price ?? '';
                document.getElementById('storage_item_status').value = data.is_active ? '1' : '0';
                document.getElementById('storage_item_sort_order').value = data.sort_order ?? 0;
                document.getElementById('submitBtn').textContent = 'Update Storage Item';
                permalinkManuallyEdited = true;

                document.getElementById('storageItemForm').scrollIntoView({
                    behavior: 'smooth'
                });
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error loading storage item data');
            });
    }

    function updateStatus(storageItemId, status, element) {
        $.ajaxSetup({
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        });

        $.ajax({
            url: `{{ url('/admin/storage-items') }}/${storageItemId}/status`,
            type: 'POST',
            data: {
                is_active: status,
                _token: '{{ csrf_token() }}'
            },
            beforeSend: function() {
                $(element).prop('disabled', true);
            },
            success: function(response) {
                Swal.fire({
                    icon: 'success',
                    title: 'Status Updated',
                    text: 'Storage item status updated successfully',
                    timer: 1500,
                    showConfirmButton: false
                });

                const row = $(element).closest('tr');
                const statusCell = row.find('.status-cell');

                if (status === '1') {
                    statusCell.html('<span class="badge bg-success-subtle text-success px-3 py-2">Active</span>');
                } else {
                    statusCell.html('<span class="badge bg-danger-subtle text-danger px-3 py-2">Inactive</span>');
                }

                setTimeout(function() {
                    window.location.reload();
                }, 1000);
            },
            error: function() {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'Failed to update status'
                });
            },
            complete: function() {
                $(element).prop('disabled', false);
            }
        });
    }

    function confirmDeleteStorageItem(id, name) {
        document.getElementById('deleteItemName').textContent = name;
        document.getElementById('confirmDeleteBtn').onclick = function() {
            deleteStorageItem(id);
        };
        new bootstrap.Modal(document.getElementById('deleteConfirmModal')).show();
    }

    function deleteStorageItem(id) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = `{{ route('admin.storage-items.delete') }}`;

        const idInput = document.createElement('input');
        idInput.type = 'hidden';
        idInput.name = 'storage_item';
        idInput.value = id;

        const tokenInput = document.createElement('input');
        tokenInput.type = 'hidden';
        tokenInput.name = '_token';
        tokenInput.value = '{{ csrf_token() }}';

        form.appendChild(idInput);
        form.appendChild(tokenInput);
        document.body.appendChild(form);
        form.submit();
    }

    $(document).ready(function() {
        loadNextStorageItemSortOrder();

        if ($.fn.DataTable.isDataTable('#data_Table')) {
            $('#data_Table').DataTable().destroy();
        }

        $('#data_Table').DataTable({
            responsive: true,
            autoWidth: false,
            columnDefs: [
                {
                    orderable: false,
                    targets: -1
                }
            ],
            language: {
                searchPlaceholder: 'Search...',
                sSearch: ''
            },
            order: [
                [4, 'asc']
            ]
        });
    });
</script>

